{ *************************************************************************** }

{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }


unit uGlobal;

interface

uses
 classes, QActnList;

function StartApp(const name: string; const arguments: array of string; out strOutput : string): Integer;
function getHomeDir : string;
function TrimSysChars( const str : string ) : string;
function AddSysChars( const str : string ) : string;
function GetCharCount( const str : string; c : char ) : integer; overload;
function GetCharCount( const str : string ) : integer; overload;
function GetLineCount( const str : string ) : integer;
function GetWordCount( const str : string ) : integer;
function GetBeginSpaces( const str : string ) : integer;
function GetEndSpaces( const str : string ) : integer;
function Compare( const subStr, str : string; bMatchCase, bWholeWord, bIgnoreAmpersand : boolean ) : boolean;
function hasEllipse( const str : string ) : boolean;
function CompareNoEllipse( const str1, str2 : string ) : boolean;
function IncludeEllipse( const str : string ) : string;
function ExcludeEllipse( const str : string ) : string;
procedure AddShortCutsToActionHints( al : TCustomActionList );

implementation

uses
  libc, QForms, SysUtils, QMenus, StrUtils;

function RemoveAmpersand( const str : string ) : string; forward;
function MatchWord( const SubStr, str : string) : boolean; forward;

function StartApp(const name: string; const arguments: array of string; out strOutput : string): Integer;
var
  pid: PID_T;
  i: Integer;
  parg: PPCharArray;
  argnum: Integer;
  pdesc: TPipeDescriptors;
  buf : array[0..1024-1] of byte;
  count : integer;
  strTemp : string;
begin
  Result    := -1;
  strOutput := '';

  pipe(pdesc);
  pid := fork;
  if pid = 0 then
  begin
    //child
    argnum := High(Arguments) + 1;
    GetMem(parg,(2 + argnum) * sizeof(PChar));
    parg[0] := PChar(Name);

    i := 0;
    while i <= high(arguments) do begin
      inc(i);
      parg[i] := PChar(arguments[i-1]);
    end;
    parg[i+1] := nil;
    dup2( pdesc.WriteDes, 1);
    dup2( pdesc.WriteDes, 2);
    __close(pdesc.ReadDes);
    __close(pdesc.WriteDes);
    execvp(PChar(name),PPChar(@parg[0]));
    FreeMem(parg);
    __close(1);
    __close(2);
    _exit(255);
  end;
  if pid > 0 then begin
    //parent
    result:=-1;
    wait4(pid,@Result,0, nil);
    fcntl(pdesc.ReadDes, F_SETFL, O_NONBLOCK);
    repeat
      count := __read(pdesc.ReadDes, buf[0], sizeof(buf) );
      if count > 0 then begin
         setlength(strTemp, count);
         move( buf[0], strTemp[1], count);
         strOutput := strOutput + strTemp;
      end;
    until count < sizeof(buf);
    strOutput := trim(strOutput);
  end;
end;

function getHomeDir : string;
begin
 if getpwuid(getuid) <> nil then
   result := getpwuid(getuid).pw_dir
 else
   result := '/';
end;


function TrimSysChars( const str : string ) : string;
var
  i : integer;
  s : string;
begin
   result := '';
   for i := 1 to length(str) do begin
     case str[i] of
        #9:  s := '\t';
        #13: s := '\n';
        #10: s := '\r';
        '\': s := '\\';
        else
          s := str[i];
     end;
     result := result + s;
   end;
end;

function AddSysChars( const str : string ) : string;
var
  i : integer;
  s : string;
begin
   result := '';
   i := 1;
   while i <= length(str) do begin
      if (i <= length(str)-1) and (str[i] = '\') then begin
        case str[i+1] of
          't': s := #9;
          'n': s := #13;
          'r': s := #10;
          '\': s := '\';
          else
            s := str[i+1];
        end;
        result := result + s;
        i := i+2;
      end else begin
        result := result + str[i];
        i := i+1;
      end;
   end;
end;

function GetCharCount( const str : string ) : integer;
begin
   result := length(str);
end;

function GetWordCount( const str : string ) : integer;
var
 Ix: Word;
 Work_Count: integer;
function Seps(As_Arg: Char): Boolean;
begin
 Seps := As_Arg in
   [#0..#$1F, '', '.', ',', '?', ':', ';', '(', ')', '/', '\'];
end;
begin
  Work_Count := 0;
  Ix := 1;
  while Ix <= Length(str) do begin
    while (Ix <= Length(str)) and (Seps(str[Ix])) do
      Inc(Ix);
    if Ix <= Length(str) then begin
      Inc(Work_Count);
      while (Ix <= Length(str)) and (not Seps(str[Ix])) do
        Inc(Ix);
    end;
  end;
  result := Work_Count;
end;


function GetLineCount( const str : string ) : integer;
var
  sl : TStringList;
begin
   sl := TStringList.create;
   sl.text := str;
   result := sl.count;
   sl.free;
end;

function GetCharCount( const str : string; c : char ) : integer;
var
  i : integer;
begin
  result := 0;
  for i := 1 to length(str) do begin
     if str[i] = c then
       inc(result);
  end;
end;

function GetBeginSpaces( const str : string ) : integer;
var
  i : integer;
begin
  result := 0;
  for i := 1 to length(str) do begin
    if str[i] = ' ' then
      inc(result)
    else
      break;
  end;
end;

function GetEndSpaces( const str : string ) : integer;
var
  i : integer;
begin
  result := 0;
  for i := length(str) downto 1 do begin
    if str[i] = ' ' then
      inc(result)
    else
      break;
  end;
end;

function RemoveAmpersand( const str : string ) : string;
var
  i : integer;
begin
  result := '';
  for i := 1 to length(str) do begin
    if str[i] <> '&' then
      result := result + str[i];
  end;
end;

function Compare( const subStr, str : string; bMatchCase, bWholeWord, bIgnoreAmpersand : boolean ) : boolean;
var
  substrtmp, strTmp : string;
begin
  subStrTmp := subStr;
  strTmp    := str;
  if bIgnoreAmpersand then begin
    subStrTmp := RemoveAmpersand(subStrTmp);
    strTmp    := RemoveAmpersand(strTmp);
  end;
  if not bMatchCase then begin
    subStrTmp := uppercase(subStrTmp);
    strTmp    := uppercase(strTmp);
  end;
  if bWHoleword then begin
    result := MatchWord(SubStrTmp, strTmp );
  end else begin
    result := pos(SubStrTmp, strTmp) > 0;
  end;
end;

function MatchWord( const SubStr, str : string) : boolean;
var
 iPos, i : integer;
 bStartOk, bEndOk : boolean;
begin
    result := false;
    iPos := pos(SubStr, str);
    while iPos > 0 do begin
      bStartOk := true;
      for i := iPos-1 downto 1 do begin
        if str[i] = ' ' then
          break;
        if upcase(str[i]) in ['A'..'Z'] then begin
          bStartOk := false;
          break;
        end;
      end;//for
      bEndOk := true;
      if bStartOk then begin
        for i := iPos+1 to length(str) do begin
          if str[i] = ' ' then
            break;
          if upcase(str[i]) in ['A'..'Z'] then begin
            bEndOK := false;
            break;
          end;
        end;
      end;//if startok
      if bStartOk and bEndOk then begin
        result := true;
        break;
      end;
      iPos := posex(subStr, str, iPos+1);
    end;
end;

procedure AddShortCutsToActionHints( al : TCustomActionList );
var
  i, iPos : integer;
  action : TCustomAction;
begin
  for i := 0 to al.ActionCount-1 do begin
     if not (al.actions[i] is TCustomAction) then continue;
     action := al.actions[i] as TCustomAction;
     if (action.Hint<>'') and (shortcutTotext(action.ShortCut)<>'') then begin
       iPos := pos('|', action.Hint);
       if iPos = 0 then
         action.Hint := action.Hint + ' (' + shortcutTotext(action.ShortCut) + ')|' + action.Hint
       else
         action.Hint := copy(action.hint, 1, iPos-1) + ' ('+ shortcutTotext(action.ShortCut) + ')'+
                        copy(action.hint, iPos, length(action.hint) );
     end;
  end;
end;


function hasEllipse( const str : string ) : boolean;

begin

   result := pos('...', str) = length(str)-2;

end;


function CompareNoEllipse( const str1, str2 : string ) : boolean;
var

 strTmp1, strTmp2 : string;

begin

  strTmp1 := str1;

  strTmp2 := str2;

  if hasEllipse(strTmp1) then

    strTmp1 := copy(strTmp1, 1, length(strTmp1)-3);

  if hasEllipse(strTmp2) then

    strTmp2 := copy(strTmp2, 1, length(strTmp2)-3);

  result := strTmp1 = strTmp2;

end;

function IncludeEllipse( const str : string ) : string;
begin
 result := str;
 if not hasEllipse(str) then
   result := result + '...';
end;

function ExcludeEllipse( const str : string ) : string;
begin
 result := str;
 if hasEllipse(str) then
   result := copy(result, 1, length(result)-3);
end;

end.
