{ *************************************************************************** }
{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }

unit uEditSourceItemFm;

interface

uses
  QGrids, uTextLists,
  SysUtils, Types, Classes, Variants, QTypes, QGraphics, QControls, QForms,
  QDialogs, QStdCtrls, QExtCtrls, QComCtrls, QActnList, QImgList;

type
  TEditSourceItemFm = class(TForm)
    panTop: TPanel;
    memComment: TMemo;
    memOldText: TMemo;
    memCurrentText: TMemo;
    cbReadOnly: TCheckBox;
    cbHide: TCheckBox;
    cbCorrection: TCheckBox;
    Label1: TLabel;
    ImageList1: TImageList;
    tbBottom: TToolBar;
    ActionList1: TActionList;
    actFirst: TAction;
    ToolButton1: TToolButton;
    ToolButton2: TToolButton;
    actUp: TAction;
    actDown: TAction;
    actLast: TAction;
    ToolButton3: TToolButton;
    ToolButton4: TToolButton;
    lblState: TLabel;
    actReadOnly: TAction;
    actHide: TAction;
    actCorrection: TAction;
    lblStat: TLabel;
    procedure FormShow(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure actFirstExecute(Sender: TObject);
    procedure actFirstUpdate(Sender: TObject);
    procedure actLastExecute(Sender: TObject);
    procedure actLastUpdate(Sender: TObject);
    procedure actUpExecute(Sender: TObject);
    procedure actUpUpdate(Sender: TObject);
    procedure actDownExecute(Sender: TObject);
    procedure actDownUpdate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure actReadOnlyExecute(Sender: TObject);
    procedure actHideExecute(Sender: TObject);
    procedure actCorrectionExecute(Sender: TObject);
    procedure memCommentKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    { Private-Deklarationen }
    m_Grid : TStringGrid;
    m_SourceTextList : TSourceTextList;
    procedure ShowContent;
    procedure SaveContent;
    function HasGriddata : boolean;
  public
    { Public-Deklarationen }
    constructor Create( AOwner : TComponent; SourceTextList : TSourceTextList; grid : TStringGrid ); reintroduce;
  end;

var
  EditSourceItemFm: TEditSourceItemFm;

implementation

{$R *.xfm}

uses
  math, uSourceListFm, uLanguage, uGlobal, uApp, QMenus, qt;

var
  iLeft : integer = -1;
  iTop : integer = -1;
  iWidth : integer = -1;
  iHeight : integer = -1;
  
constructor TEditSourceItemFm.Create( AOwner : TComponent;  SourceTextList : TSourceTextList; grid : TStringGrid );
begin
   inherited Create(AOwner);
   m_grid := grid;
   m_SourceTextList := SourceTextList;
end;

procedure TEditSourceItemFm.FormShow(Sender: TObject);
begin
   if (iLeft<>-1) and (iTop<>-1) and (iWidth<>-1) and (iHeight<>-1) then begin
      setbounds(iLeft, iTop, iWidth, iHeight);
   end else begin
      Left := screen.Width div 2 - width div 2;
      Top  := screen.Height div 2 - Height div 2;
   end;

   ShowContent;
   memCurrentText.Hint := memCurrentText.Hint + ' - ' + GetLangID(m_SourceTextList.ParentSource.LangNo);
   actUp.ShortCut    := shortcut(key_up, [ssCtrl]);
   actDown.ShortCut  := shortcut(key_down, [ssCtrl]);
   actFirst.ShortCut := shortcut(key_Home, [ssCtrl]);
   actLast.ShortCut  := shortcut(key_end, [ssCtrl]);
   AddShortCutsToActionHints( actionList1 );
end;

procedure TEditSourceItemFm.FormDestroy(Sender: TObject);
begin
   SaveContent;
   iLeft   := Left;
   iTop    := top;
   iWidth  := Width;
   iHeight := Height;
end;

procedure TEditSourceItemFm.FormResize(Sender: TObject);
var
  h : integer;
begin
  h := (clientheight - panTop.Height - tbBottom.Height) div 3;
  memComment.Height     := h;
  memOldText.Top        := memComment.top + h;
  memOldText.Height     := h;
  memCurrentText.Top    := memOldText.top + h;
  memCurrentText.Height := h;
end;

function TEditSourceItemFm.HasGriddata : boolean;
begin
   result := (m_Grid.RowCount >= 2) and ( assigned(m_Grid.objects[0,1]) );
end;

procedure TEditSourceItemFm.ShowContent;
var
  sourceItem : TSourceItem;
resourcestring
  rsCaption = 'Source String %d - String %s';
  rsIn = 'in';
  rsChanged = 'changed';
  rsNew = 'new';
begin
   sourceItem := m_grid.objects[0,m_grid.row] as TSourceItem;
   caption := format(rsCaption, [sourceitem.No, stripID(sourceitem.ID)]);
   if sourceitem.ParentSourceResItem.ResName <> '' then
     caption := caption + ' '+rsIn+' ' + uppercase(sourceitem.ParentSourceResItem.ResName);
   actReadonly.checked   := sourceitem.ReadOnly;
   actHide.checked       := sourceitem.Hidden;
   actCorrection.checked := sourceitem.Correction;
   actCorrection.enabled := sourceitem.UpdateState = usChanged;
   memComment.Text := sourceitem.Comment;
   memComment.Modified := false;
   memOldText.Text := sourceitem.OldText;
   memCurrentText.Text := sourceitem.Text;
   lblState.Caption := '';
   case sourceitem.UpdateState of
     usChanged: lblState.Caption := rsChanged;
     usNew:     lblState.Caption := rsNew;
   end;
   lblStat.caption := format('C: %d'#10'L: %d', [getCharCount(sourceitem.Text), getLineCount(sourceitem.Text)]);
end;

procedure TEditSourceItemFm.SaveContent;
var
  sourceItem : TSourceItem;
  bModified : boolean;
begin
   sourceItem := m_grid.objects[0,m_grid.row] as TSourceItem;
   bModified := (sourceItem.ReadOnly <> actReadOnly.Checked) or
                (sourceItem.Hidden <> actHide.Checked) or
                (sourceItem.Correction <> actCorrection.Checked) or
                 memComment.modified;
   sourceItem.ReadOnly   := actReadOnly.Checked;
   sourceItem.Hidden     := actHide.Checked;
   sourceItem.Correction := actCorrection.Checked;
   sourceItem.Comment    := memComment.Text;
   (Owner as TSourceListFm).UpdateCurrentTextListLine;
   if bModified then
     (Owner as TSourceListFm).SetModified(true);
end;

procedure TEditSourceItemFm.actFirstExecute(Sender: TObject);
begin
   SaveContent;
   m_Grid.Row := 1;
   ShowContent;
end;

procedure TEditSourceItemFm.actFirstUpdate(Sender: TObject);
begin
   actFirst.Enabled := HasGridData and (m_Grid.Row > 1);
end;

procedure TEditSourceItemFm.actLastExecute(Sender: TObject);
begin
   SaveContent;
   m_Grid.Row := m_Grid.RowCount-1;
   ShowContent;
end;

procedure TEditSourceItemFm.actLastUpdate(Sender: TObject);
begin
   actLast.Enabled := HasGridData and (m_Grid.Row < m_Grid.RowCount-1);
end;

procedure TEditSourceItemFm.actUpExecute(Sender: TObject);
begin
   SaveContent;
   m_Grid.Row := m_Grid.Row-1;
   ShowContent;
end;

procedure TEditSourceItemFm.actUpUpdate(Sender: TObject);
begin
   actUp.Enabled := HasGridData and (m_Grid.Row > 1);
end;

procedure TEditSourceItemFm.actDownExecute(Sender: TObject);
begin
   SaveContent;
   m_Grid.Row := m_Grid.Row+1;
   ShowContent;
end;

procedure TEditSourceItemFm.actDownUpdate(Sender: TObject);
begin
   actDown.Enabled := HasGridData and (m_Grid.Row < m_Grid.RowCount-1);
end;

procedure TEditSourceItemFm.actReadOnlyExecute(Sender: TObject);
begin
   actReadOnly.Checked := not actReadOnly.Checked;
end;

procedure TEditSourceItemFm.actHideExecute(Sender: TObject);
begin
  actHide.Checked := not actHide.Checked;
end;

procedure TEditSourceItemFm.actCorrectionExecute(Sender: TObject);
begin
  if actCorrection.Enabled then
     actCorrection.Checked := not actCorrection.Checked;
end;

procedure TEditSourceItemFm.memCommentKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
var
  i : integer;
  action : TCustomAction;
begin
  for i := 0 to actionList1.ActionCount-1 do begin
     if not (actionList1.actions[i] is TCustomAction) then continue;
     action := actionList1.actions[i] as TCustomAction;
     if (shortcut(key, shift) = action.shortcut) and action.enabled and action.visible then begin
        key := 0;
        action.execute;
        break;
     end;
  end;
end;

end.
