{ *************************************************************************** }

{                                                                             }
{ This file is part of the LinLocalize project                                }
{                                                                             }
{ Copyright (c) 2003                                                          }
{ Jens Khner <kuehner@users.sourceforge.net>                                 }
{                                                                             }
{ This program is free software; you can redistribute it and/or               }
{ modify it under the terms of the GNU General Public                         }
{ License as published by the Free Software Foundation; either                }
{ version 2 of the License, or (at your option) any later version.            }
{                                                                             }
{ This program is distributed in the hope that it will be useful,             }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           }
{ General Public License for more details.                                    }
{                                                                             }
{ You should have received a copy of the GNU General Public License           }
{ along with this program; see the file COPYING.  If not, write to            }
{ the Free Software Foundation, Inc., 59 Temple Place - Suite 330,            }
{ Boston, MA 02111-1307, USA.                                                 }
{                                                                             }
{ *************************************************************************** }


unit uApp;

interface

uses
 Contnrs, uTextLists, classes;

type
  TApp = class
    private
      m_strResbindLocation : string;
      m_bAutoSaveStringLists : boolean;
      m_bOutputVisible : boolean;
      m_iOutputHeight : integer;
      m_lstGlossaries : TObjectList;
      m_lstMRU : TStringList;
      m_bEnterSelNext : boolean;
      m_lstProjects : TObjectList;
      function getGlossaryDir : string;
    public
      constructor Create;
      destructor destroy; override;
      procedure ReadFromIni;
      procedure WriteToIni;
      function Translate( transItem : TTransitem ) : boolean; overload;
      function Translate( transItem : TTransitem; const strSource : string; out strTrans : string) : integer; overload;
      function CheckTranslation( transItem : TTransitem ) : integer;
      function FormatCheckErr( transItem : TTransItem; iErr : integer ) : string;
      procedure AddMRU( const strFileName : string );
      procedure ReloadGlossary( const strFileName : string );
      property ResbindLocation : string read m_strResbindLocation write m_strResbindLocation;
      property GlossaryDir : string read GetGlossaryDir;
      property AutoSaveStringLists : boolean read m_bAutoSaveStringLists write m_bAutoSaveStringLists;
      property OutputVisible : boolean read m_bOutputVisible write m_bOutputVisible;
      property OutputHeight : integer read m_iOutputHeight write m_iOutputHeight;
      property Glossaries : TObjectList read m_lstGlossaries;
      property MRU : TStringList read m_lstMRU;
      property Projects : TObjectList read m_lstProjects;
      property EnterSelNext : boolean read m_bEnterSelNext write m_bEnterSelNext;
  end;

var
  g_App : TApp;

implementation

uses
  IniFiles, SysUtils, uGlossary, uProject, uGlobal;

resourcestring
  rsCheckTabs = 'Different number of tabulators.';
  rsMissingAccesKeyInTrans = 'Missing access-key (''&'') in translated string.';
  rsMissingAccesKeyInSource = 'Translated string has access-key, but source-string has none.';
  rsCheckSpace = 'Differnet numer of white-spaces at beginning or end of string.';
  rsCheckFormat = 'Inconsistent printf/format-formats.';
  rsMissingEllipseInTrans = 'Missing ellipse (''...'') in translated string.';
  rsMissingEllipseInSource = 'Translated string has ellipse (''...''), but source-string has none.';

const
  astrCheckErr : array[1..7] of string = (rsCheckTabs,
                                          rsMissingAccesKeyInTrans,
                                          rsMissingAccesKeyInSource,
                                          rsCheckSpace,
                                          rsCheckFormat,
                                          rsMissingEllipseInTrans,
                                          rsMissingEllipseInSource);

constructor TApp.Create;
begin
   inherited;
   m_lstGlossaries        := TObjectList.Create;
   m_lstProjects          := TObjectList.create;
   m_lstMRU               := TStringList.Create;
   m_strResbindLocation   := '/opt/kylix3/bin/resbind';
   m_bAutoSaveStringLists := false;
   m_bEnterSelNext        := true;
   m_bOutputVisible       := true;
   m_iOutputHeight        := 120;
end;

destructor TApp.destroy;
begin
  m_lstGlossaries.Free;
  m_lstProjects.free;
  m_lstMRU.free;
  inherited;
end;

function TApp.getGlossaryDir : string;
begin
  result := extractFilePath( Paramstr(0) ) + 'glossaries/';
end;

procedure TApp.ReadFromIni;
var
  i : integer;
  Glossary : TGlossary;
  strFileName : string;
  sl : TStringList;
procedure ReadGlossary( const strFileName : string);
begin
   if fileexists(strFileName) then begin
       Glossary         := TGlossary.create;
       try
         Glossary.readFromFile( strFileName );
         Glossary.enabled := true;
         m_lstGlossaries.Add(Glossary);
       except
         Glossary.Free;
       end;
   end;
end;
begin
  with TIniFile.Create( changefileext(paramstr(0), '.ini' )) do begin
     m_strResbindLocation   := readstring('Options', 'ResbindLocation', m_strResbindLocation);
     m_bAutoSaveStringLists := readBool('Options', 'AutoSaveStringLists', m_bAutoSaveStringLists);
     m_bEnterSelNext        := readBool('Options', 'EnterSelectsNext', m_bEnterSelNext);

     m_bOutputVisible       := readBool('Output', 'OutputVisible', m_bOutputVisible);
     m_iOutputHeight        := readInteger('Output', 'OutputHeight', m_iOutputHeight);

     sl := TStringList.create;
     readsectionvalues('MRU', sl);
     for i := 0 to sl.count-1 do begin
        if fileexists(sl.ValueFromIndex[i]) then
           m_lstMRU.add(sl.ValueFromIndex[i]);
     end;
     sl.free;

     if sectionExists('Glossaries') then begin
       for i := 0 to 99 do begin
          if ValueExists( 'Glossaries', 'FileName'+inttostr(i) ) then begin
            strFileName := readstring('Glossaries', 'FileName'+inttostr(i), '' );
            if fileexists(strFileName) then begin
              Glossary := TGlossary.create;
              try
               Glossary.readFromFile( strFileName );
               Glossary.enabled := readBool('Glossaries', 'Enabled'+inttostr(i), true);
               m_lstGlossaries.Add(Glossary);
              except
                Glossary.Free;
              end;
            end;
          end else
            break;
       end;
     end else begin
        strFileName := GetGlossaryDir+'kylix_rtl.glo';
        ReadGlossary(strFileName);
        strFileName := GetGlossaryDir+'kylix_vcl.glo';
        ReadGlossary(strFileName);
        strFileName := GetGlossaryDir+'kylix_visualclx.glo';
        ReadGlossary(strFileName);
        strFileName := GetGlossaryDir+'kylix_dataclx.glo';
        ReadGlossary(strFileName);
     end;//if

  end;//with
end;

procedure TApp.WriteToIni;
var
  i : integer;
  Glossary : TGlossary;
begin
  with TIniFile.Create( changefileext(paramstr(0), '.ini' )) do begin
     writestring('Options', 'ResbindLocation', m_strResbindLocation);
     writeBool('Options', 'AutoSaveStringLists', m_bAutoSaveStringLists);
     writeBool('Options', 'EnterSelectsNext', m_bEnterSelNext);

     WriteBool('Output', 'OutputVisible', m_bOutputVisible);
     WriteInteger('Output', 'OutputHeight', m_iOutputHeight);

     EraseSection('MRU');
     for i := 0 to m_lstMRU.count-1 do begin
       writestring('MRU', 'File'+inttostr(i), m_lstMRU[i]); 
     end;

     EraseSection('Glossaries');
     for i := 0 to m_lstGlossaries.Count-1 do begin
        Glossary := (m_lstGlossaries[i] as TGlossary);
        writestring('Glossaries', 'FileName'+inttostr(i), Glossary.filename);
        writeBool('Glossaries', 'Enabled'+inttostr(i), Glossary.Enabled);
     end;
     UpdateFile;
  end;
end;

procedure TApp.AddMRU( const strFileName : string );
var
  i : integer;
begin
  i := m_lstMRU.IndexOf(strFilename);
  if i > -1 then begin
    m_lstMRU.Move(i, 0);
  end else begin
    m_lstMRU.insert(0, strFileName);
  end;
  while m_lstMRU.count > 10 do
    m_lstMRU.delete( m_lstMRU.count-1 );
end;

function TApp.Translate( transItem : TTransitem; const strSource : string; out strTrans : string) : integer;
var
  TransTextList : TTransTextList;
  project : TProject;
  iCount : integer;
  i : integer;
begin
  result := 0;
  strTrans := '';

  TransTextList := Transitem.ParentTransResItem.ParentList as TTransTextList;
  if assigned( TranstextList.ParentTranslation ) then
    project := TranstextList.ParentTranslation.ParentSource.ParentProject
  else
    project := nil;

  iCount := TransTextList.Translate(TransItem, strSource, strTrans);
  if iCount > 0 then begin
     result := iCount;
     exit;
  end;
  if assigned(project) then begin
     for i := 0 to project.Glossaries.count-1 do begin
        iCount := (project.Glossaries[i] as TGlossary).Translate(TranstextList.SourceLangNo, strSource,
                                                                 TransTextList.TransLangNo, strTrans);
        if iCount > 0 then begin
          result := iCount;
          exit;
        end;
     end;
  end;

  for i := 0 to m_lstGlossaries.count-1 do begin
     iCount := (m_lstGlossaries[i] as TGlossary).Translate(TranstextList.SourceLangNo, strSource,
                                                           TransTextList.TransLangNo, strTrans);
     if iCount > 0 then begin
        result := iCount;
        exit;
     end;
  end;
end;

function TApp.Translate( transItem : TTransitem ) : boolean;
var
  strTrans : string;
begin
  result := false;
  if Translate(Transitem, Transitem.SourceText, strTrans) > 0 then begin
        Transitem.TransText   := strTrans;
        Transitem.Translated  := true;
        TransItem.Review      := true;
        result := true;
  end;
end;

function TApp.CheckTranslation( transItem : TTransitem ) : integer;
begin
  result := 0;
  if getcharcount(Transitem.SourceText, #9) <> getcharcount(Transitem.TransText, #9) then begin
    result := 1;
    exit;
  end;
  if (getcharcount(Transitem.SourceText, '&') > 0) and (getcharcount(Transitem.TransText, '&')=0) then begin
    result := 2;
    exit;
  end;
  if (getcharcount(Transitem.SourceText, '&') = 0) and (getcharcount(Transitem.TransText, '&')>0) then begin
    result := 3;
    exit;
  end;
  if (getBeginSpaces(Transitem.SourceText) <> GetBeginSpaces(Transitem.TransText)) or
     (getEndSpaces(Transitem.SourceText) <> GetEndSpaces(Transitem.TransText)) then
  begin
    result := 4;
    exit;
  end;
  if getcharcount(Transitem.SourceText, '%') <> getcharcount(Transitem.TransText, '%') then begin
    result := 5;
    exit;
  end;
  if (length(Transitem.SourceText)>=3) and (length(Transitem.TransText)>=3) then begin
    if HasEllipse(Transitem.SourceText) and not HasEllipse(Transitem.TransText) then begin
      result := 6;
      exit;
    end;
    if not HasEllipse(Transitem.SourceText) and HasEllipse(Transitem.TransText) then begin
      result := 7;
      exit;
    end;
  end;
end;

function TApp.FormatCheckErr( transItem : TTransItem; iErr : integer ) : string;
begin
  result := format('(%d) string %s, error %d: %s',
                   [TransItem.No, TransItem.ID, iErr, astrCheckErr[iErr]]);
end;

procedure TApp.ReloadGlossary( const strFileName : string );
var
  i, t : integer;
  project : TProject;
begin
  for i := 0 to m_lstGlossaries.Count-1 do begin
     if SameFilename( (m_lstGlossaries[i] as TGlossary).FileName, strFileName ) then
         (m_lstGlossaries[i] as TGlossary).ReadFromFile(strFileName);
  end;
  for i := 0 to m_lstProjects.Count-1 do begin
    project := m_lstProjects[i] as TProject;
    for t := 0 to project.Glossaries.Count-1 do begin
       if SameFilename( (project.Glossaries[t] as TGlossary).FileName, strFileName ) then
           (project.Glossaries[t] as TGlossary).ReadFromFile(strFileName);
    end;
  end;
end;

initialization
  g_App := TApp.Create;
finalization
  g_App.free;
end.
