# LIBUPDTE.TCL  - Setup procedures for implementing library-update wizard page
#
# Copyright 1999 Wind River Systems, Inc
#
# modification history
# --------------------
# 01c,19mar99,wmd  Output to a file any debug messages.
# 01b,28jan99,tcy  moved procs here from INSTALL.TCL
# 01a,26jan99,tcy  extracted from INSTW32.TCL.
#

#############################################################################
#
# pageCreate(libUpdate) - update the library archives and displays progess in
#                         updating libraries
#
# This procedure will update the library archives and displays progess in
# updating libraries
#
# SYNOPSIS
# .tS
# pageCreate(libUpdate)
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc pageCreate(libUpdate) {} {
    global ctrlVals

    set ctrlVals(numBbrd) 0
    meterCreate [strTableGet 1180_LIB_UPDATE]
    set w [dlgFrmCreate [strTableGet 1480_TITLE_LIBUPDATE]]

    controlPropertySet $ctrlVals(meterWindow).$ctrlVals(meterWg) \
                            -background Blue -foreground Black

    controlEnable $w.backButt 0
    controlEnable $w.nextButt 0

    librariesUpdate

    meterDestroy $ctrlVals(meterWindow)

    controlEnable $w.backButt 1
    controlEnable $w.nextButt 1

    # test automation

    if { $ctrlVals(useInputScript) } {
        autoSetupLog "Library update page:"
        autoSetupLog "\tUpdated library"
    }

    nextCallback
}

#############################################################################
#
# pageProcess(libUpdate) - process inputs from libUpdate page if necessary
#
# This procedure will process inputs from libUpdate page if necessary
#
# SYNOPSIS
# .tS
# pageProcess(libUpdate)
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: 1 when successful
#
# ERRORS: N/A
#

proc pageProcess(libUpdate) {} {
    return 1
}

##############################################################################
#
# libUpdateHelper - invokes make to carry out the requested action.
#
# a -n option is used to obtain a list of commands that 'make' supposes to
# carry out.  Each command is then evaluated separately.  Asking make to carry
# out the real action will cause many console windows to pop up.
#
# SYNOPSIS
# libUpdateHelper <cpu> <tool> <type> <action> [arFlags] [objs]
#
# PARAMETERS:
#    <cpu> : CPU family, i.e. MC68000
#    <tool>: compiler family, i.e 'gnu'
#    <type>: library type, i.e. 'vx'
#    <action>: build target, i.e. 'ar' or 'ranlib'
#    [arFlags] : the default value is -cru
#    [objs] : object file list separated by space, needed when <action> is ar
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc libUpdateHelper {cpu tool type action {arFlags "-cru"} {objs ""}} {

    set binDir [file join [destDirGet] host [windHostTypeGet] bin]
    regsub -all {\\} $binDir {/} binDir

    if {![catch {exec $binDir/make -n -f [makefileCreate] CPU=$cpu \
                              TOOL=$tool TYPE=$type \
                              OBJS=$objs \
                              ARFLAGS=$arFlags $action} retVal]} {

        regsub -all {\\} $retVal {/} retVal

        if {[catch {eval "exec $binDir/$retVal"} error1]} {
            if {[catch {eval "exec $retVal"} error2]} {
                puts "$error1\n$error2"
                uninstLog setup "\t$error1"
                uninstLog setup "\t$error2"
            } else {
                dbgputs "\n$retVal"
            }
        } else {
            dbgputs "\n$binDir/$retVal"
        }
    } else {
        puts $retVal
        uninstLog setup "\t$retVal"
    }
}

##############################################################################
#
# librariesUpdate - updating the vxWorks libraries.
#
# This routine walks thru each product, updates the vxWorks libraries w/ the
# object list which was obtained and saved by the archListPut() function.  The
# default updating flag is -cru.  A particular product can override this
# default setting by provide a postInstall.tcl file that contains a call to
# the arFlagsSet() function.  To speed up the archiving process, a minimum
# amount of AR invocations is made. This is achieved by grouping the object
# list into smaller lists of distinct flags and libraries, then archiving
# each list of objects via an AR invocation.
#
#
# SYNOPSIS
# librariesUpdate
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc librariesUpdate {} {
    global setupVals
    global objects

    set setupVals(cancel) 0

    # Determine the total number of new object files for GUI updating purpose.

    set totalObjFile 0

    foreach objDir [array names objects] {
        incr totalObjFile [llength $objects($objDir)]
    }

    set percent 0

    # Updating the libraries.

    if {$totalObjFile != "0"} {
        meterUpdate $percent ""
        foreach prodIndex [cdInfoGet selectedProdIndexList] {

            searchAndProcessSection ArFlags [chooseInfFile $prodIndex] \
               $prodIndex
            set prodARFlag [arFlagsGet $prodIndex]

            foreach archKey [array names objects] {

                # typex is useful when updating special component.
                # ex: windview,...

                if {[regexp {^(.*),(.*),(.*),(.*),(.*),(.*)$} $archKey \
                    junk index obj cpu tool type typex] && \
                    "$prodIndex" == "$index"} {

                    # An arPassId is characterized by the following variables:
                    # "$prodARFlag,$cpu,$tool,$type,$typex,". It corresponds
                    # to one AR call. For example,
                    # prodARFlag,   cpu, tool, type, typex,
                    #       -cru, R4650,  gnu,   vx,    wv,

                    set arPassId $prodARFlag,$cpu,$tool,$type,$typex,

                    # Any objects that have these properties are "AR"ed
                    # together. typex is optional.

                    #initialize objs and objsArchived if necessary
                    if ![info exists objs($arPassId)] {
                        set objs($arPassId) ""
                    }

                    if ![info exists objsArchived($arPassId) ] {
                        set objsArchived($arPassId) 0
                    }

                    # now group objects by arPassId and update the
                    # number of objects archived for each arPassId
                    foreach objFile $objects($archKey) {
                        set objs($arPassId) "$objFile $objs($arPassId)"
                        incr objsArchived($arPassId)
                    }
                }
            }
        }

        set numArchivedSoFar 0
        foreach arPassId [array name objsArchived] {

            if {"$setupVals(cancel)" == "1"} { return }

            # extract elements from arPassId
            if {[regexp {^(.*),(.*),(.*),(.*),(.*),(.*)$} $arPassId \
                junk prodARFlag cpu tool type typex] } {

                set archiveFile [file join [destDirGet] target lib \
                        lib$cpu$tool$type.a]

                if {[file exists $archiveFile]} {
                    meterUpdate $percent "Updating lib$cpu$tool${type}.a"

                    # don't backup library if this is the first installation
                    if {[file extension $setupVals(uninstFile)] != ".001"} {
                        backup target/lib/lib$cpu$tool$type.a
                        # immediately flush the queue; otherwise we may not
                        # be backing up the old archive
                        backupFileQueueFlush
                    }

                    cd [file join [destDirGet] target lib \
                        obj$cpu$tool$type$typex]

                    set numArchivedSoFar \
                        [expr $numArchivedSoFar + $objsArchived($arPassId)]

                    set percent [expr $numArchivedSoFar * 100 / $totalObjFile]

                    libUpdateHelper $cpu $tool $type ar \
                        $prodARFlag $objs($arPassId)

                    # Ranlib to update the lib table of content.
                    libUpdateHelper $cpu $tool $type ranlib

                    meterUpdate $percent "Updating lib$cpu$tool${type}.a"

                    dbgputs "                       AR pass Id: $arPassId"
                    dbgputs "                          Library:\
                         lib$cpu$tool$type.a"
                    dbgputs "                 Object directory: [pwd]"
                    dbgputs "                          ARFLAGS: $prodARFlag"
                    dbgputs "Number of objects in this AR pass: \
                            $objsArchived($arPassId)"
                    dbgputs "Number of objects archived so far: \
                            $numArchivedSoFar"
                    dbgputs "          Percent archived so far: \
                            $percent% of $totalObjFile objects"
                    dbgputs "------------------------------------------------"

                } else {
                    dbgputs "skip updating $archiveFile : no such file"
                }
            } else {
                puts "librariesUpdate: cannot get arPassId"
            }
        }
        meterUpdate 100 ""
    }
}

##############################################################################
#
# makefileCreate - creating a make file for use in the updating library step.
#
# SYNOPSIS
# makefileCreate
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc makefileCreate {} {

    if {[windHostTypeGet] == "x86-win32"} {
        set Makefile "[tempDirGet]\\Makefile"
    } else {
        set Makefile "[tempDirGet]/Makefile"
    }

    if ![file exists $Makefile] {
        set f [open $Makefile "w"]

        if {"[windHostTypeGet]" == "x86-win32"} {
            puts $f "include [destDirGet]\\target\\h\\make\\make.$(CPU)$(TOOL)"
            puts $f "ARFLAGS = -cru"
            puts $f "ar:"
            puts $f "\t- $(AR) $(ARFLAGS) ..\\lib$(CPU)$(TOOL)$(TYPE).a $(OBJS)"
            puts $f "ranlib:"
            puts $f "\t- $(RANLIB) ..\\lib$(CPU)$(TOOL)$(TYPE).a"
        } else {
            puts $f "include [destDirGet]/target/h/make/make.$(CPU)$(TOOL)"
            puts $f "ARFLAGS = -cru"
            puts $f "ar:"
            puts $f "\t- $(AR) $(ARFLAGS) ../lib$(CPU)$(TOOL)$(TYPE).a $(OBJS)"
            puts $f "ranlib:"
            puts $f "\t- $(RANLIB) ../lib$(CPU)$(TOOL)$(TYPE).a"
        }
        close $f
    }

    return $Makefile
}
