# COMPSELCT.TCL - Setup procedures for implementing component-selection
#                 wizard page
#
# Copyright 1999 Wind River Systems, Inc
#
# modification history
# --------------------
# 01t,21apr99,wmd  Fix spr #26372 - Unselecting "non-native" products should
#                  not display the warning message.
# 01s,12apr99,wmd  Need to make sure that "Tornado Simulator" product installs
#                  the drivers product as well.
# 01r,31mar99,wmd  Add another name to check for arch wise: SIMHPPA.
# 01q,25mar99,wmd  Fix cpu identification for simulator products.
# 01p,24mar99,wmd  Try to fix so that x86-winXX products are recognized as
#                  such.
# 01o,22mar99,bjl  remove compatibility page for Unix if tornado not 
#                  installed, added endWaitCursor.
# 01n,19mar99,wmd  Output to a file any debug messages.
# 01m,18mar99,tcy  added further checking for deselecting non-native products
# 01m,17mar99,wmd  Fix the width of the line to display "Grey box..."
#                  completely.
# 01l,11mar99,wmd  Move the tristate checkbox before the code to uncheck
#                  non-native hosts. Fix prodHostTypeGet.
# 01k,11mar99,wmd  For single part products, indicate 1/1 selected, and add
#                  check to 'foreign' hosts into part select OK proc.
# 01j,10mar99,wmd  Use the tri-state check box for the selected products.
# 01i,08mar99,tcy  fixed bug with code filtering non-native hostnames
# 01h,03mar99,tcy  Show correct size SETUP/ takes based on platform
#                  (fix for SPR 25228)
# 01g,22feb99,wmd  Fix so that unselected products show 0 out of XXX products.
# 01f,17feb99,tcy  Added proc installedProdDescGet() to return a list
#                  of products (fix for SPR 24897) 
#                  fix bug in installedProdGet() when checking dest. directory
# 01e,11feb99,wmd  Remove the apostrophes around the '*' so that it is more
#                  visible.
# 01d,29jan99,wmd  Fix spr #24559, show which products have components.
# 01c,28jan99,wmd  Fix spr # 24523, warn users of non-native installations,
#                  and initially uncheck those products not meant for the host.
# 01b,29jan99,tcy  moved byteToMbyte() and kbyteToMbyte() from INSTALL.TCL
# 01a,26jan99,tcy  extracted from INSTW32.TCL.
#

#############################################################################
#
# pageCreate(compSelect) - display products for users to install
#
# This procedure will display products for users to install
#
# SYNOPSIS
# .tS
# pageCreate(compSelect)
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc pageCreate(compSelect) {} {
    global ctrlVals
    global setupVals

    endWaitCursor

    set ctrlVals(volatileFrm) [list \
            [list label -name msg \
                        -title [strTableGet 1140_COMP_SELECT]\
                        -x 100 -y 9 -w 205 -h 26] \
            [list label -name message2 \
                        -title "&Products (Grey box indicates additional components selectable.)" \
                        -x 100 -y 37 -w 210 -h 8] \
            [list checklist -name optionList \
                            -callback onClickProdCheckList \
                            -x 100 -y 46 -w 207 -h 80] \
            [list label -name reqText \
                        -title "Space Required:" \
                        -x 107 -y 124 -w 55 -h 8] \
            [list label -name reqSpace \
                        -title "Unknown" \
                        -x 257 -y 124 -w 43 -h 8] \
            [list label -name availText \
                        -title "Space Available:" \
                        -x 107 -y 134 -w 55 -h 8] \
            [list label -name availSpace \
                        -title "Unknown" \
                        -x 257 -y 134 -w 43 -h 8] \
            [list frame -name frm1 \
                        -xpos 100 -ypos 148 -width 207 -height 25] \
            [list label -name compSelected \
                        -x 107 -y 153 -w 140 -h 18] \
            [list button -title "&Details..." -name partDetailButt  \
                        -xpos 249 -ypos 153 -width 50 -height 14  \
                        -callback partListDlg] \
    ]

    set w [dlgFrmCreate [strTableGet 1450_TITLE_OPTION]]

    productGuiChkLstCreate
    productGuiSizeUpdate

    # test automation

    if { $ctrlVals(useInputScript) } {
        autoSetupLog "Products Selection page:"
        autoSetupLog "\tInstalled products specified by the key"
        nextCallback
    }
}

#############################################################################
#
# preventInstallation - prevents users from continuing with installation
#
# This procedure will invoke a dialog box to stop users from continuing
# will installation
#
# SYNOPSIS
# .tS
# preventInstallation <ver>
# .tE
#
# PARAMETERS:
# .IP ver
# version of Tornado
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc preventInstallation {ver} {

    set message "You are attempting to install over an existing $ver\n\
                tree.  This is not recommended and will cause \
                incompatibilities\n\
                which will render the product unusable.\n\n\
                Please push \"OK\" to choose a new installation path."\

    dialog stop Setup $message
}
#############################################################################
#
# installedProdDescGet - return a list of installed product descriptions 
#                   that were installed irrespective of the pool Id
#
# This procedure returns a list of installed product descriptions that were 
# installed irrespective of the pool Id
#
# SYNOPSIS
# .tS
# installedProdDescGet <pool>
# .tE
#
# PARAMETERS:
#
# RETURNS: 
# The variable setupVals(instProdDescList) will be set after invoking
# the procedure. The variable will be set to NULL if Setup does not detect any 
# products in the destination directory
#
# ERRORS: N/A
#

proc installedProdDescGet {} {
    global setupVals

    # open the setup.log file and determine if there are any installed products
    # from the specified CD pool

    set logfile [file join [destDirGet] setup.log]
    if [catch {open $logfile r} logfd] {
        return ""
    } else {
        set text [split [read $logfd] \n]
        set nlines [llength $text]
        close $logfd

        for {set ix 0} {$ix < $nlines} {incr ix} {
            # iterate each line and search for the pool ID passed in

            set textLine [split [lindex $text $ix ]]
            # check to see if destination dir is same

            set instDir [lindex $textLine 3]
            # make sure case is not an issue when doing comparison

            if {[string tolower $instDir] == [string tolower [destDirGet]]} {
                incr ix
                # accumulate product numbers that are in this group

                set line [split [lindex $text $ix]]
                while {$ix < $nlines} {
                    if {[string range [lindex $line 1] 0 3] == "100-"} {
                        lappend prodDescList "[lrange $line 2 end]"
                    }
                    incr ix
                    set line [split [lindex $text $ix]]
                    if {[lsearch -regexp $line "TDK-"] == 1} {
                        break
                    }
                }
            }
        }
   }

   set setupVals(instProdDescList) $prodDescList
   dbgputs "Products installed: $setupVals(instProdDescList)"
}

#############################################################################
#
# installedProdGet - return a list of installed products that were installed
#                    from the same pool
#
# This procedure returns a list of installed products that were installed
# from the same pool
#
# SYNOPSIS
# .tS
# installedProdGet <pool>
# .tE
#
# PARAMETERS:
# .IP pool
# pool ID in the form of TDK-XXXXX-XX-XX
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc installedProdGet {pool} {
    global setupVals

    set prodList ""
    set poolID [string range $pool 0 [expr [string first "ZC" $pool] -1]]
    if {$poolID == ""} {
        set poolID $pool
    }

    # open the setup.log file and determine if there are any installed products
    # from the specified CD pool

    set logfile [file join [destDirGet] setup.log]
    if [catch {open $logfile r} logfd] {
        return ""
    } else {
        set text [split [read $logfd] \n]
        set nlines [llength $text]
        close $logfd

        for {set ix 0} {$ix < $nlines} {incr ix} {

            # iterate each line and search for the pool ID passed in

            set textLine [split [lindex $text $ix ]]
            if {[lsearch $textLine $poolID*] != -1} {

                # check to see if destination dir is same

                set instDir [lindex $textLine 3]
 
                # make sure case is not an issue when doing comparison
 
                if {[string tolower $instDir] == \
                    [string tolower [destDirGet]]} {
                    incr ix

                    # accumulate product numbers that are in this group

                    set line [split [lindex $text $ix]]
                    while {$ix < $nlines} {
                        if {[string range [lindex $line 1] 0 3] == "100-"} {
                            lappend prodList [lindex $line 1]
                        }
                        incr ix
                        set line [split [lindex $text $ix]]
                        if {[lsearch -regexp $line "TDK-"] == 1} {
                            break
                        }
                    }
                }

            }
        }
   }

   # make the list unique

   set prodList [lunique $prodList]
   set setupVals(instProdList) $prodList
}

#############################################################################
#
# lunique - return a new list with unique elements of the old list
#
# This procedure will return a new list with unique elements of the old list
#
# SYNOPSIS
# .tS
# lunique
# .tE
#
# PARAMETERS:
# .IP oldList
# a list of elements
#
# RETURNS: a new list with unique elements of the old list
#
# ERRORS: N/A
#

proc lunique {oldList} {

    set newList ""
    catch {unset arr} res
    foreach item $oldList {
        set arr($item) 1
    }
    foreach indx [array name arr] {
        lappend newList $indx
    }
    return $newList
}

#############################################################################
#
# analyzeInstFiles - check if any selected products are already installed
#
# This procedure will check if any selected products are already installed.
#
# SYNOPSIS
# .tS
# analyzeInstFiles
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc analyzeInstFiles {} {
    global ctrlVals
    global setupVals

    windowTimerCallbackSet analyzing_setup_files 0 ""
    set setupVals(instProdList) ""
    set setupVals(dupCoreInstall) 0
    set setupVals(duplicateProducts) ""

    beginWaitCursor

    # check and setup list of installed products if any

    installedProdGet $setupVals(CDnumber)

    foreach prodIndex [cdInfoGet selectedProdIndexList] {
        uiEventProcess

        # if there are installed products in the destDir

        if {$setupVals(instProdList) != ""} {
            set prodNum [productInfoGet number $prodIndex]
            if {[lsearch $setupVals(instProdList) $prodNum] != -1} {

                # check for existence of a core product in the selected list

                set productName [productInfoGet name $prodIndex]
                if {$productName == "tornado"} {
                    set setupVals(dupCoreInstall) 1
                } elseif {[productInfoGet coreProd $prodIndex] == "core"} {
                    set setupVals(dupCoreInstall) 1
                }
                lappend setupVals(duplicateProducts) \
                        [productInfoGet desc $prodIndex]
            }
        }
    }
    endWaitCursor
    windowClose analyzing_setup_files
}

#############################################################################
#
# checkForPreviouslyInstalledProds - create dialog box for checking whether
#                                    any selected products have been
#                                    previously installed
#
# SYNOPSIS
# .tS
# checkForPreviouslyInstalledProds
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc checkForPreviouslyInstalledProds {} {
    global ctrlVals
    global setupVals

    set setupVals(coreInstallation) 0

    dialogCreate \
        -name analyzing_setup_files \
        -notitle \
        -parent $ctrlVals(parentDialog) \
        -helpfile $setupVals(setupHelp) \
        -width 200 -height 32 \
        -init {
            controlPropertySet analyzing_setup_files.analyze_ctrl -bold 1; \
            windowTimerCallbackSet analyzing_setup_files \
                        -milliseconds 200 analyzeInstFiles
        } \
        -controls [list \
                [list label -name "analyze_ctrl" -center \
                -title [strTableGet 4020_ANALYZING_BANNER] \
                -x 15 -y 12 -w 170 -h 10 ]
        ]

}

#############################################################################
#
# onDupProdContinue - set dupRetVal to 0 and close the duplicate product
#                     dialog box
#
# This procedure will set dupRetVal to 0 and close the duplicate product
# dialog box
#
# SYNOPSIS
# .tS
# onDupProdContinue
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onDupProdContinue {} {
    global dupRetVal

    set dupRetVal 0
    windowClose duplicate_products_warn
}

#############################################################################
#
# onDupProdSelComp - set dupRetVal to 1 and close the duplicate product
#                     dialog box
#
# This procedure will set dupRetVal to 1 and close the duplicate product
# dialog box
#
# SYNOPSIS
# .tS
# onDupProdSelComp
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onDupProdSelComp {} {
   global dupRetVal

    set dupRetVal 1
    windowClose duplicate_products_warn
}

#############################################################################
#
# dupProdDlgInit - initialize values for dupProdDlg dialog box
#
# This procedure will initialize values for dupProdDlg dialog box
#
# SYNOPSIS
# .tS
# dupProdDlgInit
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc dupProdDlgInit {} {
    global setupVals

    foreach item $setupVals(duplicateProducts) {
        set lst [format %s\r\n $item]
        append prodList $lst
    }
    controlTextSet duplicate_products_warn.dupProdText $prodList
    controlSelectionSet duplicate_products_warn.dupProdText 0 0
}

#############################################################################
#
# dupProdDlg - display warnings if products are being re-installed
#
# This procedure will display warnings if products are being re-installed
#
# SYNOPSIS
# .tS
# dupProdDlg
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: dupRetVal which is set to 1 by onDupProdSelComp or
#                                    0 by onDupProdContinue
#
# ERRORS: N/A
#

proc dupProdDlg {} {
    global setupVals
    global ctrlVals
    global dupRetVal

    set controls [list \
             [list label -title \
                    "Setup has detected that the following products \
                     were previously installed in [destDirGet] \
                     and you are about to re-install them:" \
                     -name dupProdLabel \
                     -xpos 7 -ypos 9 -width 186 -height 30] \
             [list text -name dupProdText -border \
                    -xpos 7 -ypos 37 -width 185 -height 40 \
                    -readonly -multiline -vscroll] \
             [list label -title [strTableGet 1137_DUP_PRODUCT_WARN] \
                     -name dupProdLabel2 \
                     -xpos 7 -ypos 84 -width 186 -height 40] \
             [list button -title "&Continue" -name continue -default \
                          -xpos 8 -ypos 129 -width 67 -height 14 \
                          -callback onDupProdContinue] \
             [list button -title "&Select Products" -name cancelButt \
                          -xpos 127 -ypos 129 -width 67 -height 14  \
                          -callback onDupProdSelComp ] \
    ]

    dialogCreate \
        -name duplicate_products_warn \
        -title Setup\
        -init dupProdDlgInit \
        -parent $ctrlVals(parentDialog) \
        -helpfile $setupVals(setupHelp) \
        -width 200 -height 150 \
        -controls $controls

    return $dupRetVal
}

#############################################################################
#
# pageProcess(compSelect) - process inputs from compSelect page
#
# This procedure will process inputs from compSelect page
#
# SYNOPSIS
# .tS
# pageProcess(compSelect)
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: 0 if the user needs to re-select the products
#          1 if the user is allowed to continue with installation
#          2 if user decides to quit
#
# ERRORS: N/A
#

proc pageProcess(compSelect) {} {
    global ctrlVals
    global setupVals

    set torVer "Tornado 101"
    set retVal 1

    # check to see if installing over a T101 tree and prevent it.

    set t101File "host/resource/tcl/app-config/Tornado/01WindCFG.win32.tcl"
    set t101File [file join [destDirGet] $t101File]
    if {[file exists $t101File]} {

        # don't want the user to overwrite existing t1 tree
        # test automation

        if { $ctrlVals(useInputScript) } {
            autoSetupLog "\tError: Attempt to install over an \
                            existing $torVer tree. \
                            Please choose a new installation path."
            applicationExit
            return 0
        }

        # warn and prevent the user from proceeding with installation

        preventInstallation $torVer

        # pop the user back to the destination path screen

        return [calcPage destDir]
    }


    # check for a core installation and previously installed products

    checkForPreviouslyInstalledProds

    if {$setupVals(dupCoreInstall)} {

        # overwrite existing directory automatically in autoInstall mode
        # test automation

        if { $ctrlVals(useInputScript) } {
            set retVal 1
        } else {
            messageBeep
            catch {dialog base_install_warn "Setup" \
                    [strTableGet 4000_BASE_INSTALL_WARN] \
                    question 0} result

            if {$result == 1 || $result == ""} {
                return [calcPage destDir]
            } else {
                set retVal 1
            }
        }
    }

    set dupProds $setupVals(duplicateProducts)
    if {[lindex $dupProds 0] != ""} {

        # present dialog to request permission to overwrite existing products
        # test automation

        if { $ctrlVals(useInputScript) } {
            autoSetupLog "\tAllowing overwrite of existing products."
        } else {
            messageBeep
            if {[dupProdDlg] == 1} {
                return 0
            }
        }
    }

    set reqSpace [controlValuesGet $ctrlVals(mainWindow).reqSpace]
    set availSpace [controlValuesGet $ctrlVals(mainWindow).availSpace]

    if {[llength [cdInfoGet selectedProdIndexList]] == 0} {
        messageBox [strTableGet 1151_COMP_SELECT_WARN]
        set retVal 0
    }

    # display the warnings file for each selected product

    foreach prodIndex [cdInfoGet selectedProdIndexList] {
        searchAndProcessSection WarningsFile [chooseInfFile $prodIndex]
    }

    if {$reqSpace > $availSpace} {

        # test automation

        if { $ctrlVals(useInputScript) } {
            autoSetupLog "\tWarning: not enough disk space."
            set i 1
        } else {
            messageBeep
            set i [dialog ok_proceed_exit "Setup" \
                  [strTableGet 1150_COMP_SELECT_WARN]]
        }

        switch $i {
            0 { set retVal [calcPage destDir] }
            1 { set retVal 1 }
            2 { quitCallback }
        }
    }

    # this is done now for unix since the folderSelect page is not used
    # for unix

    if {[isUnix]} {
        global tornadoInstalled

        foreach prodIndex [cdInfoGet selectedProdIndexList] {
            set productName [productInfoGet name $prodIndex]
            if {$productName == "tornado"||$productName == "tornado-vxsim"} {
                set tornadoInstalled 1
            }
        }

        if ![info exists tornadoInstalled] {
            pageRemove remoteRegistryHost
            pageRemove compatibility
            pageRemove howTo
        }
    }

    return $retVal
}

#############################################################################
#
# partListDlg - create dialog box for displaying parts within each Tornado
#               product
#
# This procedure will create dialog box for displaying parts within each
# Tornado product
#
# SYNOPSIS
# .tS
# partListDlg
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partListDlg {} {
    global ctrlVals
    global setupVals

    dialogCreate \
        -name partListDlgHnd \
        -title "Select Parts" \
        -width 263 -height 195 \
        -helpfile $setupVals(setupHelp) \
        -init partListDlgInit \
        -parent $ctrlVals(mainWindow) \
        -exit checkProductDependency \
        -controls { \
                { button -title "Cancel" -name cancelButt \
                        -xpos 206 -ypos 173 -width 50 -height 14  \
                        -callback onPartListDlgCancel \
                }
                { button -title "OK" -name okButt -default \
                        -xpos 150 -ypos 173 -width 50 -height 14  \
                        -callback onPartListDlgOk \
                }
                { checklist -name optionList  \
                        -xpos 7 -ypos 45 -width 249 -height 83  \
                        -callback onClickPartCheckList \
                }
                { label -title "Information" -name Information \
                        -xpos 12 -ypos 130 -w 50 -height 8 \
                }
                { frame -name frm1 \
                        -xpos 7 -ypos 140 -width 249 -height 27 \
                }
                { label -title "Space Required:" -name spaceRequireLabel  \
                        -xpos 16 -ypos 149 -width 170 -height 8  \
                }
                { label -title "In the part list below  click \
                                to select items you want to install or\
                                clear items you don't want to install." \
                        -name label3  \
                        -xpos 7 -ypos 7 -width 249 -height 18  \
                }
                { label -title "Parts:" -name label2  \
                        -xpos 8 -ypos 31 -width 86 -height 8  \
                }
        }
}

#############################################################################
#
# partListDlgInit - initialize partListDlg by calling partGuiUpdate
#
# This procedure will initialize partListDlg by calling partGuiUpdate
#
# SYNOPSIS
# .tS
# partListDlgInit
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partListDlgInit {} {
    partGuiUpdate 0
}

#############################################################################
#
# currSelProdIndexSet - save the index of a product in a global array ctrlVals
#
# This procedure will  save the index of a product in a global array ctrlVals
#
# SYNOPSIS
# .tS
# currSelProdIndexSet <index>
# .tE
#
# PARAMETERS:
# .IP index
# a indexing number to the product list
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc currSelProdIndexSet {index} {
    global ctrlVals

    set ctrlVals(currSelProdIndex) $index
}

#############################################################################
#
# currSelProdIndexGet - obtain the current selected product's index
#
# This procedure will obtain the current selected product's index
#
# SYNOPSIS
# .tS
# currSelProdIndexGet <info>
# .tE
#
# PARAMETERS:
# .IP info
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc currSelProdIndexGet {{info obj}} {
    global objGuiMap ctrlVals

    if {"$info" != "obj"} {
        set retVal $ctrlVals(currSelProdIndex)
    } else {
        set retVal $objGuiMap(gui2prod,$ctrlVals(currSelProdIndex))
    }

    return $retVal
}

#############################################################################
#
# onClickPartCheckList -  is invoked when a part is selected from the
#                         partListDlg dialog box and it calls helper
#                         procedures to update the GUI and the object
#
# This procedure is a callback when a part is selected from the partListDlg
# dialog box and it calls helper procedures to update the GUI and the object
#
#
# SYNOPSIS
# .tS
# onClickPartCheckList
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onClickPartCheckList {} {

    partObjUpdate
    partGuiSizeUpdate
}
#############################################################################
#
# partObjUpdate - update the Tornado part object
#
# This procedure will update the Tornado part object
#
# SYNOPSIS
# .tS
# partObjUpdate
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partObjUpdate {} {
    global ctrlVals objGuiMap

    set i 0
    foreach itemState \
                [controlItemPropertyGet partListDlgHnd.optionList -checkstate] {

        partInfoSet instFlag $objGuiMap(gui2part,$i) $itemState
        incr i
    }
}

#############################################################################
#
# partGuiUpdate - update the partListDlg GUI
#
# This procedure will update the partListDlg GUI
#
# SYNOPSIS
# .tS
# partGuiUpdate <guiPartIndex>
# .tE
#
# PARAMETERS:
# .IP guiPartIndex
# the index of selected part
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partGuiUpdate {guiPartIndex} {
    partGuiChkLstUpdate $guiPartIndex
    partGuiSizeUpdate
}

#############################################################################
#
# partGuiChkLstUpdate - update the partListDlg GUI
#
# This procedure will update the partListDlg GUI
#
# SYNOPSIS
# .tS
# partGuiChkLstUpdate <guiPartIndex>
# .tE
#
# PARAMETERS:
# .IP guiPartIndex
# the index of selected part
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partGuiChkLstUpdate {guiPartIndex} {
    global objGuiMap
    global ctrlVals

    set prodIndex [currSelProdIndexGet]

    if {"$prodIndex" == ""} {
        windowClose partListDlgHnd
        messageBox "Please select a product first"
    } else {
        set chkList {}
        set i 0

        foreach partIndex [productInfoGet partIndexList $prodIndex] {
            set objGuiMap(gui2part,$i) $partIndex
            set objGuiMap(part2gui,$partIndex) $i
            set size [partInfoGet size $partIndex]
            set formatMbSize [format "%7s MB" [byteToMbyte $size]]
            set desc " [partInfoGet desc $partIndex]\t$formatMbSize "
            lappend chkList [list $desc [partInfoGet instFlag $partIndex]]
            incr i
        }

        controlValuesSet partListDlgHnd.optionList $chkList
        controlFocusSet partListDlgHnd.optionList
        controlSelectionSet partListDlgHnd.optionList -noevent $guiPartIndex
    }
}

#############################################################################
#
# partGuiSizeUpdate - update the size of each Tornado part
#
# This procedure will update the size of each Tornado part
#
# SYNOPSIS
# .tS
# partGuiSizeUpdate
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc partGuiSizeUpdate {} {
    set totalSize [byteToMbyte [productInfoGet size [currSelProdIndexGet]]]

    controlValuesSet partListDlgHnd.spaceRequireLabel \
           "Space Required:  $totalSize MB"
}

#############################################################################
#
# onPartListDlgOk - callback when OK button is pushed and close partListDlg
#                   dialog box and save changes
#
# This procedure is a callback whne OK button is pushed and
# close partListDlg dialog box and save changes
#
# SYNOPSIS
# .tS
# onPartListDlgOk
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onPartListDlgOk {} {
    global objGuiMap

    set prodIndex [currSelProdIndexGet]
    windowClose partListDlgHnd

    switch [productInfoGet stateInfo $prodIndex] {
        changedIncr {
            productInfoSet instFlag \
                    [partInfoGet parent $objGuiMap(gui2part,0)] 1
            set desc [productInfoGet desc $prodIndex]
            set hostType [windHostTypeGet]
            set prodHostType [prodHostTypeGet $desc]
            if {[productInfoGet instFlag $prodIndex]} {
                if {$hostType != $prodHostType && $prodHostType != "unknown"} {
                    set msg "The product, $desc, is not meant for installation \
                        on this host type of $hostType.\nContinuing with the \
                        installation may lead to an incomplete installation \
                        due to missing host tools."
                    dialog ok "SETUP" $msg
                }
            }
        }

        changedToNone {
            productInfoSet instFlag \
                    [partInfoGet parent $objGuiMap(gui2part,0)] 0
        }
    }

    productGuiChkLstUpdate [currSelProdIndexGet gui]
    productInfoSet stateCommit [currSelProdIndexGet]
    productInfoSet childStateCommit [currSelProdIndexGet]
    productGuiSizeUpdate
}

#############################################################################
#
# onPartListDlgCancel - callback when cancel button is pushed and restore
#                       the previous settings
#
# This procedure
#
# SYNOPSIS
# .tS
# onPartListDlgCancel
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onPartListDlgCancel {} {
    windowClose partListDlgHnd
    productInfoSet childStateRestore [currSelProdIndexGet]
}

#############################################################################
#
# detailShow - enable detail button if the Tornado product has subparts
#
# This procedure will enable detail button if the Tornado product has subparts
#
# SYNOPSIS
# .tS
# detailShow
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc detailShow {} {
    global productObj
    global ctrlVals

    set hasSubPart [expr [llength [productInfoGet partIndexList \
                                                  [currSelProdIndexGet]]] > 1]
    set numSubPart \
        [llength [productInfoGet partIndexList [currSelProdIndexGet]]]
    if {$hasSubPart} {

        # add in indication of components/product selected

        set nCompSel 0
        set totComps 0
        foreach partIndex $productObj([currSelProdIndexGet],partIndexList) {
            if {"[partInfoGet instFlag $partIndex]" == "1"} {
                incr nCompSel
            }
            incr totComps
        }

        if {[productInfoGet instFlag [currSelProdIndexGet]] == 1} {
            controlValuesSet $ctrlVals(mainWindow).compSelected "You have \
               selected $nCompSel out of $totComps components for this product."
        } else {
            controlValuesSet $ctrlVals(mainWindow).compSelected "You have \
               selected 0 out of $totComps components for this product."
        }
        controlEnable $ctrlVals(mainWindow).partDetailButt 1
    } else { 
 
       # This is a single part product

        if {[productInfoGet instFlag [currSelProdIndexGet]] == 1} {
            controlValuesSet $ctrlVals(mainWindow).compSelected "You have \
               selected 1 out of 1 components for this product."
        } else {
            controlValuesSet $ctrlVals(mainWindow).compSelected "You have \
               selected 0 out of 1 components for this product."
        }
        controlEnable $ctrlVals(mainWindow).partDetailButt 0
    }
    return $hasSubPart
}

#############################################################################
#
# drvIndexGet - return the driverObjects/Header product index if it exists
#               on the CD.
#
# This procedure will returns the driverObjects/Header product index
# if it exists on the CD.
#
# SYNOPSIS
# .tS
# drvIndexGet
# .tE
#
# PARAMETERS:
# .IP currentProdIndex
# the current product index being selected or deselected
#
# RETURNS: Returns the index of the driverObjects product if the index of the
#          driverObjec is same as that of the current selected product
#          else returns -1
#
# ERRORS: N/A
#
proc drvIndexGet {currProdIndex} {
    global setupVals

    if {$setupVals(drvIndex) == -1} {
        return -1
    }

    foreach drvObj $setupVals(drvIndex) {
        set index [lindex $drvObj 0]
        if {$index == $currProdIndex} {
            return $index
        }
    }
    return -1
}

#############################################################################
#
# drvArchGet - return the driverObjects/Header product arch if it exists
#               on the CD.
#
# This procedure will return the driverObjects/Header product arch if it exists
# on the CD.
#
# SYNOPSIS
# .tS
# drvArchGet
# .tE
#
# PARAMETERS:
# .IP currentProdIndex
# the current product index being selected or deselected
#
# RETURNS: Returns the arch of the driverObjects product if the index of the
#          driverObjec is same as that of the current selected product
#          else returns -1
#
# ERRORS: N/A
#

proc drvArchGet {currProdIndex} {
    global setupVals

    if {$setupVals(drvIndex) == -1} {
        return -1
    }

    foreach drvObj $setupVals(drvIndex) {
        set index [lindex $drvObj 0]
        if {$index == $currProdIndex} {
            return [lindex $drvObj 1]
        }
    }
    return -1
}

#############################################################################
#
# prodHostTypeGet - proc to return a host type string
#
# SYNOPSIS
# .tS
# prodHostTypeGet
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: the hos type (a string) if it exists, else "unknown"
#
# ERRORS: N/A
#
proc prodHostTypeGet {desc} {

    set descLen [string length $desc]
    set index [string first ":" $desc]
    set host [string range $desc [expr $index + 1] $descLen]
    set index [string first " x " $host]
    set host [string trim [string range $host 0 [expr $index - 1]]]

    switch -glob $host {
        solaris {set prodHostType sun4-solaris2}
        x86-win?? {set prodHostType x86-win32}
        Windows?? {set prodHostType x86-win32}
        hp9700 {set prodHostType parisc-hpux10}
        default {set prodHostType unknown}
    }

     # look for host specfic string in the description
 
     if {"$prodHostType" == "unknown" } {
         switch -glob $desc {
            *solaris*    {set prodHostType sun4-solaris2}
            *x86-win*    {set prodHostType x86-win32}
            *hp9700*     {set prodHostType parisc-hpux10}
            *simsolaris* {set prodHostType sun4-solaris2}
            *simnt*      {set prodHostType x86-win32}
            *simhppa*    {set prodHostType parisc-hpux10}
            *SIMHPPA*    {set prodHostType parisc-hpux10}
         }
     }

    return $prodHostType
}

#############################################################################
#
# onClickProdCheckList - callback procedure invoked when product is checked
#                        on compSelect page
#
# This procedure is a callback procedure invoked when product is checked
# on compSelect page
#
# SYNOPSIS
# .tS
# onClickProdCheckList
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc onClickProdCheckList {} {
    global ctrlVals
    global setupVals

    currSelProdIndexSet [controlSelectionGet $ctrlVals(mainWindow).optionList]
    set hasSubPart [detailShow]
    set currProdIndex [currSelProdIndexGet]

    if {[lindex [controlEventGet $ctrlVals(mainWindow).optionList] 0] == \
        "chkchange"} {
        set newState [lindex \
                [controlItemPropertyGet $ctrlVals(mainWindow).optionList \
                -checkstate] [currSelProdIndexGet gui]]

        # determine if customer is trying to uncheck a driverObject/Header prod

        set drvIndex [drvIndexGet $currProdIndex]
        set drvArch [drvArchGet $currProdIndex]
        set drvObjRestored 0

        if {[info exists setupVals(tornadoIndex)] && $drvIndex != -1} {
            foreach torObj $setupVals(tornadoIndex) {
                set tornadoSelected  [isTornadoSelected [lindex $torObj 0]]
                if {$tornadoSelected && [lindex $torObj 1] == $drvArch} {
                    set savedProdIndex -1
                    messageBox [strTableGet 5010_DRIVERS_INFO] -informationicon
                    set index [guiMapCtrlIndexGet $drvIndex]
                    controlItemPropertySet $ctrlVals(mainWindow).optionList \
                        -index $index -checkstate 1
                    set drvObjRestored 1
                    break
                }
            }
        }

        # warn user about non-native installations

        set desc [productInfoGet desc $currProdIndex]
        set hostType [windHostTypeGet]
        set prodHostType [prodHostTypeGet $desc]
        if {![productInfoGet instFlag $currProdIndex]} {
            if {$hostType != $prodHostType && $prodHostType != "unknown"} {
                set msg "The product, $desc, is not meant for installation on \
                        this host type of $hostType.\nContinuing with the \
                        installation may lead to an incomplete installation \
                        due to missing host tools."
                dialog ok "SETUP" $msg
            }
        }

        # commit the user's selection

        if {!$drvObjRestored} {
            set savedProdIndex $currProdIndex
            productInfoSet instFlag $currProdIndex $newState
            if {$newState && $hasSubPart} {
                productGuiChkLstUpdate [currSelProdIndexGet gui]
            }

            productGuiSizeUpdate $newState
            productInfoSet stateCommit $currProdIndex
            productInfoSet childStateCommit $currProdIndex
        }

        # check if product just selected is tornado-obj, if so we must
        # re-select the driver object products

        set drvObjRestored 0
        if {[info exists setupVals(tornadoIndex)]} {
            foreach torObj $setupVals(tornadoIndex) {
                set tornadoSelected [isTornadoSelected [lindex $torObj 0]]
                set torIndex [lindex $torObj 0]
                set torArch [lindex $torObj 1]
                if {$setupVals(drvIndex) != -1 && $tornadoSelected && \
                    [expr $currProdIndex == $torIndex || \
                     $savedProdIndex == $torIndex] } {
                   foreach  drvObj $setupVals(drvIndex) {
                        set index [lindex $drvObj 0]
                        set arch [lindex $drvObj 1]
                        set drvObjSelected [productInfoGet instFlag $index]
                        if {!$drvObjSelected && $torArch == $arch} {
                            messageBox [strTableGet 5010_DRIVERS_INFO] \
                                -informationicon
                            productInfoSet instFlag $index 1
                            productGuiSizeUpdate 1
                            productInfoSet stateCommit $index
                            productInfoSet childStateCommit $index
                            set guiIndex [guiMapCtrlIndexGet $index]
                            controlItemPropertySet \
                                $ctrlVals(mainWindow).optionList \
                                -index $guiIndex -checkstate 1
                            set drvObjRestored 1
                            break
                        }
                    }
                }
                if {$drvObjRestored} {
                    break
                }
            }
        }
    }
}

#############################################################################
#
# guiMapCtrlIndexGet - return the GUI index of the selected product index
#
# This procedure will return the GUI index of the selected product index
#
# SYNOPSIS
# .tS
# guiMapCtrlIndexGet <prodIndex>
# .tE
#
# PARAMETERS:
# .IP prodIndex
# product index
#
# RETURNS: GUI mapped index of the product index
#
# ERRORS: N/A
#

proc guiMapCtrlIndexGet {prodIndex} {
    global objGuiMap ctrlVals

    set retVal $objGuiMap(prod2gui,$prodIndex)
    return $retVal
}

#############################################################################
#
# productGuiChkLstCreate - create the check list of products in compSelect page
#
# This procedure will create the check list of products in compSelect page
#
# SYNOPSIS
# .tS
# productGuiChkLstCreate
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc productGuiChkLstCreate {} {
    global objGuiMap
    global ctrlVals
    global setupVals
    global productObj

    # populate the list of check button.

    set i 0
    set chkList ""
    set totalSize 0

    catch {
        unset objGuiMap
    }

    foreach prodIndex [cdInfoGet productIndexList] {
        # test automation

        if { $ctrlVals(useInputScript) } {
            set desc [productInfoGet desc $prodIndex]
            if {[lsearch $setupVals(compList) "$desc"] < 0} {
                productInfoSet instFlag $prodIndex 0
            }
            foreach part [productInfoGet partIndexList $prodIndex] {
                set descp "[partInfoGet desc $part]"
                if [regexp WTX $descp] {
                    partInfoSet instFlag $part 1
                }
            }
        }
        set objGuiMap(gui2prod,$i) $prodIndex
        set objGuiMap(prod2gui,$prodIndex) $i
        set size [productInfoGet size $prodIndex]
        set formatMbSize [format "%7s MB" [byteToMbyte $size]]

        set desc " [productInfoGet desc $prodIndex]\t$formatMbSize "

        lappend chkList [list $desc [productInfoGet instFlag $prodIndex]]
        incr i
    }

    controlValuesSet $ctrlVals(mainWindow).optionList $chkList
    controlSelectionSet  $ctrlVals(mainWindow).optionList -noevent 0
    currSelProdIndexSet 0

    # use the tri-state checkbox 

    for {set ix 0} {$ix < $i} {incr ix} {
        set prodIndex $objGuiMap(gui2prod,$ix)
        set nCompSel 0
        set totComps 0
        foreach partIndex $productObj($prodIndex,partIndexList) {
            if {"[partInfoGet instFlag $partIndex]" == "1"} {
                incr nCompSel
            }
            incr totComps
        }

        if {[productInfoGet instFlag $prodIndex]} {
            if {$totComps != $nCompSel} {
                controlItemPropertySet $ctrlVals(mainWindow).optionList \
                    -index $objGuiMap(prod2gui,$prodIndex) -checkstate 2
            }
        }
    }

    # uncheck the products not meant for this host type

    set hostType [windHostTypeGet]
    for {set ix 0} {$ix < $i} {incr ix} {
        set prodIndex $objGuiMap(gui2prod,$ix)
        set desc [productInfoGet desc $prodIndex]
        set prodHostType [prodHostTypeGet $desc]
        if {$hostType != $prodHostType && $prodHostType != "unknown"} {
            productInfoSet instFlag $prodIndex 0
            controlItemPropertySet $ctrlVals(mainWindow).optionList \
                -index $ix -checkstate 0
        }
    }

    # test automation

    if { $ctrlVals(useInputScript) } {
        autoSetupLog "\tInstalled product(s):"

        # retrieve product name from the product list

        foreach i $chkList {
            autoSetupLog "\t  $i"
        }
    }

}

#############################################################################
#
# productGuiChkLstUpdate - update the check list of products in the compSelect
#                          page
#
# This procedure will update the check list of products in the compSelect page
#
# SYNOPSIS
# .tS
# productGuiChkLstUpdate <index>
# .tE
#
# PARAMETERS:
# .IP index
# product index
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc productGuiChkLstUpdate {index} {
    global objGuiMap
    global ctrlVals
    global productObj

    set prodIndex [currSelProdIndexGet]
    set size [productInfoGet size $prodIndex]
    set formatMbSize [format "%7s MB" [byteToMbyte $size]]

    # add in indication of components/product selected
    # remove this section if a UITcl tristate button list is created

    set nCompSel 0
    set totComps 0
    foreach partIndex $productObj($prodIndex,partIndexList) {
        if {"[partInfoGet instFlag $partIndex]" == "1"} {
            incr nCompSel
        }
        incr totComps
    }

    set desc " [productInfoGet desc $prodIndex]\t$formatMbSize "

    set chkButt [list [list "$desc" [productInfoGet instFlag $prodIndex]]]

    controlValuesSet $ctrlVals(mainWindow).optionList -insert $index $chkButt
    controlValuesSet $ctrlVals(mainWindow).optionList -delete [expr $index + 1]
    controlSelectionSet  $ctrlVals(mainWindow).optionList -noevent $index
    if {$totComps != $nCompSel} {
        if {[productInfoGet instFlag $prodIndex]} {
            set index $objGuiMap(prod2gui,$prodIndex)
            controlItemPropertySet $ctrlVals(mainWindow).optionList \
                        -index $index -checkstate 2
        }
    }        
}

#############################################################################
#
# productguisizeupdate - update the size of products in the compSelect page
#
# This procedure will update the size of products in the compSelect page
#
# SYNOPSIS
# .tS
# productGuiSizeUpdate [state]
# .tE
#
# PARAMETERS:
# .IP state
# init -- first time the sizes of products are updated
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc productGuiSizeUpdate {{state "init"}} {
    global ctrlVals
    global setupVals

    uiEventProcess


    set setupVals(availDiskSpace) "Unknown"
    if {[isUnix]} {
        set free [fspace [destDirGet]]
        if [regexp {^[0-9]+$} $free] {
            set setupVals(availDiskSpace) \
                [format "%3.1f" [expr $free / 1024.0]]
        }
    } else {
        if {![catch {setupDiskSpaceGet [destDirGet]} diskSpace] &&
             $diskSpace >= 0} {
             set setupVals(availDiskSpace) [kbyteToMbyte $diskSpace]
        }
    }

    set setupVals(reqDiskSpace) [cdInfoGet size]

    # add size of SETUP directory to total required space only if SETUP 
    # directory is not present in destination directory

    if [file exists [file join [destDirGet] SETUP RESOURCE TCL INSTW32.TCL]] {
        set totalSize [byteToMbyte $setupVals(reqDiskSpace)]
    } else {
        set totalSize [expr [setupSizeGet] + \
                            [byteToMbyte $setupVals(reqDiskSpace)]]
    }

    detailShow

    controlValuesSet $ctrlVals(mainWindow).reqSpace \
                     [format "%7s MB" $totalSize]
    controlValuesSet $ctrlVals(mainWindow).availSpace \
                     [format "%7s MB" $setupVals(availDiskSpace)]
}

#############################################################################
#
# checkProductDependency - check to see if dependencies are OK
#
# This is a callback procedure for the partList Dialog.  It will check to
# see if any dependencies are violated.  If so warn the user and correct
# them.  The only one is the driver/objects dependency on Tornado parts.
#
# SYNOPSIS
# .tS
# checkProductDependency
# .tE
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc checkProductDependency {} {
    global ctrlVals
    global setupVals

    set prodIndex [currSelProdIndexGet]

    if {"$prodIndex" != ""} {

        # check to see if the current product is the Tornado product
        # determine if customer is trying to uncheck a driverObject/Header prod

        set drvObjRestored 0
        if {[info exists setupVals(tornadoIndex)]} {
            foreach torObj $setupVals(tornadoIndex) {
                set tornadoSelected [isTornadoSelected [lindex $torObj 0]]
                set torIndex [lindex $torObj 0]
                set torArch [lindex $torObj 1]
                if {$setupVals(drvIndex) != -1 && $tornadoSelected && \
                    [expr [currSelProdIndexGet] == $torIndex]} {
                   foreach  drvObj $setupVals(drvIndex) {
                        set index [lindex $drvObj 0]
                        set arch [lindex $drvObj 1]
                        set drvObjSelected [productInfoGet instFlag $index]
                        if {!$drvObjSelected && $torArch == $arch} {
                            messageBox [strTableGet 5010_DRIVERS_INFO] \
                                -informationicon
                            productInfoSet instFlag $index 1
                            productGuiSizeUpdate 1
                            productInfoSet stateCommit $index
                            productInfoSet childStateCommit $index
                            set guiIndex [guiMapCtrlIndexGet $index]
                            controlItemPropertySet \
                                $ctrlVals(mainWindow).optionList \
                                -index $guiIndex -checkstate 1
                            set drvObjRestored 1
                            break
                        }
                    }
                }
                if {$drvObjRestored} {
                    break
                }
            }
        }
    }
}

#############################################################################
#
# isTornadoSelected - checks to see if the tornado product is selected
#
# This procedure checks to see if the tornado product is selected by the
# customer.  If the parts "Tornado Integrated Simulator" and|or
# "Tornado Object" are selected, then 1 is returned, else 0 is returned.
#
# SYNOPSIS
# .tS
# isTornadoSelected <prodIndex>
# .tE
#
# PARAMETERS: prodIndex - the product index of the Tornado product
#
# RETURNS: 1 if either the simulator or tornado objects are selected
#          0 otherwise
#
# ERRORS: N/A
#

proc isTornadoSelected {prodIndex} {

    set retVal 0

    # first check to see if the tornado product has been selected

    if {[productInfoGet instFlag $prodIndex]} {
        foreach part [productInfoGet partIndexList $prodIndex] {
            set desc [partInfoGet desc $part]

            # check if the simulator part is selected

            if {[string first "Tornado Integrated Simulator" $desc] != -1} {
                if {[partInfoGet instFlag $part] == 1} {
                    return 1
                }
            }

            # check if the full simulator is selected

            if {[string first "Tornado Simulator:" $desc] != -1} {
                if {[partInfoGet instFlag $part] == 1} {
                    return 1
                }
            }

            # check if the Tornado part is selected

            if {[string first "Tornado Object" $desc] != -1} {
                if {[partInfoGet instFlag $part] == 1} {
                    return 1
                }
            }
        }
    }
    return $retVal
}

#############################################################################
#
# kbyteToMbyte - converts kilo bytes to mega bytes
#
# SYNOPSIS
# kbyteToMbyte <size>
#
# PARAMETERS:
#    <kbyte> : number in kilo bytes
#
# RETURNS: number in mega bytes
#
# ERRORS: N/A
#

proc kbyteToMbyte {size} {
    return [format "%3.1f" [expr $size / 1024.0]]
}

##############################################################################
#
# byteToMbyte - converts bytes to Mbytes
#
# SYNOPSIS
# byteToMbyte <size>
#
# PARAMETERS:
#    <byte> : number in bytes
#
# RETURNS: number in mega bytes
#
# ERRORS: N/A
#

proc byteToMbyte {size} {
    return [format "%3.1f" [expr $size / 1048576.0]]
}


