(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* stolen from RWH's 15-312 *)

signature SOCKET_STREAMS =
sig

  val create : ('a,Socket.active Socket.stream) Socket.sock
               * unit SyncVar.ivar
               -> string CML.chan * string CML.chan

  val acceptLoop : ('a,Socket.passive Socket.stream) Socket.sock
                   * unit SyncVar.ivar
                   -> (('a,Socket.active Socket.stream) Socket.sock
                       * 'a Socket.sock_addr) CML.chan

  val inet_port_socket : int -> (INetSock.inet,Socket.passive Socket.stream) Socket.sock

  val unix_socket : string -> (UnixSock.unix, Socket.passive Socket.stream) Socket.sock

end

structure SocketStreams :> SOCKET_STREAMS =
struct

  open CML

  fun addNewCleaner(n,f) =
      case RunCML.addCleaner(n,[RunCML.AtExit,RunCML.AtShutdown],f)
        of SOME(w,f') => (RunCML.addCleaner(n,w,f');
                          addNewCleaner(n ^ "'",f))
         | NONE => ()

  fun inet_port_socket(port) =
      let val socket = INetSock.TCP.socket()
          val addr = INetSock.any port;
       in Socket.bind(socket, addr);
	  Socket.Ctl.setREUSEADDR(socket, true);
          (* addNewCleaner("CleanListenSocket",fn _ => Socket.close(socket));*)
          Socket.listen(socket, 20);
          socket
      end

  fun unix_socket address =
    let 
      val socket = UnixSock.Strm.socket ()
      val addr = UnixSock.toAddr address
      val _ = ((Posix.FileSys.unlink address) handle _ => ())
    in
      Socket.bind(socket, addr);
      Socket.Ctl.setREUSEADDR(socket, true);
      Socket.listen(socket, 20);
      socket
    end

  fun acceptLoop(socket,pleaseDie) =
      let val ch = channel()

          val acceptE = wrap(OS.IO.pollEvt
                             [OS.IO.pollIn(Socket.pollDesc socket)],
                             fn _ => Socket.accept(socket))
	  val dieE = SyncVar.iGetEvt pleaseDie
          fun acc() =
	    (CML.send (ch, select[acceptE, wrap(dieE, cleanup)]);
	     acc())

          and cleanup() = (TextIO.print "accept loop stopped...\n";
			   Socket.close(socket);
			   CML.exit ())

       in TextIO.print "accept loop started...\n";
          addNewCleaner("CleanAcceptLoop",
                        fn _ => SyncVar.iPut(pleaseDie,()) handle _ => ());
          spawn(acc);
          ch
      end

  fun create(socket,pleaseDie) =
    let 
      val inch = channel()
      val outch = channel()
	
      fun cleanup signal () = 
	(TextIO.print "remote connection closed...\n";
	 Socket.close(socket);
	 if not signal then SyncVar.iPut (pleaseDie, ())
	 else ();
	   CML.exit ())

      val inEvts = [Socket.recvVecEvt(socket,1024),
		    wrap(SyncVar.iGetEvt pleaseDie, cleanup true)]
	
      fun readLoop() =
	let
	  (* val _ = TextIO.print "Here i am\n" *)
	  val v = select inEvts
	(* val _ = TextIO.print "read something\n" *)
	in if Word8Vector.length v = 0 then cleanup false ()
	   else (send(inch,Byte.bytesToString(v));
		 readLoop())
	end
      
      fun doout(v,i) =
	(if i <> 0 andalso i <> Word8Vector.length(v) then 
	   TextIO.print ("out: " ^ (Int.toString i) ^ "\n")
	 else ();
	   if i < Word8Vector.length(v)
	     then doout(v,i+Socket.sendVec(socket,{buf=v,i=i,sz=NONE}))
	   else ())
	   
      val outEvts = [CML.recvEvt outch,
		     wrap(SyncVar.iGetEvt pleaseDie, CML.exit)]
	
      (* handle exceptions *)
      fun writeLoop() =
	let
	  val v = CML.select outEvts
	in
	  (doout(Byte.stringToBytes v, 0);
	   writeLoop())
	end
      
    in TextIO.print "remote connection opened...\n";
      addNewCleaner("CleanSocketReader",
		    fn _ => SyncVar.iPut(pleaseDie,()) handle _ => ());
      spawn(readLoop);
      spawn(writeLoop);
      (inch,outch)
    end
  
  fun writev' (sock, nil, written) = written
    | writev' (sock, a::l, written) =
    let
      val x = Socket.sendVec (sock, {buf=a, i=0, sz=NONE})
    in
      if (x = Word8Vector.length a)
	then writev' (sock, l, written + x)
      else written
    end

  (* attempt to load the C version of writev, otherwise default to the
   above compatibility layer *)
  val writev =
    (Unsafe.CInterface.c_function "CMU-LIB" "writev" )
(*           : ('a, Socket.active Socket.stream) Socket.sock * Word8Vector.vector list 
             -> int) *)
    handle (Unsafe.CInterface.CFunNotFound _) =>
      fn (sock, l) => writev' (sock, l, 0)

end
