(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Config =
struct
  exception NotImplemented
  exception Config of string
  exception BadDSetsFile
  exception Impossible of string

  val debug = false

  val FNAME_config = "/etc/acapd.conf"

  val FNAME_dsets = ref "/var/acap/dsets"

  val FNAME_socket = ref "/var/acap/socket"

  val FDIR_spool = ref "/var/spool/acap"

  fun makeDataFile (dset) = (!FDIR_spool) ^ dset ^ "#data"

  val SweepTime = ref (Time.fromSeconds 180)

  (* we should process /etc/acapd.conf *)
  fun init () =
    let
      fun notEOL #"\n" = false | notEOL _ = true
      val scanComment = ParserComb.seqWith (fn _ => ())
	                               (ParserComb.char #"#",
					ParserComb.eatChar notEOL)
      val good = ParserComb.result ()
      val bad = ParserComb.failure
				       
      fun toggle ("configdirectory", d) =
	(FNAME_dsets := d ^ "/dsets";
	 FNAME_socket := d ^ "/socket";
	 good)
	| toggle ("datadirectory", d) =
	(FDIR_spool := d;
	 good)
	| toggle ("sweeptime", v) =
	((SweepTime := (Time.fromSeconds (valOf (Int32.fromString v)));
	  good) handle _ => bad)
	| toggle (_, _) = bad

      val scan = ParserComb.seq (ParserComb.skipBefore Char.isSpace
				 (ParserComb.token (fn c => c <> #":")),
		 ParserComb.seqWith #2 (ParserComb.char #":",
                 ParserComb.seqWith #1 (ParserComb.skipBefore Char.isSpace
				          (ParserComb.token notEOL),
					ParserComb.char #"\n")))
      val scan = ParserComb.bind (scan, toggle)

      val scanFile = ParserComb.zeroOrMore (ParserComb.or (scanComment, scan))

      val f = (TextIO.openIn FNAME_config
	       handle _ => 
		 raise Config (FNAME_config ^ " doesn't exist"))
    in
      (case TextIO.scanStream scanFile f of
	 NONE => raise Config "malformed configuration file"
       | (SOME _) => if TextIO.endOfStream f then ()
		     else raise Config "malformed configuration file");
      TextIO.closeIn f
    end
  
end
