(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure RMulticast :> RMULTICAST =
struct
  (* uses referencing counting; another implementation would be to use weak
   pointers *)
  structure MC = Multicast

  exception NullPort

  type 'a event = 'a CML.event 
  type 'a mchan = 'a MC.mchan * int SyncVar.mvar
  datatype 'a port = 
    Real of 'a MC.port * int SyncVar.mvar
  | Null

  (* val mChannel : unit -> 'a mchan *)
  fun mChannel () = (MC.mChannel (), SyncVar.mVarInit 0)

  fun inc count =
    let
      val c = SyncVar.mTake count
      val _ = SyncVar.mPut (count, c + 1)
    in
      count
    end

  fun dec count =
    let
      val c = SyncVar.mTake count
      val _ = SyncVar.mPut (count, c - 1)
    in
      count
    end

  val nullport = Null
  
  (* val port : 'a mchan -> 'a port *)
  fun port (chan, count) = Real (MC.port chan, inc count)

  (* val port' : 'a mchan -> 'a port *)
  fun port' (chan, count) = Real (MC.port chan, count)

  (* val copy : 'a port -> 'a port *)
  fun copy (Real (port, count)) = Real (MC.copy port, inc count)
    | copy (Null) = Null

  (* val copy' : 'a port -> 'a port *)
  fun copy' (Real (port, count)) = Real (MC.copy port, count)
    | copy' (Null) = Null

  (* val recv : 'a port -> 'a *)
  fun recv (Real (port, _)) = MC.recv port
    | recv Null = raise NullPort

  (* val recvEvt : 'a port -> 'a event  *)
  fun recvEvt (Real (port, _)) = MC.recvEvt port
    | recvEvt Null = CML.never

  (* val multicast : ('a mchan * 'a) -> unit *)
  fun multicast ((chan, _), x) = MC.multicast (chan, x)

  (* val release : 'a port -> unit *)
  fun release (Real (_, count)) = ignore (dec count)
    | release Null = ()

  (* val listeners : 'a mchan -> int *)
  fun listeners (chan, count) = SyncVar.mGet count

end
