(* ACAP lexer *)
(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* this lexer sucks two ways:
 * (1) case insensitivity
 * (2) munging the internal lexer state
 *)

(* note that we munge the internal lexer state here *)

type pos = int
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult  = (svalue,pos) token

fun eof() = Tokens.EOF(0,0)

(* remove beginning and trailing quotes and quoted chars 
   error checking here? *)
fun fixstr (str) = String.substring(str, 1, (size str) - 2)

%%
%s LINE;
%header (functor AcapLexFun(structure Tokens : Acap_TOKENS));

crlf = \013\010;
digit = [0-9];
string = "\"" ([^"\""\013\010]|"\\\\"|"\\\"")* "\"";
dktag = ([\033\035-\039\044-\091\093-\122\124-\126]){1,20};
tag = (\033 | [\035-\039] | [\044-\091] | [\093-\122] | [\124-\126]){1,20};

%%

<INITIAL>{tag}	=> (YYBEGIN LINE; 
		    Tokens.TAG(yytext, yypos, yypos+size(yytext)));
<INITIAL,LINE>{string} => (Tokens.STRING(fixstr(yytext), yypos, yypos+size(yytext)));
<INITIAL,LINE>{crlf} => (YYBEGIN INITIAL; 
		 Tokens.CRLF(yypos,yypos+1));
<LINE>{digit}+ => (Tokens.NUMBER(valOf (Int.fromString yytext), yypos,
				 yypos+size(yytext)));
<INITIAL,LINE>"{"{digit}+"+}"{crlf} => 
   (let
      val litlen = valOf(Int.fromString (String.substring (yytext, 1,
						        (size yytext - 3))))

      val have = !yybl - !yybufpos

      fun yyread (str, n) = let val s=yyinput n 
				val sz=size s
			    in if (sz>n) then
			      (str ^ (String.substring (s, 0, n)), 
			       (String.extract (s, n, NONE)))
			       else yyread (str^s, n-sz)
			    end

      val str = if (have >= litlen) then
                  (* enough, grab it from buffer *)
	          (String.substring(!yyb, !yybufpos, litlen)
		   before yybufpos := !yybufpos + litlen)
		else (* grab the rest of the buffer and start reading *)
		  let
		    val need = litlen - have
		    val (str, buf) = yyread (String.substring(!yyb, 
							      !yybufpos, have),
					     need)
		  in
		    yyb := buf;
		    yybufpos := 0;
		    yybl := size buf;
		    str
		  end
    in
      Tokens.STRING(str, yypos, yypos+litlen)
    end);
<LINE>" "  => (Tokens.SP(yypos,yypos+1));
<LINE>"("  => (Tokens.LPAREN(yypos,yypos+1));
<LINE>")"  => (Tokens.RPAREN(yypos,yypos+1));
<LINE>[Dd][Ee][Ff][Aa][Uu][Ll][Tt] => (Tokens.DEFAULT(yypos,yypos+7));
<LINE>[Nn][Ii][Ll] => (Tokens.NIL(yypos, yypos+3));
<LINE>[Nn][Oo][Cc][Rr][Ee][Aa][Tt][Ee] => (Tokens.NOCREATE(yypos, yypos+8));
<LINE>[Uu][Nn][Cc][Hh][Aa][Nn][Gg][Ee][Dd][Ss][Ii][Nn][Cc][Ee] => (Tokens.UNCHANGEDSINCE(yypos, yypos+14));
<LINE>[Aa][Ll][Ll] => (Tokens.ALL(yypos, yypos+3));
<LINE>[Nn][Oo][Tt] => (Tokens.NOT(yypos, yypos+3));
<LINE>[Oo][Rr] => (Tokens.OR(yypos, yypos+2));
<LINE>[Aa][Nn][Dd] => (Tokens.AND(yypos, yypos+3));
<LINE>[Dd][Ee][Pp][Tt][Hh] => (Tokens.DEPTH(yypos, yypos+5));
<LINE>[Hh][Aa][Rr][Dd][Ll][Ii][Mm][Ii][Tt] => (Tokens.HARDLIMIT(yypos, yypos+9));
<LINE>[Ll][Ii][Mm][Ii][Tt] => (Tokens.LIMIT(yypos, yypos+5));
<LINE>[Mm][Aa][Kk][Ee][Cc][Oo][Nn][Tt][Ee][Xx][Tt] => (Tokens.MAKECONTEXT(yypos, yypos+11));
<LINE>[Nn][Oo][Tt][Ii][Ff][Yy] => (Tokens.NOTIFY(yypos, yypos+6));
<LINE>[Ee][Nn][Uu][Mm][Ee][Rr][Aa][Tt][Ee] => (Tokens.ENUMERATE(yypos, yypos+9));
<LINE>[Nn][Oo][Ii][Nn][Hh][Ee][Rr][Ii][Tt] => (Tokens.NOINHERIT(yypos, yypos+9));
<LINE>[Rr][Ee][Tt][Uu][Rr][Nn] => (Tokens.RETURN(yypos, yypos+6));
<LINE>[Ss][Oo][Rr][Tt] => (Tokens.SORT(yypos, yypos+4));
<LINE>[Ee][Qq][Uu][Aa][Ll] => (Tokens.ISEQUAL(yypos, yypos+5));
<LINE>[Cc][Oo][Mm][Pp][Aa][Rr][Ee] => (Tokens.COMPARE(yypos, yypos+7));
<LINE>[Pp][Rr][Ee][Ff][Ii][Xx] => (Tokens.PREFIX(yypos, yypos+6));
<LINE>[Rr][Aa][Nn][Gg][Ee] => (Tokens.RANGE(yypos, yypos+5));
<LINE>[Cc][Oo][Mm][Pp][Aa][Rr][Ee][Ss][Tt][Rr][Ii][Cc][Tt] => (Tokens.COMPARESTRICT(yypos, yypos+13));
<LINE>[Ss][Uu][Bb][Ss][Tt][Rr][Ii][Nn][Gg] => (Tokens.SUBSTRING(yypos, yypos+9));
<LINE>[Ll][Ii][Ss][Tt][Rr][Ii][Gg][Hh][Tt][Ss] => (Tokens.LISTRIGHTS(yypos, yypos+10));
<LINE>[Mm][Yy][Rr][Ii][Gg][Hh][Tt][Ss] => (Tokens.MYRIGHTS(yypos, yypos+8));
<LINE>[Ss][Ee][Aa][Rr][Cc][Hh] => (Tokens.SEARCH(yypos, yypos+6));
<LINE>[Ss][Ee][Tt][Aa][Cc][Ll] => (Tokens.SETACL(yypos, yypos+6));
<LINE>[Ss][Tt][Oo][Rr][Ee] => (Tokens.STORE(yypos, yypos+5));
<LINE>[Dd][Ee][Ll][Ee][Tt][Ee][Aa][Cc][Ll] => (Tokens.DELETEACL(yypos, yypos+9));
<LINE>[Uu][Pp][Dd][Aa][Tt][Ee][Cc][Oo][Nn][Tt][Ee][Xx][Tt] => (Tokens.UPDATECONTEXT(yypos, yypos+13));
<LINE>[Ff][Rr][Ee][Ee][Cc][Oo][Nn][Tt][Ee][Xx][Tt] => (Tokens.FREECONTEXT(yypos, yypos+11));
<LINE>[Gg][Ee][Tt][Qq][Uu][Oo][Tt][Aa] => (Tokens.GETQUOTA(yypos, yypos+8));
<LINE>[Ll][Aa][Nn][Gg] => (Tokens.LANG(yypos, yypos+4));
<LINE>[Aa][Uu][Tt][Hh][Ee][Nn][Tt][Ii][Cc][Aa][Tt][Ee] => (Tokens.AUTHENTICATE(yypos, yypos+12));
<LINE>[Nn][Oo][Oo][Pp] => (Tokens.NOOP(yypos, yypos+4));
<LINE>[Ll][Oo][Gg][Oo][Uu][Tt] => (Tokens.LOGOUT(yypos, yypos+6));
.	=> (Tokens.LEXERROR(yypos, yypos+1));
