/* actest.c--acap tester program
 * Rob Earhart
 */
/* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* This essentially exists to do an ACAP authenticate, and then
 * given command-line control to the user. */

/* This is a massive, MASSIVE hack. */

#define ACTEST_C
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdlib.h>
#include <termios.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <sasl.h>

const char     *progname = NULL;

#define ACAP_DEFAULT_PORT 674
#define BUF_SIZE 32768

static int
password_proc(sasl_conn_t * conn,
	      void *context __attribute__((unused)),
	      int id,
	      sasl_secret_t ** psecret)
{
  int             result = SASL_OK;

  struct termios  orig,
                  noecho;

  char            buf[BUF_SIZE];

  size_t          len;

  int             got_newline;

  if (!conn || !psecret || id != SASL_CB_PASS)
    return SASL_BADPARAM;

  if (tcgetattr(fileno(stdin), &orig))
    return SASL_FAIL;
  noecho = orig;
  noecho.c_lflag &= ~ECHO;
  noecho.c_lflag |= ECHONL;
  if (tcsetattr(fileno(stdin), TCSAFLUSH, &noecho))
    return SASL_FAIL;
  fputs("Password: ", stdout);
  fflush(stdout);

  fgets(buf, BUF_SIZE, stdin);

  len = strlen(buf);
  if (0 < len && buf[len - 1] == '\n') {
    --len;
    buf[len] = '\0';
    if (0 < len && buf[len - 1] == '\r') {
      --len;
      buf[len] = '\0';
    }
    got_newline = 1;
  } else {
    got_newline = 0;
  }
  *psecret = (sasl_secret_t *) malloc(sizeof(sasl_secret_t) + len);
  if (!*psecret) {
    result = SASL_NOMEM;
    goto cleanup_tty;
  }
  (*psecret)->len = len;
  memcpy((*psecret)->data, buf, (*psecret)->len);

cleanup_tty:
  while (!got_newline) {
    fgets(buf, BUF_SIZE, stdin);
    if (buf[0] && buf[strlen(buf) - 1] == '\n')
      got_newline = 1;
  }

  memset(buf, 0, sizeof(buf));

  tcsetattr(fileno(stdin), TCSANOW, &orig);

  return result;
}

static int
sasl_get_user(void *context, int id,
	      const char **result, unsigned *len)
{
    char *user;

    if (id == SASL_CB_USER || id == SASL_CB_AUTHNAME) {
	user = getenv("USER");
	*result = (char *) malloc(strlen(user)+1);
	strcpy((char *)(*result),user);
	if (len) { *len = strlen(*result); }
	return SASL_OK;
    }
    return SASL_FAIL;
}

static int
sasl_opt_proc(void *context,
	      const char *plugin_name,
	      const char *option,
	      const char **result,
	      unsigned *result_len)
{
  const char     *opt = *(const char **) context;

  size_t          len;

  size_t          plugin_name_len = plugin_name ? strlen(plugin_name) : 0;

  size_t          option_len = option ? strlen(option) : 0;

  unsigned        is_plugin_specific;

  unsigned        got_plugin_specific = 0;

  *result = NULL;
  *result_len = 0;

  printf("sasl_opt_prog: getting option \"%s\" for plugin \"%s\"\n",
	 option, plugin_name);

  while (opt && *opt) {
    is_plugin_specific = 0;
    len = strcspn(opt, ":,=");
    if (len == 0) {
      opt++;
      continue;
    }
    if (opt[len] == ':') {
      if (len == plugin_name_len
	  && !strncmp(plugin_name, opt, len)) {
	opt += len + 1;
	len = strcspn(opt, ",=");
	is_plugin_specific = 1;
      } else {
	opt = strchr(opt, ',');
	if (opt)
	  opt++;
	continue;
      }
    }
    if (got_plugin_specific && !is_plugin_specific) {
      opt = strchr(opt, ',');
      if (opt)
	opt++;
      continue;
    }
    if (opt[len] == ',')
      continue;
    if (len == option_len
	&& !strncmp(option, opt, len)) {
      opt += len + 1;
      len = strcspn(opt, ",");
      *result = opt;
      *result_len = len;
      if (is_plugin_specific)
	got_plugin_specific = 1;
      opt += len + 1;
    }
  }

  if (!*result) {
    puts("sasl_opt_prog: no such option");
    return SASL_FAIL;
  }
  printf("sasl_opt_prog: returning value \"%*s\"\n",
	 *result_len, *result);

  return SASL_OK;
}

int
main(int argc, char *argv[])
{
    int             c;

    int             errflg = 0;

    int             sasl_result;

    const char     *port = "acap";

    struct sockaddr_in sin;

    int             fd;

    char            buf[BUF_SIZE];

    ssize_t         n_read;

    size_t          len;

    static const char *sasl_opts = NULL;

    struct servent *se;

    struct hostent *he;

    sasl_conn_t    *sasl_conn;

    char           *s,
	*t;

    char *in;
    unsigned inlen;

    char           *want_mech = NULL;

    int             do_newline_trans;

    const char     *mech;

    int             auth_size;
    
    int             prefix_len;

    char serverFQDN[1024];

    sasl_interact_t *interact;

    sasl_callback_t sasl_callbacks[] = {
	{SASL_CB_GETOPT, &sasl_opt_proc, &sasl_opts},
	{SASL_CB_PASS, &password_proc, NULL},
	{SASL_CB_USER, &sasl_get_user, NULL},
	{SASL_CB_GETREALM, &sasl_opt_proc, &sasl_opts},
	{SASL_CB_AUTHNAME, &sasl_get_user, NULL},
	{SASL_CB_LIST_END, NULL, NULL}
    };

    progname = strrchr(argv[0], '/');
    if (progname)
	progname++;
    else
	progname = argv[0];

    while ((c = getopt(argc, argv, "p:s:m:")) != EOF)
	switch (c) {
	case 'p':
	    port = optarg;
	    break;
	case 's':
	    sasl_opts = optarg;
	    break;
	case 'm':
	    want_mech = optarg;
	    break;
	default:
	    errflg = 1;
	}
    if (optind != argc - 1)
	errflg = 1;

    if (errflg) {
	fprintf(stderr, "%s: Usage: %s [-p port] [-s opt=val,plugin:opt=val...] [-m mechanism] server\n",
		progname, progname);
	return EXIT_FAILURE;
    }
    if ((sasl_result = sasl_client_init((void *) &sasl_callbacks)) != SASL_OK) {
	fprintf(stderr, "%s: SASL initialization failed: %s\n",
		progname, sasl_errstring(sasl_result, NULL, NULL));
	return EXIT_FAILURE;
    }
    memset(&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(atoi(port));
    if (!sin.sin_port) {
	se = getservbyname(port, "tcp");
	if (se)
	    sin.sin_port = se->s_port;
	else
	    sin.sin_port = htons(ACAP_DEFAULT_PORT);
    }
    he = gethostbyname(argv[optind]);
    if (!he) {
	fprintf(stderr, "%s: Unable to lookup %s\n",
		progname, argv[optind]);
	return EXIT_FAILURE;
    }
    strncpy(serverFQDN, he->h_name, 1023);
    memcpy(&sin.sin_addr, he->h_addr, sizeof(sin.sin_addr));

    sasl_result = sasl_client_new("acap", serverFQDN, sasl_callbacks,
				  SASL_SECURITY_LAYER, &sasl_conn);

    if (sasl_result != SASL_OK) {
	fprintf(stderr, "%s: SASL client connection init failed: %s\n",
		progname, sasl_errstring(sasl_result, NULL, NULL));
	return EXIT_FAILURE;
    }
    if ((fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
	perror("socket");
	return EXIT_FAILURE;
    }
    if (connect(fd, (struct sockaddr *) & sin, sizeof(struct sockaddr_in))) {
	perror("connect");
	return EXIT_FAILURE;
    }
    sasl_result = sasl_setprop(sasl_conn, SASL_IP_REMOTE, &sin);
    if (sasl_result == SASL_OK) {
	int             len = sizeof(sin);

	if (getsockname(fd, (struct sockaddr *) & sin, &len)) {
	    perror("getsockname");
	    return EXIT_FAILURE;
	}
	sasl_result = sasl_setprop(sasl_conn, SASL_IP_LOCAL, &sin);
    }
    if (sasl_result != SASL_OK) {
	fprintf(stderr, "%s: SASL setprop failed: %s\n",
		progname, sasl_errstring(sasl_result, NULL, NULL));
	return EXIT_FAILURE;
    }
    do {
	n_read = read(fd, buf, BUF_SIZE);
	if (n_read < 0) {
	    perror("read");
	    return EXIT_FAILURE;
	}
	if (n_read == 0) {
	    puts("__Connection closed by foreign host__");
	    return EXIT_FAILURE;
	}
	buf[n_read] = '\0';
	printf("buf = %s\n",buf);
    } while (strncmp(buf,"* Acap",6)!=0);

    write(STDOUT_FILENO, buf, n_read);
    s = buf;
    while (*s) {
	if (*s == 's' || *s == 'S') {
	    if (!strncasecmp(s, "SASL", 4))
		break;
	}
	s++;
    }      
    if (!s) {
	fputs("Unable to find SASL information\n", stderr);
	return EXIT_FAILURE;
    }
    s += 5;
    if (want_mech) {
	/* is our mechanism supported by the server? */
	unsigned int i=0, j=0;
	while ((j < strlen(want_mech)) && (i < strlen(s))) {
	    if (toupper(s[i]) == toupper(want_mech[j])) {
		i++; j++;
	    } else {
		i = i-j+1; j = 0;
	    }
	}
	if (j != strlen(want_mech)) { /* not there */
	    fprintf(stderr, "'%s' not supported by the server, using default mechanism\n", want_mech);
	    want_mech = NULL;
	}
    }
    t = strchr(s, ')');
    if (t)
	*t = '\0';			/* throw away the rest */

    t = NULL;
    do {
	sasl_result = sasl_client_start(sasl_conn,
					want_mech ? want_mech : s,
					NULL,
					&interact,
					&t,
					&len,
					&mech);
	if (sasl_result == SASL_INTERACT) {
	    fputs("Interaction required; not supported yet\n", stderr);
	    return EXIT_FAILURE;
	}
    } while (sasl_result == SASL_INTERACT);
    if (sasl_result != SASL_CONTINUE
	&& sasl_result != SASL_OK) {
	fprintf(stderr, "%s: SASL client start failed: %s\n",
		progname, sasl_errstring(sasl_result, NULL, NULL));
	return EXIT_FAILURE;
    }
    sprintf(buf, ". Authenticate \"%s\"", mech);
    if (t && len) {
	sprintf(buf + strlen(buf), " {%d+}\r\n", len);
	write(STDOUT_FILENO, buf, strlen(buf));
	write(fd, buf, strlen(buf));
	write(fd, t, len);
	write(fd, "\r\n", 2);
	free(t);
	t = NULL;
    } else {
	strcat(buf, "\r\n");
	write(STDOUT_FILENO, buf, strlen(buf));
	write(fd, buf, strlen(buf));
    }
    if (sasl_result == SASL_OK) {
	n_read = read(fd, buf, BUF_SIZE);
	if (n_read < 0) {
	    perror("read");
	    return EXIT_FAILURE;
	}
	if (n_read == 0) {
	    puts("__Connection closed by foreign host__");
	    return EXIT_FAILURE;
	}
	buf[n_read] = '\0';
    } else
	while (sasl_result == SASL_CONTINUE || sasl_result == SASL_OK) {

	    n_read = read(fd, buf, BUF_SIZE);
	    if (n_read < 0) {
		perror("read");
		return EXIT_FAILURE;
	    }
	    if (n_read == 0) {
		puts("__Connection closed by foreign host__");
		return EXIT_FAILURE;
	    }
	    buf[n_read] = '\0';
	    if (sasl_result != SASL_CONTINUE)
		break;
	    if (buf[0] == '.')
		break;
	    if (n_read < 2
		|| buf[n_read - 1] != '\n'
		|| buf[n_read - 2] != '\r'
		|| sscanf(buf, "+ {%d}\r\n%n", &auth_size, &prefix_len) != 1
		|| auth_size + prefix_len + 2 != n_read) {
		t = strchr(buf, '\n');
		if (t)
		    write(STDOUT_FILENO, buf, t - buf);
		fputs(buf, stderr);
		return EXIT_FAILURE;
	    }
	    write(STDOUT_FILENO, buf, prefix_len);
	    do {
		sasl_result = sasl_client_step(sasl_conn,
					       buf + prefix_len,
					       auth_size,
					       &interact,
					       &t,
					       &len);
		if (sasl_result == SASL_INTERACT) {
		    fputs("Interaction required; not supported yet\n", stderr);
		    return EXIT_FAILURE;
		}
	    } while (sasl_result == SASL_INTERACT);
	    if (sasl_result == SASL_CONTINUE || sasl_result == SASL_OK) {
		sprintf(buf, "{%d+}\r\n", len);
		write(STDOUT_FILENO, buf, strlen(buf));
		write(fd, buf, strlen(buf));
		write(fd, t, len);
		write(fd, "\r\n", 2);
		free(t);
		t = NULL;
	    } else {
		fprintf(stderr, "%s: SASL client step failed: %s\n",
			progname, sasl_errstring(sasl_result, NULL, NULL));
		return EXIT_FAILURE;
	    }
	}

    if (t)
	free(t);

    write(STDOUT_FILENO, buf, n_read);
    if (strncasecmp(buf, ". OK", 3)) {
	puts("__Authentication failed; aborting connection__");
	return EXIT_FAILURE;
    }

    /* ACAP sends the last server response with the OK/NO response */
    if (strncasecmp(buf+5,"(SASL ",6)==0) {
	if (sscanf(buf+5,"(SASL {%d}\r\n%n",&auth_size, &prefix_len)==-1) {
	    puts("Parse Error\n");
	    return EXIT_FAILURE;
	}
	in = buf+5+prefix_len;
	inlen = auth_size;	    
    } else {
	in = NULL;
	inlen = 0;
    }

    do {
	sasl_result = sasl_client_step(sasl_conn,
				       in,
				       inlen,
				       &interact,
				       &t,
				       &len);
	if (sasl_result == SASL_INTERACT) {
	    fputs("Interaction required; not supported yet\n", stderr);
	    return EXIT_FAILURE;
	}
    } while (sasl_result == SASL_INTERACT);

    if ((sasl_result != SASL_OK) || (len > 0))
    {
	printf("__Authentication failed; aborting connection__ %d",len);
	return EXIT_FAILURE;
    }

    if (sasl_result != SASL_OK) {
	fprintf(stderr, "%s: SASL client startup failed: %s\n",
		progname, sasl_errstring(sasl_result, NULL, NULL));
	return EXIT_FAILURE;
    } else {
	sasl_ssf_t      *ssfp;

	sasl_result = sasl_getprop(sasl_conn, SASL_SSF, (void **) &ssfp);
	if (sasl_result != SASL_OK) {
	    fprintf(stderr, "%s: SASL getprop failed: %s\n",
		    progname, sasl_errstring(sasl_result, NULL, NULL));
	    return EXIT_FAILURE;
	}
	printf("__Authentication Succeeded: SSF=%u__\n", *ssfp);
    }

    do_newline_trans = isatty(STDIN_FILENO);

    while (1) {
	fd_set          fds;

	FD_ZERO(&fds);
	FD_SET(STDIN_FILENO, &fds);
	FD_SET(fd, &fds);
	if (select(fd + 1, &fds, NULL, NULL, NULL) < 0) {
	    perror("select");
	    return EXIT_FAILURE;
	}
	if (FD_ISSET(STDIN_FILENO, &fds)) {
	    n_read = read(STDIN_FILENO, buf, BUF_SIZE - 1);
	    if (n_read < 0) {
		perror("read");
		return EXIT_FAILURE;
	    }
	    if (n_read == 0) {
		puts("__Disconnecting__");
		break;
	    }
	    if (do_newline_trans) {
		if (buf[n_read - 1] == '\n') {
		    buf[n_read - 1] = '\r';
		    buf[n_read] = '\n';
		    n_read++;
		}
	    }
	    s = NULL;
	    if (sasl_encode(sasl_conn, buf, n_read,
			    &s, &len) != SASL_OK) {
		puts("__SASL encode failed: disconnecting__");
		break;
	    }
	    if (s && len)
		write(fd, s, len);
	    if (s)
		free(s);
	}
	if (FD_ISSET(fd, &fds)) {
	    n_read = read(fd, buf, BUF_SIZE);
	    if (n_read < 0) {
		perror("read");
		return EXIT_FAILURE;
	    }
	    if (n_read == 0) {
		puts("__Connection closed by foreign host__");
		break;
	    }
	    s = NULL;
	    if (sasl_decode(sasl_conn, buf, n_read,
			    &s, &len) != SASL_OK) {
		puts("__SASL decode failed: disconnecting__");
		break;
	    }
	    if (s && len)
		write(STDOUT_FILENO, s, len);
	    if (s)
		free(s);
	}
    }

    return EXIT_SUCCESS;
}
