/* imspd.c -- IMSP c-client based server
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 */

/* Parameter files */

#include <stdio.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "mail.h"
#include "imap2.h"
#include "osdep.h"
#include "misc.h"
#include "imutil.h"
#include "support.h"
#include "unixfile.h"
#include "acte.h"

extern struct acte_server *login_acte_server[];

/* type for literal lists */
typedef struct literal_list {
    struct literal_list *next;
    char str[1];
} literal_list;

/* Daemon files */
#define ANOFILE "/etc/anonymous.newsgroups"

/* Autologout timer */
#define TIMEOUT 60*30

/* Login tries */
#define LOGTRY 3

/* Size of temporary buffers */
#define TMPLEN 8192

/* Global storage */

char *version = "0.5a3";	/* version number of this server */
struct itimerval timer;		/* timeout state */
long idletime = 0;		/* time we became idle */
int logtry = LOGTRY;		/* login tries */
SUPPORTSTREAM *sstream;		/* support stream */
abook *ab = NULL;		/* last opened address book */
char *host = NIL;		/* local host name */
char *user = NIL;		/* user name */
char *pass = NIL;		/* password */
char *home = NIL;		/* home directory */
char cmdbuf[TMPLEN];		/* command buffer */
char *tag;			/* tag portion of command */
char *cmd;			/* command portion of command */
char *arg;			/* pointer to current argument of command */
char *lsterr = NIL;		/* last error message from c-client */
char *response = NIL;		/* command response */
literal_list *litlist = NULL;	/* buffer to hold literals */
keyvalue *kv = NULL;		/* buffer to hold keyvalues */

/* Response texts */
char *win = "%s OK %s completed\015\012";
char *altwin = "%s OK %s\015\012";
char *lose = "%s NO %s failed: %s\015\012";
char *altlose = "%s NO %s\015\012";
char *didlgn = "%s NO %s failed: Already logged in\015\012";
char *miscom = "%s BAD Missing command\015\012";
char *misarg = "%s BAD Missing required argument to %s\015\012";
char *badarg = "%s BAD Argument given to %s when none expected\015\012";
char *badkv  = "%s BAD Misformed arguments to command %s\015\012";
char *badopt = "%s BAD Bad option given to command %s\015\012";
char *nouser = "%s NO Need to login to use command %s\015\012";
char *saybye = "* BYE %s IMSP server terminating connection\015\012";
char *toobig = "* BAD Command line too long\015\012";
char *nulcmd = "* BAD Null command\015\012";
char *argrdy = "+ Ready for argument\015\012";

/* Function prototypes */
void main (int argc,char *argv[]);
void clkint ();
char *snarf (char **arg);
keyvalue *snarfkv (char **arg, int *count, char **namestr);

/* Main program */

void main (int argc,char *argv[])
{
    int i, done, didlogin;
    char *s,*t = "OK",*u,*v;
    struct hostent *hst;
    struct stat sbuf;
    void (*f) () = NIL;
    struct sockaddr_in saddr;
    literal_list *ll;

    /* initialize support stream */
    sstream = support_new();
    support_add_odriver(sstream, &unixfile_option_drvr);
    support_add_adriver(sstream, &unixfile_abook_drvr);

    /* determine remote host, if any */
    i = sizeof (saddr);
    if (getpeername(fileno(stdin), &saddr, &i) >= 0) {
	hst = gethostbyaddr((char *) &saddr.sin_addr, i, AF_INET);
	unixfile_set_lock_host(hst ? hst->h_name : inet_ntoa(saddr.sin_addr));
    }
  
    /* send greeting */
    gethostname (cmdbuf,TMPLEN-1); /* get local name */
    host = cpystr ((hst = gethostbyname (cmdbuf)) ? hst->h_name : cmdbuf);
    rfc822_date (cmdbuf);	/* get date/time now */
    didlogin = 0;
    if (i = getuid ()) {	/* logged in? */
	if (!(s = (char *) getlogin ())) s = (getpwuid (i))->pw_name;
	user = cpystr (s);	/* set up user name */
	pass = cpystr ("*");	/* and fake password */
	t = "PREAUTH";		/* pre-authorized */
	didlogin = 1;
    }
    printf ("* %s %s IMSP Service %s at %s\015\012",t,host,version,cmdbuf);
    fflush (stdout);		/* dump output buffer */

    /* setup autologout timeout */
    signal (SIGALRM,clkint);	/* prepare for clock interrupt */
    timer.it_interval.tv_sec = TIMEOUT;
    timer.it_interval.tv_usec = 0;

    /* command processing loop */
    done = 0;
    do {
	/* get a command under timeout */
	idletime = time (0);	/* get the idle time now */
	timer.it_value.tv_sec = TIMEOUT; timer.it_value.tv_usec = 0;
	setitimer (ITIMER_REAL,&timer,NIL);
	if (!fgets (cmdbuf,TMPLEN-1,stdin)) {
	    support_done(sstream);
	    _exit (1);
	}
	timer.it_value.tv_sec = timer.it_value.tv_usec = 0;
	setitimer (ITIMER_REAL,&timer,NIL);
	idletime = 0;		/* not idle any more */

	/* no more last error, literal, or keyvalue */
	if (lsterr) fs_give ((void **) &lsterr);
	while (litlist) {
	    ll = litlist->next;
	    fs_give((void **) &litlist);
	    litlist = ll;
	}
	if (kv) fs_give((void **) &kv);

	/* find end of line */
	if (!strchr (cmdbuf,'\012')) fputs (toobig,stdout);
	else if (!(tag = strtok (cmdbuf," \015\012"))) fputs (nulcmd,stdout);
	else if (!(cmd = strtok (NIL," \015\012"))) printf (miscom, tag);
	else {			/* parse command */
	    response = win;	/* set default response */
	    ucase (cmd);	/* canonicalize command case */
	    /* snarf argument */
	    arg = strtok (NIL,"\015\012");

	    if (!strcmp (cmd,"LOGOUT")) {
		if (arg) response = badarg;
		else {
		    printf(saybye, host);
		    done = 1;
		}
	    } else if (!strcmp (cmd,"NOOP")) {
		if (arg) response = badarg;
	    } else if (!strcmp (cmd, "CAPABILITY")) {
		if (arg) response = badarg;
		else printf("* CAPABILITY\015\012");
	    } else if (!strcmp (cmd,"LOGIN")) {
		struct passwd *pwd;
		if (didlogin) response = didlgn;
		else {
		    if (user) fs_give ((void **) &user);
		    if (pass) fs_give ((void **) &pass);
		    lsterr = NULL;
		    /* two arguments */
		    if (!((user = cpystr (snarf (&arg))) &&
			  (pass = cpystr (snarf (&arg))))) response = misarg;
		    else if (arg) response = badarg;
		    /* see if username and password are OK */
		    else if (server_login (user,pass,&home,argc,argv)) {
			if (lsterr) response = altwin;
			didlogin = 1;
		    } else if (!stat(ANOFILE, &sbuf)
			       && !strcmp(user, "anonymous")
			       && (pwd = getpwnam("nobody"))) {
			setgid(pwd->pw_gid);
			setuid(pwd->pw_uid);
			didlogin = 1;
		    }
		    else if (lsterr) response = lose;
		    else {
			response =
			    "%s NO Bad %s user name and/or password\015\012";
			sleep (3); /* slow the cracker down */
			if (!--logtry) {
			    printf("* BYE Too many login failures\015\012");
			    done = 1;
			}
		    }
		}
	    } else if (!strcmp (cmd,"AUTHENTICATE")) {
		char *cmdbufend;
		struct acte_server **mech;
		void *m_state;
		char *in, *out, *s, *p;
		int inlen, outlen, slen;
		int r;
		char *err;
		char *u;
		int protlevel, (*encodefunc)(), (*decodefunc)(), maxplain;
		struct passwd *pwd;
		/* one argument */
		response = 0;
		if (arg) cmdbufend = arg + strlen(arg);
		if (!(in = snarf (&arg))) response = misarg;
		else if (arg) response = badarg;
		else if (didlogin) response = didlgn;
		/* see if mechanism name is OK */
		if (!response) {
		    ucase(in);
		    for (mech = login_acte_server; *mech; mech++) {
			if (!strcmp(in, (*mech)->auth_type)) break;
		    }
		    if (!*mech) response = "%s NO Unrecognized authentication mechanism\015\012";
		}
 
		if (!response) {
		    r = (*mech)->start(server_authproc, ACTE_PROT_NONE, 0, 0, 0,
				       &outlen, &out, &m_state, &err);
		    if (r) response = "%s NO Authentication failed";
		}
		if (!response) {
		    while (r == 0) {
			s = rfc822_binary(out, outlen, &slen);
			putchar('+');
			putchar(' ');
			for (p = s; *p; p++) {
			    if (*p != '\015' && *p != '\012') putchar(*p);
			}
			putchar('\015');
			putchar('\012');
			fs_give((void **)s);
			fflush(stdout);
			alarm (TIMEOUT); /* get a reply under timeout */
			if (!fgets (cmdbufend,TMPLEN-1-(cmdbufend-cmdbuf),stdin)) _exit (1);
			alarm (0); /* make sure timeout disabled */
			if (*cmdbufend == '*' ||
			    !(in = rfc822_base64(cmdbufend, strlen(cmdbufend), &inlen))) {
			    response = "%s BAD Invalid authentication reply\015\012";
			    (*mech)->free_state(m_state);
			    break;
			}
			r = (*mech)->auth(m_state, inlen, in, &outlen, &out, &err);
			fs_give((void **)in);
		    }
		    if (r == ACTE_FAIL) {
			(*mech)->free_state(m_state);
			response = "%s NO Authentication failed\015\012";
		    }
		}
		if (!response) {
		    (*mech)->query_state(m_state, &u, &protlevel, &encodefunc,
					 &decodefunc, &maxplain);
		    fs_give ((void **) &user);
		    user = cpystr(u);
		    (*mech)->free_state(m_state);
		    if (server_authenticate(user,&home,argc,argv)) {
			didlogin = 1;
			response = win;
		    }
		    else {
			response = "%s NO Authentication failed\015\012";
		    }
		}
	    } else if (!strcmp(cmd, "CREATEADDRESSBOOK")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = misarg;
		else if (!abook_create(sstream, s)) response = lose;
	    } else if (!strcmp(cmd, "DELETEADDRESSBOOK")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = misarg;
		else if (!abook_delete(sstream, s)) response = lose;
	    } else if (!strcmp(cmd, "RENAMEADDRESSBOOK")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg))) {
		    response = misarg;
		} else if (!abook_rename(sstream, s, t)) response = lose;
	    } else if (!strcmp(cmd, "ADDRESSBOOK")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = misarg;
		else if (arg) response = badarg;
		else if (!abook_find(sstream, s)) response = lose;
	    } else if (!strcmp(cmd, "SEARCHADDRESS")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = misarg;
		else {
		    if (ab && strcmp(s, ab->name)) {
			abook_close(ab);
			ab = NULL;
		    }
		    if (!ab) ab = abook_open(sstream, s, NULL);
		    t = "*";
		    if (!ab) response = lose;
		    else if (!arg) {
			if (!abook_getlist(ab)) response = lose;
		    } else if (!(kv = snarfkv(&arg, &i, &t))) response = badkv;
		    else if (!abook_search(ab, t, kv, i)) response = lose;
		}
	    } else if (!strcmp(cmd, "FETCHADDRESS")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg))) {
		    response = misarg;
		} else if (arg) response = badarg;
		else {
		    if (ab && strcmp(s, ab->name)) {
			abook_close(ab);
			ab = NULL;
		    }
		    if (!ab) ab = abook_open(sstream, s, NULL);
		    if (!ab) response = lose;
		    else if (!abook_fetch(ab, t)) response = lose;
		}
	    } else if (!strcmp(cmd, "STOREADDRESS")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg)) || !arg) {
		    response = misarg;
		} else {
		    if (ab && strcmp(s, ab->name)) {
			abook_close(ab);
			ab = NULL;
		    }
		    if (!ab) ab = abook_open(sstream, s, NULL);
		    if (!ab) response = lose;
		    else if (!(kv = snarfkv(&arg, &i, NULL))) response = badkv;
		    else if (!abook_store(ab, t, kv, i)) response = lose;
		    else (void) abook_fetch(ab, t);
		}
	    } else if (!strcmp(cmd, "DELETEADDRESS")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg))) {
		    response = misarg;
		} else if (arg) response = badarg;
		else {
		    if (ab && strcmp(s, ab->name)) {
			abook_close(ab);
			ab = NULL;
		    }
		    if (!ab) ab = abook_open(sstream, s, NULL);
		    if (!ab) response = lose;
		    else if (!abook_deleteent(ab, t)) response = lose;
		}
	    } else if (!strcmp(cmd, "GET")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = misarg;
		else if (!option_get(sstream, s)) response = lose;
	    } else if (!strcmp(cmd, "SET")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg))) {
		    response = misarg;
		} else if (arg) response = badarg;
		else if (!option_set(sstream, s, t)) response = lose;
		else (void) option_get(sstream, s);
	    } else if (!strcmp(cmd, "UNSET")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg))) response = lose;
		else if (arg) response = badarg;
		else if (!option_set(sstream, s, NULL)) response = lose;
	    } else if (!(i = strcmp(cmd, "LOCK")) || !strcmp(cmd, "UNLOCK")) {
		if (!user) response = nouser;
		else if (!(s = snarf(&arg)) || !(t = snarf(&arg))) {
		    response = misarg;
		} else {
		    ucase(s);
		    if (!strcmp(s, "OPTION")) {
			if (arg) response = badarg;
			else if (!i) {
			    if (!option_lock(sstream, t)) {
				response = altlose;
			    } else (void) option_get(sstream, t);
			} else if (i) {
			    option_unlock(sstream, t);
			}
		    } else if (!strcmp(s, "ADDRESSBOOK")) {
			if (!(s = snarf(&arg))) response = misarg;
			else if (arg) response = badarg;
			else {
			    if (ab && strcmp(t, ab->name)) {
				abook_close(ab);
				ab = NULL;
			    }
			    if (!ab) ab = abook_open(sstream, t, NULL);
			    if (!ab) response = lose;
			    else if (!i) {
				if (!abook_lock(ab, s)) {
				    response = altlose;
				} else {
				    (void) abook_fetch(ab, s);
				}
			    } else if (i) {
				abook_unlock(ab, s);
			    }
			}			
		    } else {
			response = badopt;
		    }
		}
	    } else {
		response = "%s BAD Command unrecognized: %s\015\012";
	    }
	    /* get text for alternative win message now */
	    if (response == altwin || response == altlose) cmd = lsterr;
	    /* output response */
	    printf (response,tag,cmd,lsterr);
	}
	fflush (stdout);	/* make sure output blatted */
    } while (!done);

    /* clean up support stream */
    support_done(sstream);
    
    exit (0);			/* all done */
}


/* Clock interrupt
 */

void clkint ()
{
  fputs ("* BYE Autologout; idle for too long\015\012",stdout);
  fflush (stdout);		/* make sure output blatted */
				/* try to gracefully close the stream */
  if (sstream) support_done(sstream);
  sstream = NIL;
  exit (0);			/* die die die */
}

/* Snarf an argument
 * Accepts: pointer to argument text pointer
 * Returns: argument
 */

char *snarf (char **arg)
{
  long i;
  char *c = *arg;
  char *s = c + 1;
  char *t = NIL;
  literal_list *ll;
  
  if (!c) return NIL;		/* better be an argument */
  switch (*c) {			/* see what the argument is */
  case '\0':			/* catch bogons */
  case ' ':
    return NIL;
  case '"':			/* quoted string */
    if (!(strchr (s,'"') && (c = strtok (c,"\"")))) return NIL;
    break;
  case '{':			/* literal string */
    if (isdigit (*s)) {		/* be sure about that */
      i = strtol (s,&t,10);	/* get its length */
				/* validate end of literal */
      if (*t++ != '}' || *t++) return NIL;
      fputs (argrdy,stdout);	/* tell client ready for argument */
      fflush (stdout);		/* dump output buffer */
				/* get a literal buffer */
      ll = (literal_list *) fs_get(sizeof (literal_list) + i);
      ll->next = litlist;
      litlist = ll;
      c = ll->str;
				/* start timeout */
      timer.it_value.tv_sec = TIMEOUT; timer.it_value.tv_usec = 0;
      setitimer (ITIMER_REAL,&timer,NIL);
      while (i--) *c++ = getchar ();
      *c++ = NIL;		/* make sure string tied off */
      c = ll->str;		/* return value */
    				/* get new command tail */
      if (!fgets ((*arg = t),TMPLEN - (t - cmdbuf) - 1,stdin)) _exit (1);
				/* clear timeout */
      timer.it_value.tv_sec = timer.it_value.tv_usec = 0;
      setitimer (ITIMER_REAL,&timer,NIL);
      if (!strchr (t,'\012')) {	/* have a newline there? */
	response = toobig;	/* lost it seems */
	return NIL;
      }
      break;
    }
				/* otherwise fall through (third party IMAP) */
  default:			/* atomic string */
    c = strtok (c," \015\012");
    break;
  }
				/* remainder of arguments */
  if ((*arg = strtok (t,"\015\012")) && **arg == ' ') ++*arg;
  return c;
}

/* snarf a keyvalue list
 * Accepts: pointer to argument
 *          pointer to kv size counter
 *          pointer to "name" string (may be NULL)
 */
#define CHUNKSIZE 10
keyvalue *snarfkv(char **arg, int *count, char **namestr)
{
    int i, entries;

    entries = CHUNKSIZE;
    kv = fs_get(sizeof (keyvalue) * entries);
    i = 0;
    while (kv[i].key = snarf(arg)) {
	if (!(kv[i].value = snarf(arg))) return (NULL);
	if (namestr && !strcasecmp(kv[i].key, "name")) {
	    *namestr = kv[i].value;
	} else if (++i == entries) {
	    fs_resize((void **) &kv, sizeof (keyvalue) * (entries *= 2));
	}
    }
    *count = i;

    return (kv);
}

/* Notification event
 * Accepts: IMAP2 stream
 *	    string to log
 *	    error flag
 */

void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
  switch (errflg) {		/* action depends upon the error flag */
  case NIL:			/* information message, set as OK response */
  case PARSE:			/* parse glitch, output unsolicited OK */
    printf ("* OK %s\015\012",string);
    break;
  case WARN:			/* warning, output unsolicited NO (kludge!) */
    printf ("* NO %s\015\012",string);
    break;
  case ERROR:			/* error that broke command */
  default:			/* default should never happen */
    printf ("* BAD %s\015\012",string);
    break;
  }
}

/* Log an event for the user to see
 * Accepts: string to log
 *	    error flag
 */

void mm_log (char *string,long errflg)
{
  switch (errflg) {		/* action depends upon the error flag */
  case NIL:			/* information message, set as OK response */
    if (response == win) {	/* only if no other response yet */
      response = altwin;	/* switch to alternative win message */
      fs_give ((void **) &lsterr);
      lsterr = cpystr (string);	/* copy string for later use */
    }
    break;
  case PARSE:			/* parse glitch, output unsolicited OK */
    printf ("* OK [PARSE] %s\015\012",string);
    break;
  case WARN:			/* warning, output unsolicited NO (kludge!) */
    break;
  case ERROR:			/* error that broke command */
  default:			/* default should never happen */
    response = lose;		/* set fatality */
    fs_give ((void **) &lsterr);/* flush old error */
    lsterr = cpystr (string);	/* note last error */
    break;
  }
}


/* Log an event to debugging telemetry
 * Accepts: string to log
 */
void mm_dlog (char *string)
{
  mm_log (string,WARN);		/* shouldn't happen normally */
}

/* Get user name and password for this host
 * Accepts: host name
 *	    where to return user name
 *	    where to return password
 *	    trial count
 */
void mm_login (char *host,char *username,char *password,long trial)
{
  strcpy (username,user);	/* set user name */
  strcpy (password,pass);	/* and password */
}

/* About to enter critical code
 * Accepts: stream
 */
void mm_critical (MAILSTREAM *s)
{
  /* Not doing anything here for now */
}

/* About to exit critical code
 * Accepts: stream
 */
void mm_nocritical (MAILSTREAM *s)
{
  /* Not doing anything here for now */
}

/* Disk error found
 * Accepts: stream
 *	    system error code
 *	    flag indicating that mailbox may be clobbered
 * Returns: abort flag
 */
long mm_diskerror (MAILSTREAM *s,long errcode,long serious)
{
  if (serious) {		/* try your damnest if clobberage likely */
    fputs ("* BAD Retrying to fix probable mailbox damage!\015\012",stdout);
    fflush (stdout);		/* dump output buffer */
    alarm (0);			/* make damn sure timeout disabled */
    sleep (60);			/* give it some time to clear up */
    return NIL;
  }
				/* otherwise die before more damage is done */
  printf ("* BYE Aborting due to disk error %s\015\012",strerror (errcode));
  return T;
}

/* Log a fatal error event
 * Accepts: string to log
 */

void mm_fatal (char *string)
{
  mm_log (string,ERROR);	/* shouldn't happen normally */
}

/* Message exists (i.e. there are that many messages in the mailbox)
 * Accepts: IMAP2 stream
 *	    message number
 */
void mm_exists (MAILSTREAM *stream,long number)
{
    /* shouldn't be called */
}

/* BBoard found
 * Accepts: BBoard name
 */
void mm_bboard (char *string)
{
    /* shouldn't be called */
}

/* Mailbox found
 * Accepts: Mailbox name
 */
void mm_mailbox (char *string)
{
    /* shouldn't be called */
}

/* Message expunged
 * Accepts: IMAP2 stream
 *	    message number
 */
void mm_expunged (MAILSTREAM *stream,long number)
{
    /* shouldn't be called */
}

/* Message matches a search
 * Accepts: IMAP2 stream
 *	    message number
 */
void mm_searched (MAILSTREAM *stream,long msgno)
{
    /* shouldn't be called */
}

/* Message option
 * Accepts: SUPPORTSTREAM *
 *          option name
 *          option value
 *          read-write flag
 */
void mm_option(SUPPORTSTREAM *s, char *name, char *value, int rwflag)
{
    im_send(NULL, "* OPTION %a %s %a\015\012", name, value,
	    rwflag ? "[READ-WRITE]" : "[READ-ONLY]");
}

/* Message address book name
 * Accepts: SUPPORTSTREAM *
 *          address book name
 */
void mm_addressbook(SUPPORTSTREAM *s, char *name, char sep_char)
{
    char tmp[4];

    if (!sep_char) strcpy(tmp, "NIL");
    else sprintf(tmp, "\"%c\"", sep_char);
    im_send(NULL, "* ADDRESSBOOK () %a %s\015\012", tmp, name);
}

/* Message address book entry
 * Accepts: open address book pointer
 *          entry name
 *	    list of attributes (kv, count)
 */
void mm_address(abook *ab, char *name, keyvalue *kv, int count)
{
    if (!count) {
	im_send(NULL, "* SEARCHADDRESS %s\015\012", name);
    } else {
	im_send(NULL, "* FETCHADDRESS %s %s%k\015\012",
		ab->name, name, count, kv);
    }
}

/* Message address book access control list
 * Accepts:  open address book pointer
 *           optional ACL identifier
 *           ACL rights string
 */
void mm_abookacl(abook *ab, char *identifier, char *rights)
{
    if (identifier) {
	printf("ACL: %s %s\n", identifier, rights);
	im_send(NULL, "* ACL ADDRESSBOOK %s %s %s\015\012",
		ab->name, identifier, rights);
    } else {
	im_send(NULL, "* MYRIGHTS ADDRESSBOOK %s %s\015\012",
		ab->name, rights);
    }
}
