/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _BROWSEVIEWHANDLERS_H_
#define _BROWSEVIEWHANDLERS_H_

# include "CvsAlert.h"

// interface describing a handler to process a selection-related command
class KiSelectionHandler
{
public:
	// called for each folder in selection
	virtual void OnFolder(const char* fullpath) = 0;

	// called after all folders are processed and there are selected files
	virtual void OnFiles(MultiFiles* mf) = 0;
};


// handles moving files to trash
class KoMoveToTrashHandler : public KiSelectionHandler
{
public:
	// called for each folder in selection
	virtual void OnFolder(const char* fullpath)
	{ /* ignore */ }

	// called after all folders are processed and there are selected files
	virtual void OnFiles(MultiFiles* mf)
	{
		if( !CvsAlert("Are you sure you want to erase selected files?", "Yes", "No") )
		{
			return;
		}

		while(mf->next()) 
		{
			int count = mf->NumFiles();
			for (int i = 0; i < count; i++)
			{
				CStr path, fileName, currRev;
				if (mf->get(i, path, fileName, currRev))
				{
					char fn[_MAX_PATH];
					_makepath(fn, NULL, path, fileName, NULL);
					if(!CompatMoveToTrash(fn))
					{
						cvs_err("Unable to remove '%s' (error %d)\n", fn, errno);
					}
					else
					{
						cvs_out("'%s' has been moved successfully to the recycle bin...\n", fn);
					}
				}
			}
		}
	}

};

// handles figuring out if moving files to trash is enabled
class KoUpdateMoveToTrashHandler  : public KiSelectionHandler
{
// Attributes
	bool m_isFolders, m_isFiles;

public:
	KoUpdateMoveToTrashHandler()
	: m_isFolders(false), m_isFiles(false)
	{}

	virtual void OnFolder(const char* fullpath)
	{ 
		m_isFolders = true;
	}

	// called after all folders are processed and there are selected files
	virtual void OnFiles(MultiFiles* /*mf*/)
	{
		m_isFiles = true;
	}

	bool IsEnabled() const
	{
		return !m_isFolders && m_isFiles;
	}
};


// handles create tag request on the selected items
class KoCreateTagHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdTagCreateFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdTagCreateFiles(pmf);
	}
};

// handles delete tag request on the selected items
class KoDeleteTagHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdTagDeleteFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdTagDeleteFiles(pmf);
	}
};

// handles branch request on the selected items
class KoBranchTagHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdTagBranchFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdTagBranchFiles(pmf);
	}
};

// handles edit start request on the selected items
class KoEditHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdEditFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdEditFiles(pmf);
	}
};

// handles reserved edit start request on the selected items
class KoReservedEditHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdEditFolder(fullpath, kEditReserved);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdEditFiles(pmf, kEditReserved);
	}
};

// handles force edit start request on the selected items
class KoForceEditHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdEditFolder(fullpath, kEditForce);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdEditFiles(pmf, kEditForce);
	}
};

// handles edit stop request on the selected items
class KoUneditHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdUneditFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdUneditFiles(pmf);
	}
};

// handles releasing on the selected items
class KoReleaseHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdReleaseFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* /*pmf*/)
	{	}
};

// handles watchers on the selected items
class KoWatchersHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdWatchersFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdWatchersFiles(pmf);
	}
};

// handles editors on the selected items
class KoEditorsHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdEditorsFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdEditorsFiles(pmf);
	}
};

// handles watch on on the selected items
class KoWatchOnHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdWatchOnFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdWatchOnFiles(pmf);
	}
};

// handles watch off on the selected items
class KoWatchOffHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdWatchOffFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdWatchOffFiles(pmf);
	}
};

// handles diffing on the selected items
class KoDiffHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdDiffFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdDiffFiles(pmf);
	}
};

// handles log queries on the selected items
class KoLogHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdLogFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdLogFiles(pmf);
	}
};

// handles graph log queries on the selected items
class KoGraphHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* /*fullpath*/)
	{
		/*we avoid processing directory*/
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdLogFiles(pmf, true);
	}
};

// handles status queries on the selected items
class KoStatusHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdStatusFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdStatusFiles(pmf);
	}
};

// handles file remove requests
class KoFileRemoveHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{ /*folder cannot be removed */ }

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdRemoveFiles(pmf);
	}
};

// handles locking on the selected items
class KoLockHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdLockFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdLockFiles(pmf);
	}
};

// handles unlocking on the selected items
class KoUnlockHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdUnlockFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdUnlockFiles(pmf);
	}
};

// handles addition of the selected items
class KoAddHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdAddFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdAddFiles(pmf);
	}
};

// handles binary addition of the selected items
class KoAddBinaryHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdAddFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdAddBinaryFiles(pmf);
	}
};

// handles unicode addition of the selected items
class KoAddUnicodeHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdAddFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdAddUnicodeFiles(pmf);
	}
};

// handles commits of the selected items
class KoCommitHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdCommitFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdCommitFiles(pmf);
	}
};

// handles updates on the selected items
class KoUpdateHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdUpdateFolder(fullpath);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdUpdateFiles(pmf);
	}
};

// handles query updates on the selected items
class KoQueryUpdateHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* fullpath)
	{
		CvsCmdUpdateFolder(fullpath, true);
	}

	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdUpdateFiles(pmf, true);
	}
};

// handles annotate on the selected items
class KoAnnotateHandler : public KiSelectionHandler
{
	virtual void OnFolder(const char* /*fullpath*/)
	{
		/*we avoid processing directory*/
	}
	
	virtual void OnFiles(MultiFiles* pmf)
	{
		CvsCmdAnnotateFiles(pmf);
	}
};

#endif
// OWSEVIEWHANDLERS_H_
