/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@hotmail.com> --- August 2001
 */

/*
 * PythonGlue.cpp --- glue to the Python language
 */

#include "stdafx.h"

#include <stdarg.h>
#include <string.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef WIN32
#	include "resource.h"
#	include "NoPython.h"
#endif /* WIN32 */

#include "PythonGlue.h"
#include "CvsLog.h"
#include "AppConsole.h"
#include "MacrosSetup.h"
#include "umenu.h"
#include "PythonContainer.h"
#include "PythonImpl.h"
#include "TclGlue.h"
#include "AppGlue.h"
#include "CvsArgs.h"
#include "FileTraversal.h"
#include "AskYesNo.h"
#include "dll_loader.h"
#include "CvsPrefs.h"

#if USE_PYTHON
#include <Python.h>
#include <osdefs.h>
#endif

#if qMacCvsPP
#	include <macglue.h>
#	include "MacCvsApp.h"
#	include "MacCvsConstant.h"
#	include "MacMisc.h"
#endif

#ifdef WIN32
#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#	include "wincvs_winutil.h"
#endif /* WIN32 */

#ifndef NEW
#define NEW new
#endif

#if USE_PYTHON

PyThreadState *CPython_Interp::m_mainInterpreter = 0L;
static bool gUICacheValid = false;
static std::vector<ENTNODE> gUICache;
static PyObject *gUICacheObj = NULL;

class CPyThreadState
{
public:
	CPyThreadState(PyThreadState *p) : m_p(p)
	{
		m_p = PyThreadState_Swap(p);
	}
	~CPyThreadState()
	{
		PyThreadState_Swap(m_p);
	}
protected:
	PyThreadState *m_p;
};

enum
{
	_stdout_style = 15,
	_stderr_style = 16,
	_trace_style = 17
};

static PyObject *cvsgui_write(PyObject *self, PyObject *args)
{
	char *str;
	int style;
	PyObject *shell;

	if(!PyArg_ParseTuple(args, "siO", &str, &style, &shell))
		return NULL;

	TRY
	{
		if(style == _stdout_style)
			cvs_outstr(str, strlen(str));
		else
			cvs_errstr(str, strlen(str));
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return PyContainer::Void();
}

static PyObject *cvsgui_GetMainMenuWidget(PyObject *self, PyObject *args)
{
	int res = 0;

	if(!PyArg_ParseTuple(args, ""))
		return NULL;

	TRY
	{
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return PyInt_FromLong(res);
}

static PyObject *cvsgui_GetTopMacroSelectionMenu(PyObject *self, PyObject *args)
{
	static UMENU macroMenu = 0L;

	if(!PyArg_ParseTuple(args, ""))
		return NULL;

	PyAutoObject<PyMenuMgr::obj_t> res = PyMenuMgr::New();
	if (!res.IsValid())
		return NULL;

	TRY
	{
#ifdef WIN32
		int pos;
		if(macroMenu == 0L)
		{
			CMenu *top = GetMenuEntry(ID_MACRO_SEL, pos);
			if(top)
			{
				top->DeleteMenu(pos, MF_BYPOSITION);
				macroMenu = (UMENU)(HMENU)*top;
			}
		}
#endif
#if qMacCvsPP
		if(macroMenu == 0L)
		{
			LMenuBar *menuBar	= LMenuBar::GetCurrentMenuBar();
			LMenu *menu = menuBar->FetchMenu(menu_MacroSel);
			if(menu)
				macroMenu = menu->GetMenuID();
		}
#endif

		res->m_object = new PyMenuMgr(macroMenu);
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return res.CopyReturn();
}

static PyObject *cvsgui_GetTopMacroAdminMenu(PyObject *self, PyObject *args)
{
	static UMENU macroMenu = 0L;

	if(!PyArg_ParseTuple(args, ""))
		return NULL;

	PyAutoObject<PyMenuMgr::obj_t> res = PyMenuMgr::New();
	if (!res.IsValid())
		return NULL;

	TRY
	{
#ifdef WIN32
		int pos;
		if(macroMenu == 0L)
		{
			CMenu *top = GetMenuEntry(ID_MACRO_ADMIN, pos);
			if(top)
			{
				top->DeleteMenu(pos, MF_BYPOSITION);
				macroMenu = (UMENU)(HMENU)*top;
			}
		}
#endif
#if qMacCvsPP
		if(macroMenu == 0L)
		{
			LMenuBar *menuBar	= LMenuBar::GetCurrentMenuBar();
			LMenu *menu = menuBar->FetchMenu(menu_MacroAdmin);
			if(menu)
				macroMenu = menu->GetMenuID();
		}
#endif
		res->m_object = new PyMenuMgr(macroMenu);
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return res.CopyReturn();
}

static PyObject *cvsgui_GetSelection(PyObject *self, PyObject *args)
{
	if(!PyArg_ParseTuple(args, ""))
		return NULL;

	if(gUICacheObj != 0L)
	{
		Py_INCREF(gUICacheObj);
		return gUICacheObj;
	}

	gUICacheObj = PyTuple_New(gUICache.size());
	if(gUICacheObj == NULL)
		return NULL;

	std::vector<ENTNODE>::const_iterator i;
	int cnt;
	for(cnt = 0, i = gUICache.begin(); i != gUICache.end(); ++i, ++cnt)
	{
		PyAutoObject<PyCvsEntry::obj_t> res = PyCvsEntry::New();
		if (!res.IsValid())
			return NULL;

		res->m_object = new PyCvsEntry(*i);

		PyTuple_SET_ITEM(gUICacheObj, cnt, res.CopyReturn());
	}

	Py_INCREF(gUICacheObj);
	return gUICacheObj;
}

class CFillPyEntries : public TraversalReport
{
public:
	CSortList<ENTNODE> & m_entries;
	std::vector<CStr> m_ignlist;

	CFillPyEntries(CSortList<ENTNODE> & entries) :
		m_entries(entries) {}

	virtual ~CFillPyEntries() {}

	virtual kTraversal EnterDirectory(const char *fullpath, const char *dirname, const FSSpec * macspec)
	{
		Entries_Open (m_entries, fullpath);
		BuildIgnoredList(m_ignlist, fullpath);
		return kContinueTraversal;
	}

	virtual kTraversal ExitDirectory(const char *fullpath)
	{
		m_ignlist.erase(m_ignlist.begin(), m_ignlist.end());
		return kContinueTraversal;
	}

	virtual kTraversal OnError(const char *err, int errcode)
	{
		return kTraversalError;
	}

	virtual kTraversal OnIdle(const char *fullpath)
	{
		return kContinueTraversal;
	}

	virtual kTraversal OnDirectory(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, const FSSpec * macspec)
	{
#if defined(WIN32) || defined(macintosh)
		if(stricmp(name, "CVS") == 0)
#else
		if(strcmp(name, "CVS") == 0)
#endif
			return kSkipFile;

		/*EntnodeData *data = */Entries_SetVisited(fullpath, m_entries, name, dir, true, &m_ignlist);

		return kSkipFile;
	}

	virtual kTraversal OnAlias(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, const FSSpec * macspec)
	{
		return OnFile(fullpath, fullname, name, dir, macspec);
	}

	virtual kTraversal OnFile(const char *fullpath,
		const char *fullname,
		const char *name,
		const struct stat & dir, const FSSpec * macspec)
	{
		/*EntnodeData *data = */Entries_SetVisited(fullpath, m_entries, name, dir, false, &m_ignlist);

		return kContinueTraversal;
	}
};

static PyObject *cvsgui_GetCvsEntries(PyObject *self, PyObject *args)
{
	char *path;

	if(!PyArg_ParseTuple(args, "s", &path))
		return NULL;

	CSortList<ENTNODE> entries(200, ENTNODE::Compare);
	CFillPyEntries traverse(entries);
	/*kTraversal res = */FileTraverse(path, traverse);

	// add the missing files
	Entries_SetMissing(entries);

	int numEntries = entries.NumOfElements();
	PyObject *obj = PyTuple_New(numEntries);
	if(obj == NULL)
		return NULL;

	for(int i = 0; i < numEntries; i++)
	{
		const ENTNODE & theNode = entries.Get(i);
		EntnodeData *data = theNode.Data();

		PyAutoObject<PyCvsEntry::obj_t> res = PyCvsEntry::New();
		if (!res.IsValid())
			return NULL;

		res->m_object = new PyCvsEntry(data);

		PyTuple_SET_ITEM(obj, i, res.CopyReturn());
	}

	return obj;
}

#if defined(WIN32)
//
// Get correct (UTC) file modification time.
// Used for dealing with notorious DST bug.
//
static PyObject * cvsgui_GetFileModTime(PyObject *self, PyObject *args)
{
    struct wnt_stat sb;
    struct tm * p_utc = NULL;

	const char *path;

// PyObject_Length exists only for debugging version (ActiveState Python 2.1)
#if defined(_DEBUG)
    int len = PyObject_Length(args);
    if (len < 1)
    {
        char errstr[255];
        sprintf(errstr, "Too few arguments (%d, expected 1)", len);
        PyErr_SetString(gCvsguiErr, errstr);
        return NULL;
    }
    else if (len > 1)
    {
        char errstr[255];
        sprintf(errstr, "Too many arguments (%d, expected 1)", len);
        PyErr_SetString(gCvsguiErr, errstr);
        return NULL;
    }
#endif

	if(!PyArg_ParseTuple(args, "s", &path))
		return NULL;

    if (wnt_stat(path, &sb) != 0)
    {
        char errstr[255];
        sprintf(errstr, "Could not find file \"%s\"", path);
        PyErr_SetString(gCvsguiErr, errstr);
        return NULL;
    }

    p_utc = gmtime(& sb.st_mtime);
    return Py_BuildValue("s", asctime(p_utc));
}
#endif	// defined(_WIN32)

static PyObject *cvsgui_LaunchTclMacro(PyObject *self, PyObject *args)
{
	char *macroName;

	if(!PyArg_ParseTuple(args, "s", &macroName) || !CTcl_Interp::IsAvail())
		return NULL;

	std::vector<ENTNODE>::const_iterator i;
	TclBrowserReset();
	for(i = gUICache.begin(); i != gUICache.end(); ++i)
	{
		EntnodeData *data = *i;
		TclBrowserAppend((*data)[EntnodeData::kPath], data);
	}

	CTcl_Interp interp;
	CStr path = macroName;
	CTcl_Interp::Unixfy(path);
	interp.DoScriptVar("source \"%s\"", (const char *)path);

	return PyContainer::Void();
}

class CBuffer
{
public:
	CBuffer() : m_curSize(0)
	{
	}

	void append(const char *ptr, size_t s)
	{
		if((s + m_curSize) > m_buf.size())
		{
			size_t step = s > 1024 ? s : 1024;
			m_buf.AdjustSize(m_buf.size() + step);
		}
#ifdef WIN32
		char *tmp = (char *)m_buf + m_curSize;
		size_t incs = 0;
		while(s--)
		{
			char c = *ptr++;
			if(c == '\r')
				continue;
			*tmp++ = c;
			incs++;
		}
		m_curSize += incs;
#elif defined(macintosh)
		char *tmp = (char *)m_buf + m_curSize;
		size_t cnt = s;
		while(cnt--)
		{
			char c = *ptr++;
			if(c == '\r')
				c = '\n';
			*tmp++ = c;
		}
		m_curSize += s;
#else
		memcpy((char *)m_buf + m_curSize, ptr, s * sizeof(char));
		m_curSize += s;
#endif
	}

	inline size_t size() const { return m_curSize; }
	inline operator char *(void) const { return m_buf; }

protected:
	size_t m_curSize;
	CStaticAllocT<char> m_buf;
};

class CPyCvsConsole : public CCvsConsole
{
public:
	CPyCvsConsole()	{}
	virtual ~CPyCvsConsole() {}
	
	virtual long cvs_out(char *txt, long len)
	{
		m_out.append(txt, len);
		return len;
	}
	
	virtual long cvs_err(char *txt, long len)
	{
		m_err.append(txt, len);
		return len;
	}

	CBuffer m_out;
	CBuffer m_err;
};

static PyObject *cvsgui_PromptMessage(PyObject *self, PyObject *args)
{
	char *msg, *title, *ok, *cncl;

	if(!PyArg_ParseTuple(args, "ssss", &msg, &title, &ok, &cncl))
		return NULL;

	bool res;

	TRY
	{
		res = PromptMessage(msg, title, ok, cncl);
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	if(res)
	{
		return PyContainer::True();
	}
	
	return PyContainer::False();
}

static PyObject *cvsgui_PromptEditMessage(PyObject *self, PyObject *args)
{
	char *msg, *title, *ok, *cncl, *edit;

	if(!PyArg_ParseTuple(args, "sssss", &msg, &edit, &title, &ok, &cncl))
		return NULL;

	bool res;
	UStr ed;

	TRY
	{
		ed = edit;
		res = PromptEditMessage(msg, ed, title, ok, cncl);
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return Py_BuildValue("(is)", res ? 1 : 0, ed.empty() ? "" : ed.c_str());
}

static PyObject *cvsgui_LaunchBrowser(PyObject *self, PyObject *args)
{
	char *url;

	if(!PyArg_ParseTuple(args, "s", &url))
		return NULL;

#ifdef WIN32
	ShellExecute(NULL, "open", url, /*postData*/0L, NULL, SW_SHOWNORMAL);
#endif

	return PyContainer::Void();
}

static void outputXML(FILE *out, CLogNode *node)
{
	switch(node->GetType())
	{
		case kNodeHeader :
		{
			fputs("<RCS>\n", out);
			const CRcsFile rcs = **(CLogNodeHeader *)node;
			fprintf(out, "<RCSFILE>%s</RCSFILE>\n", rcs.RcsFile().c_str());
			fprintf(out, "<WORKINGFILE>%s</WORKINGFILE>\n", rcs.WorkingFile().c_str());
			fprintf(out, "<HEADREV>%s</HEADREV>\n", rcs.HeadRev().c_str());
			fprintf(out, "<BRANCHREV>%s</BRANCHREV>\n", rcs.BranchRev().c_str());
			fprintf(out, "<KEYWORDSUBST>%s</KEYWORDSUBST>\n", rcs.KeywordSubst().c_str());
			fprintf(out, "<DESCRIPTION>%s</DESCRIPTION>\n", rcs.DescLog().c_str());
			fprintf(out, "<SELREVISIONS>%d</SELREVISIONS>\n", rcs.SelRevisions());
			fprintf(out, "<TOTREVISIONS>%d</TOTREVISIONS>\n", rcs.TotRevisions());
			fprintf(out, "<LOCKSTRICT>%d</LOCKSTRICT>\n", rcs.LockStrict() ? 1 : 0);

			fputs("<ACCESSLIST>\n", out);
			const std::vector<CLogStr> & ac = rcs.AccessList();
			std::vector<CLogStr>::const_iterator i;
			for(i = ac.begin(); i != ac.end(); ++i)
			{
				fprintf(out, "<ACCESS>%s</ACCESS>\n", (*i).c_str());
			}
			fputs("</ACCESSLIST>\n", out);

			fputs("<SYMBOLICLIST>\n", out);
			const std::vector<CRevNumber> & sy = rcs.SymbolicList();
			std::vector<CRevNumber>::const_iterator j;
			for(j = sy.begin(); j != sy.end(); ++j)
			{
				fprintf(out, "<SYMBOLIC>%s</SYMBOLIC>\n", (*j).c_str());
			}
			fputs("</SYMBOLICLIST>\n", out);

			fputs("<LOCKLIST>\n", out);
			const std::vector<CRevNumber> & lo = rcs.LocksList();
			for(j = lo.begin(); j != lo.end(); ++j)
			{
				fprintf(out, "<LOCK>%s</LOCK>\n", (*j).c_str());
			}
			fputs("</LOCKLIST>\n", out);
			break;
		}
		case kNodeRev :
		{
			const CRevFile & rev = **(CLogNodeRev *)node;
			fputs("<REVISION>\n", out);
			fprintf(out, "<NUMBER>%s</NUMBER>\n", rev.RevNum().c_str());
			const struct tm & tm = rev.RevTime();
			fprintf(out, "<DATE>%d/%d/%d %d:%d:%d</DATE>\n", (tm.tm_year + 1900),
				tm.tm_mon, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
			fprintf(out, "<AUTHOR>%s</AUTHOR>\n", rev.Author().c_str());
			fprintf(out, "<STATE>%s</STATE>\n", rev.State().c_str());
			fprintf(out, "<CHANGES>+%d%d</CHANGES>\n", rev.ChgPos(), rev.ChgNeg());
			fprintf(out, "<DESCRIPTION>%s</DESCRIPTION>\n", rev.DescLog().c_str());

			fputs("<BRANCHLIST>\n", out);
			const std::vector<CRevNumber> & sy = rev.BranchesList();
			std::vector<CRevNumber>::const_iterator j;
			for(j = sy.begin(); j != sy.end(); ++j)
			{
				fprintf(out, "<REVBRANCH>%s</REVBRANCH>\n", (*j).c_str());
			}
			fputs("</BRANCHLIST>\n", out);
			break;
		}
		case kNodeBranch :
		{
			fputs("<BRANCH>\n", out);
			const CLogStr & branch = **(CLogNodeBranch *)node;
			fprintf(out, "<BRANCHNAME>%s</BRANCHNAME>\n", branch.c_str());
			break;
		}
		case kNodeTag :
		{
			fputs("<TAG>\n", out);
			const CLogStr & tag = **(CLogNodeTag *)node;
			fprintf(out, "<TAGNAME>%s</TAGNAME>\n", tag.c_str());
			break;
		}
	}

	if(node->Childs().size() > 0)
	{
		fputs("<REVISIONS>\n", out);
		std::vector<CLogNode *>::const_iterator i;
		for(i = node->Childs().begin(); i != node->Childs().end(); ++i)
		{
			outputXML(out, *i);
		}
		fputs("</REVISIONS>\n", out);
	}

	switch(node->GetType())
	{
		case kNodeHeader :
			fputs("</RCS>\n", out);
			break;
		case kNodeRev :
			fputs("</REVISION>\n", out);
			break;
		case kNodeBranch :
			fputs("</BRANCH>\n", out);
			break;
		case kNodeTag :
			fputs("</TAG>\n", out);
			break;
	}

	if(node->Next() != 0L)
	{
		outputXML(out, node->Next());
	}
}

static PyObject *cvsgui_GetHistoryAsXML(PyObject *self, PyObject *args)
{
	PyObject *logfile, *xmlfile;

	if(!PyArg_ParseTuple(args, "O!O!", GetPyFile_Type(), &logfile,
		GetPyFile_Type(), &xmlfile))
			return NULL;

	TRY
	{
		std::vector<CRcsFile> & allRcs = CvsLogParse(PyFile_AsFile(logfile));

		std::vector<CRcsFile>::iterator i;
		FILE *xml = PyFile_AsFile(xmlfile);
		fputs("<ROOT>\n", xml);
		for(i = allRcs.begin(); i != allRcs.end(); ++i)
		{
			std::auto_ptr<CLogNode> node(CvsLogGraph(*i));
			outputXML(xml, node.get());
		}
		fputs("</ROOT>\n", xml);

		CvsLogReset();
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return PyContainer::Void();
}

static PyObject *cvsgui_RunCvs(PyObject *self, PyObject *args)
{
	CvsArgs arg;
	PyObject *cargs;
	int exitOnError = 1;
	int exitc;
	CPyCvsConsole console;

	if(!PyArg_ParseTuple(args, "O!|i", GetPyTuple_Type(), &cargs, &exitOnError))
		return NULL;

	TRY
	{
		int argc = PyTuple_Size(cargs);
		for(int i = 0; i < argc; i++)
		{
			PyObject *o = PyTuple_GET_ITEM(cargs, i);
			if(!PyString_Check(o))
			{
				PyErr_SetString(gCvsguiErr, "Wrong argument type, should be string");
				return NULL;
			}

			arg.add(PyString_AsString(o));
		}

		exitc = launchCVS(0L, arg.Argc(), arg.Argv(), &console);

		if(exitOnError && exitc != 0)
		{
			UStr str;
			str.set(console.m_err, console.m_err.size());
			str << "cvs returned an non-zero error code (";
			str << exitc;
			str << ")\n";
			PyErr_SetString(gCvsguiErr, str);
			return NULL;
		}
	}
	CATCH_ALL(e)
	{
		PyHandleCvsguiError(e);
		return NULL;
	}
	END_CATCH_ALL

	return Py_BuildValue("(is#s#)", exitc, (char *)console.m_out, console.m_out.size(),
		(char *)console.m_err, console.m_err.size());
}

static PyMethodDef cvsgui_methods[] =
{
	{"write", (PyCFunction)cvsgui_write, METH_VARARGS, ""},
	{"GetMainMenuWidget", (PyCFunction)cvsgui_GetMainMenuWidget, METH_VARARGS, ""},
	{"GetTopMacroSelectionMenu", (PyCFunction)cvsgui_GetTopMacroSelectionMenu, METH_VARARGS, ""},
	{"GetTopMacroAdminMenu", (PyCFunction)cvsgui_GetTopMacroAdminMenu, METH_VARARGS, ""},
	{"GetSelection", (PyCFunction)cvsgui_GetSelection, METH_VARARGS, ""},
	{"GetCvsEntries", (PyCFunction)cvsgui_GetCvsEntries, METH_VARARGS, ""},
#if defined(WIN32)
	{"GetFileModTime", (PyCFunction)cvsgui_GetFileModTime, METH_VARARGS, ""},
#endif
	{"LaunchTclMacro", (PyCFunction)cvsgui_LaunchTclMacro, METH_VARARGS, ""},
	{"RunCvs", (PyCFunction)cvsgui_RunCvs, METH_VARARGS, ""},
	{"PromptMessage", (PyCFunction)cvsgui_PromptMessage, METH_VARARGS, ""},
	{"PromptEditMessage", (PyCFunction)cvsgui_PromptEditMessage, METH_VARARGS, ""},
	{"LaunchBrowser", (PyCFunction)cvsgui_LaunchBrowser, METH_VARARGS, ""},
	{"GetHistoryAsXML", (PyCFunction)cvsgui_GetHistoryAsXML, METH_VARARGS, ""},
	{NULL, NULL}
};

static std::vector<PyMethodDef> gAllMethods;

static void InitializeInterp(PyThreadState *interp)
{
	CPyThreadState lock(interp);

	// first time, create the method table for cvsgui module
	if(gAllMethods.empty())
	{
		PyMethodDef *tmp = cvsgui_methods;
		while(tmp->ml_name)
		{
			gAllMethods.push_back(*tmp);
			tmp++;
		}

		PyRegistry::RegisterAllObjects(gAllMethods);
	}

	// this module contains the glue code functions
	PyObject *m = Py_InitModule4("_cvsgui", &gAllMethods[0],
		"The cvsgui glue functions", (PyObject*)NULL,PYTHON_API_VERSION);
	
	if(m == NULL)
	{
		cvs_err("Could not create the Python cvsgui module !\n");
		return;
	}

	// this is the exception we raised when the error comes from us
	PyObject *d = PyModule_GetDict(m);
	if(gCvsguiErr == NULL)
		gCvsguiErr = PyErr_NewException("_cvsgui.error", NULL, NULL);
	if (gCvsguiErr != NULL)
		PyDict_SetItemString(d, "error", gCvsguiErr);

	PyRegistry::CallInitValues(d);

	// Ideally we need to use Jonathan's mechanism
	PyObject *ver = PyString_FromString("1.3");
	if (ver != NULL) {
		PyDict_SetItemString(d, "CVSGUI_VERSION", ver);
		Py_DECREF(ver);
	}

	char *script = NULL;

#ifdef WIN32
	HINSTANCE hInst = ::AfxGetInstanceHandle();
	HRSRC rsrc = FindResource(hInst, "PYSHELL", "PYSCRIPT");

	if(rsrc)
	{
		HGLOBAL h = LoadResource(hInst, rsrc);
		if(h)
		{
			DWORD size = SizeofResource(hInst, rsrc);
			script = (char *)malloc(size + 1);
			char *ptr = (char *)LockResource(h);

			if(script)
			{
				char *tmp = script;
				while(size--)
				{
					char c = *ptr++;
					if(c == '\r')
						continue;
					*tmp++ = c;
				}
				*tmp = '\0';
			}

			FreeResource(h);
		}
	}

#endif // WIN32
#if qMacCvsPP
	StResource rsrcH(FOUR_CHAR_CODE('PYTH'), 128);
	size_t size = GetHandleSize(rsrcH);
	script = (char *)malloc(size + 1);
	if(script)
	{
		char *ptr = (char *)*rsrcH;
		char *tmp = script;
		while(size--)
		{
			char c = *ptr++;
			if(c == '\r')
				c = '\n';
			*tmp++ = c;
		}
		*tmp = '\0';
	}
#endif

	// augment the library path of python with our library
	CStr ourLibPath;
	PythonLibGetLoc(ourLibPath);

	UStr libPath = Py_GetPath();
	if(!ourLibPath.empty())
	{
		libPath << DELIM;
		libPath << ourLibPath;
		PySys_SetPath((char *)(const char *)libPath);
	}

	// this is the script to setup stdout/stderr and reroute it
	// to the console window
	if(script)
	{
		PyRun_SimpleString(script);

		free(script);
	}
	else
	{
		cvs_err("Could not initialize properly the Python interpreter !\n");
	}
}
#endif // USE_PYTHON

CPython_Interp::CPython_Interp(bool mainInterp)
{
	fInterp = 0L;

#if USE_PYTHON
	if(!CPython_Interp::IsAvail())
	{
		cvs_err("Python is not available !\n");
		return;
	}
	
	if(mainInterp)
	{
		fInterp = m_mainInterpreter;
		gCvsPrefs.SetTclFileRunning(true);
		return;
	}

	{
		CPyThreadState lock(0L);

		fInterp = Py_NewInterpreter();
		if(fInterp == NULL)
			return;
	}

	InitializeInterp(fInterp);
	gCvsPrefs.SetTclFileRunning(true);
#endif
}

CPython_Interp::~CPython_Interp()
{
#if USE_PYTHON
	if(fInterp != 0L)
	{
		gCvsPrefs.SetTclFileRunning(false);

		if(fInterp != m_mainInterpreter)
		{
			CPyThreadState lock(fInterp);

			Py_EndInterpreter(fInterp);
		}
	}
#endif
}

bool CPython_Interp::IsAvail(void)
{
#if USE_PYTHON
#	if qMacCvsPP
	return PyThreadState_Swap != 0L;
#	elif defined(WIN32) && !qCvsDebug /* || qCarbon */
	static bool firstTime = true;
	static bool pyAvail = false;
	if(firstTime)
	{
#	if defined(WIN32) || qCarbon
		CompatConnectID connID;
#	endif

#	ifdef WIN32
		if(!pyAvail)
			pyAvail = CompatLoadLibrary(&connID, "python21.dll") != 0 ||
				CompatLoadLibrary(&connID, "python22.dll") != 0;
#	endif
	
#	if qCarbon
		if (UEnvironment::GetOSVersion() >= 0x1000)
			pyAvail = CompatLoadLibrary(&connID, "/System/Library/Frameworks/Tcl.framework");
		else
			pyAvail = CompatLoadLibrary(&connID, "tcl8.3");
#	endif

#	if defined(WIN32) || qCarbon
#	define LOAD_CODEFRAG(name) \
		if(pyAvail) \
			pyAvail = (*(void **)&name##_glue = \
				_CompatCallLibrary(connID, #name)) != 0L
#	endif
#	if defined(WIN32) || TARGET_API_MAC_CARBON
		LOAD_CODEFRAG(PyThreadState_Swap);
		LOAD_CODEFRAG(PyDict_SetItemString);
		LOAD_CODEFRAG(PyErr_Clear);
		LOAD_CODEFRAG(PyInt_FromLong);
		LOAD_CODEFRAG(PyErr_SetString);
		LOAD_CODEFRAG(PyArg_ParseTuple);
		LOAD_CODEFRAG(PyTuple_New);
		LOAD_CODEFRAG(Py_BuildValue);
		LOAD_CODEFRAG(PyFile_AsFile);
		LOAD_CODEFRAG(PyString_AsString);
		LOAD_CODEFRAG(PyTuple_Size);
		LOAD_CODEFRAG(PyRun_SimpleString);
		LOAD_CODEFRAG(PyRun_SimpleFile);
		LOAD_CODEFRAG(Py_SetProgramName);
		LOAD_CODEFRAG(Py_Initialize);
		LOAD_CODEFRAG(Py_Finalize);
		LOAD_CODEFRAG(Py_IsInitialized);
		LOAD_CODEFRAG(Py_NewInterpreter);
		LOAD_CODEFRAG(Py_EndInterpreter);
		LOAD_CODEFRAG(PyString_FromString);
		LOAD_CODEFRAG(PyErr_NewException);
		LOAD_CODEFRAG(PyModule_GetDict);
		LOAD_CODEFRAG(Py_InitModule4);
		LOAD_CODEFRAG(PyEval_InitThreads);
		LOAD_CODEFRAG(Py_FindMethod);
		LOAD_CODEFRAG(_PyObject_New);
		LOAD_CODEFRAG(_PyObject_Del);
		LOAD_CODEFRAG(PySys_SetPath);
		LOAD_CODEFRAG(Py_GetPath);

		LOAD_CODEFRAG(PyType_Type);
		LOAD_CODEFRAG(PyFile_Type);
		LOAD_CODEFRAG(PyString_Type);
		LOAD_CODEFRAG(PyTuple_Type);
		LOAD_CODEFRAG(PyInt_Type);
		LOAD_CODEFRAG(_Py_NoneStruct);
		LOAD_CODEFRAG(_Py_ZeroStruct);
		LOAD_CODEFRAG(_Py_TrueStruct);

		if(!pyAvail)
		{
			WarnNoPython();
		}
#	endif /* WIN32 */

		firstTime = false;
	}

	return pyAvail;
#	else
	return true;
#	endif
#else // !USE_PYTHON
	return false;
#endif // !USE_PYTHON
}

int CPython_Interp::DoScript(const char *script)
{
#if USE_PYTHON
	if(fInterp == 0L)
		return -1;

	CPyThreadState lock(fInterp);

	PyRun_SimpleString((char *)script);

	return 0;
#else
	return -1;
#endif
}

int CPython_Interp::DoScriptVar(const char *format, ...)
{
#if USE_PYTHON
	if(fInterp == 0L)
		return -1;

	va_list args;
	char script[1024] = {'\0'};

	va_start (args, format);
	vsprintf (script, format, args);
	va_end (args);
	
	return DoScript(script);
#else
	return -1;
#endif
}

int CPython_Interp::DoFile(const char *file)
{
#if USE_PYTHON
	if(fInterp == 0L)
		return -1;

	CPyThreadState lock(fInterp);

	FILE *f = fopen(file, "r");
	if(f)
	{
		PyRun_SimpleFile(f, (char *)file);
		fclose(f);
	}

	return 0;
#else
	return -1;
#endif
}

#if qMacCvsPP
	// work around for a bug of Python 2.2 which draws the
	// Sioux menu even though we have a custom console
extern int console_output_state;
#endif

void CPython_Interp::InitPythonSupport(void)
{
#if USE_PYTHON
	if(!CPython_Interp::IsAvail())
		return;

#ifdef WIN32
	Py_SetProgramName("wincvs");
	Py_Initialize();
#elif qMacCvsPP
	console_output_state = 1;
	
	PyMac_SetConsoleHandler(PyMac_DummyReadHandler, PyMac_DummyWriteHandler,
		PyMac_DummyWriteHandler);
	
	PyMacSchedParams scp;
	
	PyMac_GetSchedParams(&scp);
	scp.process_events = 0;
	PyMac_SetSchedParams(&scp);

	Py_SetProgramName("maccvs");
	PyMac_Initialize();
#else
	Py_SetProgramName("gcvs");
	Py_Initialize();
#endif

	PyEval_InitThreads();
	m_mainInterpreter = PyThreadState_Swap(NULL);

	InitializeInterp(m_mainInterpreter);
#endif
}

void CPython_Interp::ClosePythonSupport(void)
{
#if USE_PYTHON
	if(!CPython_Interp::IsAvail())
		return;

	PyThreadState_Swap(m_mainInterpreter);
//	PyEval_ReleaseLock();

	PyInvalUICache();
	
	Py_Finalize();
	m_mainInterpreter = 0L;
#endif
}

void PyMacrosReloadAll(void)
{
	// get the macro folder
	UStr path;
	MacrosGetLoc(path);

#ifdef WIN32
	CString newpath(path);
	newpath.Replace("\\", "\\\\");
	path = newpath;
#endif

	CPython_Interp interp;
	interp.DoScriptVar("from cvsgui.MacroRegistry import *; "
		"gMacroRegistry.LoadMacros(\"%s\")", (char *)path.c_str());;
}

void PyDoCmdUI(UCmdUI *ucmdui)
{
	CPython_Interp interp;
	interp.DoScriptVar("from cvsgui.MacroRegistry import *; "
		"from cvsgui.CmdUI import *; "
		"gMacroRegistry.DoCmdUI(CmdUI(%d), %d)", (long)ucmdui, ucmdui->m_nID);
}

void PyDoPython(int cmd)
{
	CPython_Interp interp;
	interp.DoScriptVar("from cvsgui.MacroRegistry import *; "
		"gMacroRegistry.DoCmd(%d)", cmd);
}

bool PyIsUICacheValid()
{
#if USE_PYTHON
	return gUICacheValid;
#else
	return false;
#endif
}

void PyValidUICache()
{
#if USE_PYTHON
	gUICacheValid = true;
#endif
}

void PyInvalUICache()
{
#if USE_PYTHON
	gUICache.erase(gUICache.begin(), gUICache.end());
	gUICacheValid = false;
	Py_XDECREF(gUICacheObj);
	gUICacheObj = 0L;
#endif
}

void PyAppendCache(EntnodeData *data)
{
#if USE_PYTHON
	Py_XDECREF(gUICacheObj);
	gUICacheObj = 0L;
	gUICache.push_back(ENTNODE(data));
	gUICacheValid = true;
#endif
}
