/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * plugin.c modified for the cvsgui project by :
 * Alexandre Parenteau <aubonbeurre@hotmail.com> --- November 1999
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include <stack>
#include <vector>
#include <algorithm>

#include "cvsgui_process.h"
#include "cvsgui_protocol.h"

static int  cvs_process_write                 (int                fd,
										   guint8            *buf,
										   gulong             count);
static int  cvs_process_flush                 (int                fd);
static void cvs_process_push                  (CvsProcess            *cvs_process);
static void cvs_process_pop                   (void);
static void cvs_process_recv_message          (CvsProcess *p,
											   gint               id);
static void cvs_process_handle_message        (WireMessage       *msg);
static void cvs_process_close (CvsProcess *cvs_process, int kill_it);
static void cvs_process_destroy (CvsProcess *cvs_process);
static CvsProcess* cvs_process_new (char *name, int argc, char **argv);

static std::deque<CvsProcess *> cvs_process_stack;
static std::vector<CvsProcess *> open_cvs_process;
static CvsProcess *current_cvs_process = NULL;
static int current_write_buffer_index = 0;
static char *current_write_buffer = NULL;

static char process_write_buffer[WRITE_BUFFER_SIZE];	/* Buffer for writing, only for the process */

static void
_gp_quit_read (int fd, WireMessage *msg)
{
	GPT_QUIT *t = (GPT_QUIT *)malloc(sizeof(GPT_QUIT));
	if(t == 0L)
		return;

	if (!wire_read_int32 (fd, (guint32 *)&t->code, 1))
		return;

	msg->data = t;
}

static void
_gp_quit_write (int fd, WireMessage *msg)
{
	GPT_QUIT *t = (GPT_QUIT *)msg->data;
	if (!wire_write_int32 (fd, (guint32 *)&t->code, 1))
		return;
}

static void
_gp_quit_destroy (WireMessage *msg)
{
	free(msg->data);
}

static void
_gp_getenv_read (int fd, WireMessage *msg)
{
	GPT_GETENV *t = (GPT_GETENV *)malloc(sizeof(GPT_GETENV));
	if(t == 0L)
		return;

	if (!wire_read_int8 (fd, &t->empty, 1))
		return;
	if (!wire_read_string (fd, &t->str, 1))
		return;

	msg->data = t;
}

static void
_gp_getenv_write (int fd, WireMessage *msg)
{
	GPT_GETENV *t = (GPT_GETENV *)msg->data;
	if (!wire_write_int8 (fd, &t->empty, 1))
		return;
	if (!wire_write_string (fd, &t->str, 1))
		return;
}

static void
_gp_getenv_destroy (WireMessage *msg)
{
	GPT_GETENV *t = (GPT_GETENV *)msg->data;
	free(t->str);
	free(t);
}

static void
_gp_console_read (int fd, WireMessage *msg)
{
	GPT_CONSOLE *t = (GPT_CONSOLE *)malloc(sizeof(GPT_CONSOLE));
	if(t == 0L)
		return;

	if (!wire_read_int8 (fd, &t->isStderr, 1))
		return;
	if (!wire_read_int32 (fd, &t->len, 1))
		return;
 	if (!wire_read_string (fd, &t->str, 1))
		return;

	msg->data = t;
}

static void
_gp_console_write (int fd, WireMessage *msg)
{
	GPT_CONSOLE *t = (GPT_CONSOLE *)msg->data;
	if (!wire_write_int8 (fd, &t->isStderr, 1))
		return;
	if (!wire_write_int32 (fd, &t->len, 1))
		return;
	if (!wire_write_string (fd, &t->str, 1))
		return;
}

static void
_gp_console_destroy (WireMessage *msg)
{
	GPT_CONSOLE *t = (GPT_CONSOLE *)msg->data;
	free(t->str);
	free(t);
}

void
gp_init ()
{
	wire_register (GP_QUIT,
				   _gp_quit_read,
				   _gp_quit_write,
				   _gp_quit_destroy);
	wire_register (GP_GETENV,
				   _gp_getenv_read,
				   _gp_getenv_write,
				   _gp_getenv_destroy);
	wire_register (GP_CONSOLE,
				   _gp_console_read,
				   _gp_console_write,
				   _gp_console_destroy);
}

int
gp_quit_write (int fd, int code)
{
	WireMessage msg;
	GPT_QUIT *t = (GPT_QUIT *)malloc(sizeof(GPT_QUIT));
	
	msg.type = GP_QUIT;
	msg.data = t;

	t->code = code;

	if (!wire_write_msg (fd, &msg))
		return FALSE;
	if (!wire_flush (fd))
		return FALSE;
	return TRUE;
}

int
gp_getenv_write (int fd, const char *env)
{
	WireMessage msg;
	GPT_GETENV *t = (GPT_GETENV *)malloc(sizeof(GPT_GETENV));

	msg.type = GP_GETENV;
	msg.data = t;

	t->empty = env == 0L;
	t->str = strdup(env == 0L ? "" : env);

	if (!wire_write_msg (fd, &msg))
		return FALSE;
	if (!wire_flush (fd))
		return FALSE;
	return TRUE;
}

char *
gp_getenv_read (int fd)
{
	WireMessage msg;
	char *res;

	memset (&msg, 0, sizeof (WireMessage));
	if (!wire_read_msg (fd, &msg) || msg.type != GP_GETENV)
	{
		fprintf(stderr, "cvsgui protocol error !\n");
		exit(-1);
	}
	GPT_GETENV *t = (GPT_GETENV *)msg.data;
	res = t->empty ? 0L : strdup(t->str);
	wire_destroy (&msg);

	return res;
}

int  gp_console_write (int fd, const char *str, int len, int isStderr)
{
	WireMessage msg;
	GPT_CONSOLE *t = (GPT_CONSOLE *)malloc(sizeof(GPT_CONSOLE));

	msg.type = GP_CONSOLE;
	msg.data = t;

	t->isStderr = isStderr;
	t->len = len;
	t->str = (char *)malloc(len * sizeof(char));
	memcpy(t->str, str, len * sizeof(char));

	if (!wire_write_msg (fd, &msg))
		return FALSE;
	if (!wire_flush (fd))
		return FALSE;
	return TRUE;
}

void
cvs_process_init ()
{
	/* initialize the gimp protocol library and set the read and
	 *  write handlers.
	 */
	gp_init ();
	wire_set_writer (cvs_process_write);
	wire_set_flusher (cvs_process_flush);
}

static CvsProcess*
cvs_process_new (char *path, int argc, char **argv)
{
	CvsProcess *cvs_process;

	cvs_process_init();

	cvs_process = (CvsProcess *)malloc(sizeof(CvsProcess));
	if(cvs_process == 0L)
		return 0L;

	cvs_process->open = FALSE;
	cvs_process->destroy = FALSE;
	cvs_process->synchronous = FALSE;
	cvs_process->pid = 0;
	cvs_process->callbacks = 0L;
	cvs_process->argc = argc + 4;
	cvs_process->args = (char **)malloc((cvs_process->argc + 1) * sizeof(char *));
	cvs_process->args[0] = strdup (path);
	cvs_process->args[1] = strdup ("-cvsgui");
	cvs_process->args[2] = (char *)malloc(16 * sizeof(char));
	cvs_process->args[3] = (char *)malloc(16 * sizeof(char));
	for(int i = 0; i < argc; i++)
	{
		cvs_process->args[4 + i] = strdup(argv[i]);
	}
	cvs_process->args[cvs_process->argc] = 0L;
	cvs_process->my_read = 0;
	cvs_process->my_write = 0;
	cvs_process->his_read = 0;
	cvs_process->his_write = 0;
	cvs_process->write_buffer_index = 0;

	return cvs_process;
}

static void
cvs_process_destroy (CvsProcess *cvs_process)
{
	if (cvs_process)
    {
		cvs_process_close (cvs_process, TRUE);

		for(int i = 0; i < 4; i++)
			if (cvs_process->args[i])
				free (cvs_process->args[i]);

		if (cvs_process == current_cvs_process)
			cvs_process_pop ();

		if (!cvs_process->destroy)
			free (cvs_process);
    }
}

CvsProcess *
cvs_process_open (char *name, int argc, char **argv, CvsProcessCallbacks *callbacks)
{
	int my_read[2];
	int my_write[2];
	CvsProcess *cvs_process;

	cvs_process = cvs_process_new(name, argc, argv);

	if (!cvs_process)
		return 0L;

	cvs_process->callbacks = callbacks;

	/* Open two pipes. (Bidirectional communication).
		 */
	if ((pipe (my_read) == -1) || (pipe (my_write) == -1))
	{
		fprintf(stderr, "unable to open pipe\n");
		cvs_process_destroy (cvs_process);
		return 0L;
	}

	cvs_process->my_read = my_read[0];
	cvs_process->my_write = my_write[1];
	cvs_process->his_read = my_write[0];
	cvs_process->his_write = my_read[1];

	/* Remember the file descriptors for the pipes.
		 */
	sprintf (cvs_process->args[2], "%d", cvs_process->his_read);
	sprintf (cvs_process->args[3], "%d", cvs_process->his_write);

		/* Fork another process. We'll remember the process id
		 *  so that we can later use it to kill the filter if
		 *  necessary.
		 */
	cvs_process->pid = fork ();

	if (cvs_process->pid == 0)
	{
		close(cvs_process->my_read);
		close(cvs_process->my_write);
		/* Execute the filter. The "_exit" call should never
		 *  be reached, unless some strange error condition
		 *  exists.
		 */
		execvp (cvs_process->args[0], cvs_process->args);
		_exit (1);
	}
	else if (cvs_process->pid == -1)
	{
		cvs_process_destroy (cvs_process);
		return 0L;
	}

	close(cvs_process->his_read);  cvs_process->his_read  = -1;
	close(cvs_process->his_write); cvs_process->his_write = -1;

	if (!cvs_process->synchronous)
	{
		open_cvs_process.push_back(cvs_process);
	}

	cvs_process->open = TRUE;
	return cvs_process;
}

static void
cvs_process_close (CvsProcess *cvs_process,
			   int     kill_it)
{
	int status;

	if (cvs_process && cvs_process->open)
    {
		cvs_process->open = FALSE;

		/* If necessary, kill the filter.
		 */
		if (kill_it && cvs_process->pid)
			status = kill (cvs_process->pid, SIGKILL);

		/* Wait for the process to exit. This will happen
		 *  immediately if it was just killed.
		 */
		if (cvs_process->pid)
			waitpid (cvs_process->pid, &status, 0);

		/* Close the pipes.
		 */
		if (cvs_process->my_read)
			close (cvs_process->my_read);
		if (cvs_process->my_write)
			close (cvs_process->my_write);
		if (cvs_process->his_read)
			close (cvs_process->his_read);
		if (cvs_process->his_write)
			close (cvs_process->his_write);

		wire_clear_error();

		/* Set the fields to null values.
		 */
		cvs_process->pid = 0;
		cvs_process->my_read = 0;
		cvs_process->my_write = 0;
		cvs_process->his_read = 0;
		cvs_process->his_write = 0;

		cvs_process->synchronous = FALSE;

		std::vector<CvsProcess *>::iterator i = std::find(open_cvs_process.begin(), open_cvs_process.end(), cvs_process);
		if(i != open_cvs_process.end())
			open_cvs_process.erase(i);
    }
}

int
cvs_process_is_active(CvsProcess *cvs_process)
{
	std::vector<CvsProcess *>::iterator i = 
		std::find(open_cvs_process.begin(), open_cvs_process.end(), cvs_process);
	return i != open_cvs_process.end() ? 1 : 0;
}

void
cvs_process_kill(CvsProcess *cvs_process)
{
	if(cvs_process_is_active(cvs_process))
	{
		cvs_process_destroy(cvs_process);
	}
}

void
cvs_process_give_time(void)
{
	fd_set rset;
	int ready;
	int maxfd = 0;
	int fd;
	struct timeval tv;

	FD_ZERO (&rset);
	
	std::vector<CvsProcess *>::iterator i;
	
	for (i = open_cvs_process.begin(); i != open_cvs_process.end(); ++i)
	{
		fd = (*i)->my_read;

		FD_SET (fd, &rset);
		if (fd > maxfd)
			maxfd = fd;
	}
	
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	ready = select (maxfd + 1, &rset, 0L, 0L, &tv);

	std::vector<CvsProcess *> toFire;
	if (ready > 0)
	{
		for (i = open_cvs_process.begin(); i != open_cvs_process.end(); ++i)
		{
			fd = (*i)->my_read;
			
			if (FD_ISSET (fd, &rset))
				toFire.push_back(*i);
		}
	}

	for (i = toFire.begin(); i != toFire.end(); ++i)
	{
		fd = (*i)->my_read;
		
		if (FD_ISSET (fd, &rset))
			cvs_process_recv_message (*i, fd);
	}
}

static void
cvs_process_recv_message (CvsProcess *p,
						  gint              id)
{
	WireMessage msg;

	cvs_process_push (p);

	memset (&msg, 0, sizeof (WireMessage));
	if (!wire_read_msg (p->my_read, &msg))
		cvs_process_close (p, TRUE);
	else
    {
		cvs_process_handle_message (&msg);
		wire_destroy (&msg);
    }

	if (!current_cvs_process->open)
		cvs_process_destroy (current_cvs_process);
	else
		cvs_process_pop ();
}

static void
cvs_process_handle_message (WireMessage *msg)
{
	switch (msg->type)
    {
    case GP_QUIT:
	{
		GPT_QUIT *t = (GPT_QUIT *)msg->data;
		current_cvs_process->callbacks->exit(t->code);
		cvs_process_close (current_cvs_process, FALSE);
		break;
	}
    case GP_GETENV:
	{
		GPT_GETENV *t = (GPT_GETENV *)msg->data;
		cvs_process_push (current_cvs_process);
		gp_getenv_write (current_cvs_process->my_write,
						 current_cvs_process->callbacks->getenv(t->str));
		cvs_process_pop ();
		break;
	}
    case GP_CONSOLE:
	{
		GPT_CONSOLE *t = (GPT_CONSOLE *)msg->data;
		if(t->isStderr)
			current_cvs_process->callbacks->consoleerr(t->str, t->len);
		else
			current_cvs_process->callbacks->consoleout(t->str, t->len);
		break;
	}
    }
}

static int
cvs_process_write (int     fd,
			   guint8 *buf,
			   gulong  count)
{
	gulong bytes;

	if(current_write_buffer == 0L)
		current_write_buffer = process_write_buffer;

	while (count > 0)
    {
		if ((current_write_buffer_index + count) >= WRITE_BUFFER_SIZE)
		{
			bytes = WRITE_BUFFER_SIZE - current_write_buffer_index;
			memcpy (&current_write_buffer[current_write_buffer_index], buf, bytes);
			current_write_buffer_index += bytes;
			if (!wire_flush (fd))
				return FALSE;
		}
		else
		{
			bytes = count;
			memcpy (&current_write_buffer[current_write_buffer_index], buf, bytes);
			current_write_buffer_index += bytes;
		}

		buf += bytes;
		count -= bytes;
    }

	return TRUE;
}

static int
cvs_process_flush (int fd)
{
	int count;
	int bytes;

	if (current_write_buffer_index > 0)
    {
		count = 0;
		while (count != current_write_buffer_index)
        {
			do {
				bytes = write (fd, &current_write_buffer[count],
							   (current_write_buffer_index - count));
			} while ((bytes == -1) && (errno == EAGAIN));

			if (bytes == -1)
				return FALSE;

			count += bytes;
        }

		current_write_buffer_index = 0;
    }

	return TRUE;
}

static void
cvs_process_push (CvsProcess *cvs_process)
{
	if (cvs_process)
    {
		current_cvs_process = cvs_process;
		cvs_process_stack.push_back(current_cvs_process);

		current_write_buffer_index = current_cvs_process->write_buffer_index;
		current_write_buffer = current_cvs_process->write_buffer;
    }
	else
    {
		current_write_buffer_index = 0;
		current_write_buffer = NULL;
    }
}

static void
cvs_process_pop ()
{
	if (current_cvs_process)
    {
		current_cvs_process->write_buffer_index = current_write_buffer_index;

		cvs_process_stack.pop_back();
    }

	if (!cvs_process_stack.empty())
    {
		current_cvs_process = cvs_process_stack.back();
		current_write_buffer_index = current_cvs_process->write_buffer_index;
		current_write_buffer = current_cvs_process->write_buffer;
    }
	else
    {
		current_cvs_process = NULL;
		current_write_buffer_index = 0;
		current_write_buffer = NULL;
    }
}
