/*
 * Copyright (c) 1995, 1996, 1997, 1998, 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Copyright (C) 1999  Carnegie Mellon University
 * Extensions and modifications to Arla code base for Coda by
 * Philip A. Nelson
 */

/* Started with: xfs_vnodeops.c,v 1.22 1999/06/16 20:24:07 assar Exp */
/* $Id: coda_vnodeops.c,v 1.7 2000/03/07 20:30:50 phil Exp $ */

/*
 * CODA operations.
 */

#include <coda.h>
#include <coda_sys.h>
#include <coda_dev.h>
#include <coda_common.h>
#include <coda_fs.h>
#include <coda_debug.h>
#include <coda_upcalls.h>
#include <sys/fcntl.h>
#include <sys/flock.h>
#include <sys/fs_subr.h>
#include <sys/fs/ufs_inode.h>

/* static struct cred root_cred = {1,0,0,0,0,0,0,0,{0}}; */

/* Should come from a .h file ???? */
#define _VICEIOCTL(id)  (_IOW('V', id, struct ViceIoctl))

static int
coda_open(struct vnode **vpp,
	  int flag,
	  struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(*vpp);
  struct coda_node *cn = VNODE_TO_CNODE(*vpp);
  struct vnode *cachevp;
  int error = 0;

#ifdef CAN_USE_DEV_INODE
  struct vfs *cachefsp;
  struct ufid ufsfid;
  struct inode *inop;
  dev_t dev;
  ino_t inode;
#endif
  
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_open: vp=0x%x flag=0x%x", *vpp, flag));

  /* venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Check for open of control file. */
  if (*vpp == coda_ctlvp) {
    if (flag & (FWRITE | O_TRUNC | O_CREAT | O_EXCL))
      return EACCES;
    else
      return 0;
  }
  
  /* Check cache? XXXXX */

#ifdef CAN_USE_DEV_INODE
  /* This currently doesn't work. */
  error = coda_venus_open (codap->minor, cred, flag, &cn->VFid, &dev, &inode);
  if (error)
    return error;
 
  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: venus returned %d/%d",
			dev, inode));

  cachefsp = vfs_devsearch(dev);
  if (!cachefsp) {
    cmn_err (CE_WARN, "coda_open: cachefs not found");
    return ENXIO;
  }

  error = ufs_iget(cachefsp, inode, &inop, cred);
  if (error) {
    cmn_err (CE_WARN, "coda_open: ufs_iget error=%d", error);
    return ENXIO;
  }
    
  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: cachefsp (0x%x) ino/gen (%d/%d)",
			 cachefsp, inode, inop->i_gen));
  
  ufsfid.ufid_len = sizeof(ufsfid);
  ufsfid.ufid_ino = inode;
  ufsfid.ufid_gen = inop->i_gen;

  error = VFS_VGET(cachefsp, &cachevp, (struct fid *)&ufsfid);

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: cachefsp (0x%x) cachevp (0x%x) "
			"error (%d)", cachefsp, cachevp, error));
#else
  {
    char name[CODA_MAXNAMLEN];
    uid_t old_uid;

    error = coda_venus_open_by_path (codap->minor, cred, flag, &cn->VFid,
				     name);

    if (error) {
      CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: venus returned error %d",
			     error));
      return error;
    }

    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: venus returned %s", name));

    /* Lookup the name as root! (Is this right?) */
    old_uid = cred->cr_uid;
    cred->cr_uid = 0;
    error = lookupname(name, UIO_SYSSPACE, FOLLOW, 0, &cachevp);
    cred->cr_uid = old_uid;
    
    if (error) {
      CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: lookupname error %d",
			     error));
      return error;
    }

    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: lookup of %s succesful",
			   name));
  }
#endif


  if (!cachevp) {
    cmn_err (CE_WARN, "coda_open: cachevp not found");
    return ENXIO;
  }

  /* Attach to cnode and do the open. */
  if (cn->ovp == NULL) {
    cn->ovp = cachevp;
  } else {
    if (cn->ovp != cachevp) {
      cmn_err (CE_WARN, "coda_open: cp->c_ovp != cachevp");
      return ENXIO;
    }
  }
  cn->ocount++;
  
  /* Flush the attribute cached if writing the file. */
  if (flag & FWRITE) {
    cn->owrite++;
  }

#ifdef CAN_USE_DEV_INODE
  /* Save device/inode */
  cn->device = dev;
  cn->inode  = inode;
#endif

  /* Open the cache file. */
  error = VOP_OPEN(&cachevp, flag, cred); 

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_open: after VOP_OPEN  error=%d",
			 error));

  return error;
} 


static int
coda_close(struct vnode *vp,
	   int flag,
	   int count,
	   offset_t offset,
	   struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;
  int verror = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_close: vp=0x%x flag=0x%x count=%d "
		      "offset=0x%x", vp, flag, count, offset));
  
  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;
 
  /* Control file ... */
  if (vp == coda_ctlvp) {
    return 0;
  }

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_close: ovp is NULL!");
    return EINVAL;
  }

  /* What does count mean?   It appears not to match an open... */
  if (count > 1) {
    error = VOP_CLOSE(cn->ovp, flag, count, offset, cred);
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_close: count > 1"));
    return error;
  }

  error = VOP_CLOSE(cn->ovp, flag, count, offset, cred);
  VN_RELE(cn->ovp);

  if (cn->ocount == 0)
    cmn_err (CE_WARN, "coda_close: cn->ocount would go negative!"); 
  else
    cn->ocount--;

  if (!cn->ocount)
    cn->ovp = NULL;
  if (flag & FWRITE) 
    cn->owrite--;

  verror = coda_venus_close (codap->minor, cred, &cn->VFid, flag);
  
  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_close: error=%d", error));

  return error;
}

static int
coda_read(struct vnode *vp,
	  struct uio *uiop,
	  int ioflag,
	  struct cred *cred)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;
  
  CODADEB(CDEBVNOPSRW, (CE_NOTE, "coda_read: vp=0x%x", vp));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (vp == coda_ctlvp)
    return EINVAL;

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_read: ovp is NULL!");
    return EINVAL;
  }
  
  error = VOP_READ(cn->ovp, uiop, ioflag, cred);

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_read: error=%d", error));
    
  return error;
}

static int
coda_write(struct vnode *vp,
	   struct uio *uiop,
	   int ioflag,
	   struct cred *cred)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;
  
  CODADEB(CDEBVNOPSRW, (CE_NOTE, "coda_write: vp=0x%x", vp));
  
  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  if (vp->v_type != VREG)
    return EISDIR;
  
  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_write: ovp is NULL!");
    return EINVAL;
  }
  
  error = VOP_WRITE(cn->ovp, uiop, ioflag, cred);

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_write: error=%d", error));

  return error;  
}

static int
coda_ioctl(struct vnode *vp,
	   int enccmd,
	   intptr_t arg,
	   int flag,
	   struct cred *cred,
	   int *result)
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  /* struct coda_node *cn = VNODE_TO_CNODE(vp); */
  struct PioctlData *ioargp = (struct PioctlData *)arg;
  struct PioctlData ioarg;
  struct vnode *pathvp;
  struct coda_node *pathcn;
  int error = 0;

  int cmd  = (enccmd & 0xffff) - ('V'<<8);  /* XXX was 0xff only */
  int cchr = (enccmd >> 8) & 0xff;
  int size = (enccmd >> 16) & 0xff;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_ioctl: cmd = %d, chr=%c, size=%d",
		      cmd, cchr, size));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* It must be the .CONTROL file. */
  if (vp != coda_ctlvp) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: not control file"));
    return EOPNOTSUPP;
  }

  /* Check size.... */
  if (size != sizeof(struct PioctlData)) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: size is wrong"));
    return EINVAL;
  }

  /* Get the PioctlData! */
  error = copyin (ioargp, &ioarg, sizeof(struct PioctlData));
  if (error) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: ioargp copy error"));
    return error;
  }

  /* Get the vnode for the name. */
  error = lookupname((char *)ioarg.path, UIO_USERSPACE,
		     (ioarg.follow ? FOLLOW : NO_FOLLOW), 0, &pathvp);
  if (error) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: lookupname error"));
    return error;
  }

  /* Is the path in the request a coda file? */
  if (pathvp->v_op != &coda_vnodeops) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: path not a coda path"));
    /* VN_RELE(pathvp); XXXXX Needed? */
    return EINVAL;
  }
  
  if (ioarg.vi.in_size > VC_MAXDATASIZE
      || ioarg.vi.out_size > VC_MAXDATASIZE) {
    CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: bad in_size or out_size"));
    /*  VN_RELE(pathvp); XXXXX needed? */
    return(EINVAL);
  }

  /* Call venus. */    
  pathcn = VNODE_TO_CNODE(pathvp);
  error = coda_venus_ioctl (codap->minor, cred, &pathcn->VFid, _VICEIOCTL(cmd),
			   &ioarg);

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_ioctl: returning error=%d", error));

  return error;
}

static int
coda_setfl(struct vnode *vp,
	   int oflags,
	   int nflags,
	   struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_setfl"));
  return fs_setfl (vp, oflags, nflags, cred);
}

static int
coda_getattr(struct vnode *vp,
	     struct vattr *vap,
	     int flags,
	     struct cred *cred)     
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_getattr flags=0x%x", flags));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (vp == coda_ctlvp) {
    /* Make up a fake vattr! */
    bzero (vap, sizeof(struct vattr));
    vap->va_mask = (AT_TYPE | AT_MODE | AT_NLINK | AT_FSID); 
    vap->va_type = VREG;
    vap->va_mode = 0444;
    vap->va_nlink = 1;
    vap->va_fsid = codap->rdev;
    return 0;
  }

  /* Is the attr in the cnode valid? */
#if 0
  if (cn->flags & C_VATTR) {
    *vap = cn->attr;
    return 0;
  }
#endif

  error = coda_venus_getattr (codap->minor, cred, &cn->VFid, vap);

#if 0
  if (!error) {
    cn->attr = *vap;
    cn->flags |= C_VATTR;
  }
#endif  

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_getattr error=%d", error));
  return error;
}

static int
coda_setattr(struct vnode *vp,
	     struct vattr *vap,
	     int flags,
	     struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;
  
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_setattr flags=0x%x", flags));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (vp == coda_ctlvp)
    return EACCES;

  error = coda_venus_setattr (codap->minor, cred, &cn->VFid, vap);

  if (!error)
    cn->flags &= ~C_VATTR;

  return error;
}


static int
coda_access(struct vnode *vp,
	    int mode,
	    int flags,
	    struct cred *cred)     
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_access mode = 0%o", mode));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (vp == coda_ctlvp)
    return(((mode & VREAD) && !(mode & (VWRITE | VEXEC))) ? 0 : EACCES);

  /* Check in cache? */

  error = coda_venus_access (codap->minor, cred, &cn->VFid, mode);

  return error;
}

static int
coda_lookup(struct vnode *dvp,
	    char *nm,
	    struct vnode **vpp,
	    struct pathname *pnp,
	    int flags,
	    struct vnode *rdir,
	    struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  struct coda_node *newcn;
  vtype_t vtype;
  ViceFid newVFid;
  int dontcache;
  int error = 0;
  
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_lookup: nm=(%s) len=%d flags=%d",
		      nm, strlen(nm), flags));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (dvp == CNODE_TO_VNODE(codap->root) && strcmp(nm,".CONTROL") == 0) {
    *vpp = coda_ctlvp;
    VN_HOLD(*vpp);
    return 0;
  }

  /* Lookup in cache */

  /* Call venus. */

  error = coda_venus_lookup (codap->minor, cred, &cn->VFid, nm, &newVFid,
			     &vtype, &dontcache);

  if (!error) {
#if 0 
    /* LOOKUP_DIR flag?? XXXXX what is this for???? */
    if (flags & LOOKUP_DIR && vtype != VDIR) {
      *vpp = NULL;
      return ENOTDIR;
    }
#endif
    newcn = new_coda_node (codap, &newVFid, vtype);
    *vpp = CNODE_TO_VNODE(newcn);
    VN_HOLD(*vpp);
    /* Enter name/newcn into cache. */
  } else {
    *vpp = NULL;
  }

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_lookup() = %d", error));
  return error;
}

static int
coda_create(struct vnode *dvp,
	    char *nm,
	    struct vattr *va,
	    vcexcl_t exclusive,
	    int mode,
	    struct vnode **vpp,
	    struct cred *cred
#ifdef _LARGEFILE64_SOURCE
	    ,int file_awareness  /* Solaris 2.6+ */
#endif
	   )     
{
  struct coda *codap = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  struct coda_node *newcn;
  ViceFid newVFid;
  int error = 0;
  
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_create: mode =0x%x",mode));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (dvp == CNODE_TO_VNODE(codap->root) && strcmp(nm,".CONTROL") == 0) {
    return EACCES;
  }

  /* Lookup in cache? */

  /* Call venus. */

  error = coda_venus_create (codap->minor, cred, &cn->VFid, nm, mode, va,
			    exclusive, &newVFid);

  if (!error) {
      newcn = new_coda_node (codap, &newVFid, va->va_type);
      *vpp = CNODE_TO_VNODE(newcn);
      VN_HOLD(*vpp);
      /* Enter name/newcn into cache? */
      newcn->create = 1;
  } else {
      *vpp = NULL;
  }

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_create() = %d", error));

  return error;
}

static int
coda_remove(struct vnode *dvp,
	    char *nm,
	    struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  int error;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_remove: %s", nm));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  error = coda_venus_remove (codap->minor, cred, &cn->VFid, nm);

  return error;
}

static int
coda_link(struct vnode *tdvp,
	  struct vnode *vp,
	  char *tnm,
	  struct cred *cred)     
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cdir = VNODE_TO_CNODE(tdvp);
  struct coda_node *cfile = VNODE_TO_CNODE(vp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_link: (%s)", tnm));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (tdvp == CNODE_TO_VNODE(codap->root) && strcmp(tnm,".CONTROL") == 0) {
      return EACCES;
  }
  error = coda_venus_link (codap->minor, cred, &cfile->VFid, &cdir->VFid,
			   tnm);

  return error;
}

static int
coda_rename(struct vnode *sdvp,
	    char *onm,
	    struct vnode *tdvp,
	    char *nnm,
	    struct cred *cred)     
{
  struct coda *codap = CODA_FROM_VNODE(sdvp);
  struct coda_node *srcn = VNODE_TO_CNODE(sdvp);
  struct coda_node *dstn = VNODE_TO_CNODE(tdvp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_rename"));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (tdvp == CNODE_TO_VNODE(codap->root) && strcmp(nnm,".CONTROL") == 0) {
    return EACCES;
  }

  error = coda_venus_rename (codap->minor, cred, &srcn->VFid, onm,
			     &dstn->VFid, nnm);
 
  return error;
}

static int
coda_mkdir(struct vnode *dvp,
	   char *nm,
	   struct vattr *va,
	   struct vnode **vpp,
	   struct cred *cred)     
{
  struct coda *codap = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  struct coda_node *newcn;
  ViceFid newVFid;
  int error = 0;
  
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_mkdir"));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (dvp == CNODE_TO_VNODE(codap->root) && strcmp(nm,".CONTROL") == 0) {
    return EACCES;
  }

  /* Cache??? */

  error = coda_venus_mkdir (codap->minor, cred, &cn->VFid, nm, va, &newVFid);

  if (!error) {
      newcn = new_coda_node (codap, &newVFid, va->va_type);
      *vpp = CNODE_TO_VNODE(newcn);
      VN_HOLD(*vpp);
      /* Enter name/newcn into cache? */
  } else {
      *vpp = NULL;
  }
    
  return error;
}

static int
coda_rmdir(struct vnode *dvp,
	   char *nm,
	   struct vnode *foo,
	   struct cred *cred)     
{
  struct coda *codap  = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  int error = 0;

  struct coda_node *cn2 = VNODE_TO_CNODE(foo);

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_rmdir: VFid (%d.%d.%d)",
		      cn2->VFid.Volume, cn2->VFid.Vnode, cn2->VFid.Unique));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  error = coda_venus_rmdir (codap->minor, cred, &cn->VFid, nm);

  return error;
}

static int
coda_readdir(struct vnode *vp,
	    struct uio *uiop,
	    struct cred *cred,
	    int *eofp)		/* XXX */
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_readdir: vp=0x%x, uiop=0x%x, "
		      "cred=0x%x, eofp=0x%x", vp, uiop, cred, eofp));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Control file? */
  if (vp == coda_ctlvp) {
    return EACCES;
  }

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_readdir: ovp is NULL!");
    return EINVAL;
  }

  error = VOP_READDIR(cn->ovp, uiop, cred, eofp); 

  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_readdir: error = %d", error));
  return error;
}

static int
coda_symlink(struct vnode *dvp,
	     char *lnm,
	     struct vattr *tva,
	     char *tnm,
	     struct cred *cred)
{
  struct coda *codap  = CODA_FROM_VNODE(dvp);
  struct coda_node *cn = VNODE_TO_CNODE(dvp);
  int error = 0;
 
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_symlink lnm (%s) tnm (%s)", lnm, tnm));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  error = coda_venus_symlink (codap->minor, cred, &cn->VFid, tnm, tva, lnm);
   
  return error;
}

static int
coda_readlink(struct vnode *vp,
	      struct uio *uiop,
	      struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_readlink"));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  error = coda_venus_readlink (codap->minor, cred, &cn->VFid, uiop);

  return error;
}

static int
coda_fsync(struct vnode *vp,
	   int syncflag,
	   struct cred *cred)
{
  struct coda *codap = CODA_FROM_VNODE(vp);
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error = 0;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_fsync"));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  error = coda_venus_fsync (codap->minor, cred, &cn->VFid);

  if (!error && cn->ovp != NULL)
    VOP_FSYNC (cn->ovp, syncflag, cred);

  return error;
}

static void
coda_inactive(struct vnode *vp,
	      struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_inactive: 0x%x", (int)vp));

  free_coda_node(VNODE_TO_CNODE(vp));
}

static int
coda_fid(struct vnode *vp,
	 struct fid *fid)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_fid"));
  return ENOSYS;
}

static void
coda_rwlock(struct vnode *vp,
	    int write_lock)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  CODADEB(CDEBVNOPSRW, (CE_NOTE, "coda_rwlock"));

  /* Venus running? */
  if (!coda_dev_opened)
      return;

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_rwlock: ovp is NULL!");
    return;
  }

  VOP_RWLOCK(cn->ovp, write_lock);

}

static void
coda_rwunlock(struct vnode *vp,
	      int write_lock)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  CODADEB(CDEBVNOPSRW, (CE_NOTE, "coda_rwunlock"));

  /* Venus running? */
  if (!coda_dev_opened)
      return;

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_rwunlock: ovp is NULL!");
    return;
  }

  VOP_RWUNLOCK(cn->ovp, write_lock);
}

static int
coda_seek(struct vnode *vp,
	  offset_t offset,
	  offset_t *roffset)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  int error;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_seek: vp=0x%x offset=0x%x", vp, offset));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  if (cn->ovp == NULL) {
    cmn_err (CE_WARN, "coda_seek: ovp is NULL!");
    return EINVAL;
  }

  error = VOP_SEEK(cn->ovp, offset, roffset);

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_seek: error=%d", error));
  return error;
}

static int
coda_cmp(struct vnode *vp1, struct vnode *vp2)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_cmp"));
  return vp1 == vp2;
}

static int
coda_frlock(struct vnode *vp,
	    int foo,
#ifdef _LARGEFILE64_SOURCE
	    struct flock64 *fl,
#else
	    struct flock *fl,
#endif
	    int bar,
	    offset_t off,
	    struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_frlock"));
  return ENOSYS;
}

static int
coda_space(struct vnode *vp,
	   int cmd,
#ifdef _LARGEFILE64_SOURCE
	   struct flock64 *fl,
#else
	   struct flock *fl,
#endif
	   int flag,
	   offset_t offset,
	   struct cred *cred)
{
  int error = 0;
  struct vattr attr;

  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_space"));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  if (cmd != F_FREESP) {
      error = EINVAL;
      goto done;
  }
      
  error = convoff (vp, fl, 0, offset);
  if (error)
      goto done;

  if (fl->l_len != 0) {
      error = EINVAL;
      goto done;
  }

  attr.va_mask = AT_SIZE;
  attr.va_size = fl->l_start;
  error = coda_setattr (vp, &attr, 0, cred);
done:
  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_space: %d", error));
  return error;
}

static int
coda_realvp(struct vnode *vp,
	    struct vnode **vpp)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_realvp"));
  return ENOSYS;
}

static int
coda_getpage(struct vnode *vp,
	     offset_t off,
	     size_t len,
	     uint_t *protp,
	     struct page *pl[],
	     size_t plsz,
	     struct seg *seg,
	     caddr_t addr,
	     enum seg_rw rw,
	     struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_getpage"));
  return ENOSYS;
}

static int
coda_putpage(struct vnode *vp,
	     offset_t off,
	     size_t len,
	     int flags,
	     struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_putpage"));
  return ENOSYS;
}

static int
coda_map(struct vnode *vp,
	 offset_t off,
	 struct as *as,
	 caddr_t *addrp,
	 size_t len,
	 uchar_t prot,
	 uchar_t maxprot,
	 uint_t flags,
	 struct cred *cred)
{
  struct coda_node *cn = VNODE_TO_CNODE(vp);
  struct vattr ta;
  int error = 0;


  CODADEB(CDEBVNOPS,
	 (CE_NOTE, "coda_map(0x%x, 0x%lx, 0x%x, 0x%x, 0x%lx, 0x%x, 0x%x, "
	  "0x%x, 0x%x)",
	  (int) vp, (unsigned long)off,
	  (int) as, (int) addrp, (unsigned long)len, prot, maxprot,
	  flags, (int) cred));

  /* Venus running? */
  if (!coda_dev_opened)
      return EPERM;

  /* Check to make sure it is open! */
  if (cn->ovp == NULL) {
    cmn_err (CE_NOTE, "coda_map: ovp is NULL!");
    return EINVAL;
  }

/* How to do this in coda???

  if ((prot & PROT_WRITE) && (flags & MAP_SHARED))
    error = coda_data_valid(vp, cred, CODA_DATA_W);
  else
    error = coda_data_valid(vp, cred, CODA_DATA_R);
*/

  /* is the attr valid? */
  if (!(cn->flags & C_VATTR))
    error = coda_getattr (vp, &ta, flags, cred);

  if (error)
    goto out;

  if (off + len > cn->attr.va_size) {
    cmn_err (CE_NOTE, "coda_map: off+len (0x%x) > va_size (0x%x)!",
	     (unsigned long)off+len, cn->attr.va_size);
    error = EINVAL;
#if 0
  } else if ((prot & PROT_WRITE) && (flags & MAP_SHARED)) {
    error = EROFS;		/* XXX This is currently not supported */
#endif
  } else {
    VOP_RWLOCK(cn->ovp, 1);
    error = VOP_MAP(cn->ovp, off, as, addrp, len, prot, maxprot, flags, cred);
    /* XXX Patch vnode so that we can intercept get/putpage and inactive. */
    VOP_RWUNLOCK(cn->ovp, 1);
  }

out:
  CODADEB(CDEBVNOPSDET, (CE_NOTE, "coda_map: %d", error));
  return error;
}

static int
coda_addmap(struct vnode *vp,
	    offset_t off,
	    struct as *as,
	    caddr_t addr,
	    size_t len,
	    uchar_t prot,
	    uchar_t maxprot,
	    uint_t flags,
	    struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_addmap"));
  return ENOSYS;

}

static int
coda_delmap(struct vnode *vp,
	    offset_t off,
	    struct as *as,
	    caddr_t addr,
	    size_t len,
	    uint_t prot,
	    uint_t maxprot,
	    uint_t flags,
	    struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_delmap"));
  return ENOSYS;
  
}


static int
coda_poll(struct vnode *vp,
	  short events,
	  int anyyet,
	  short *revents,
	  struct pollhead **ph)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_poll"));
  return fs_poll(vp, events, anyyet, revents, ph);
}

static int
coda_dump(struct vnode *dumpvp,
	  caddr_t addr,
	  int bn,
	  int count)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_dump"));
  return ENOSYS;
}

static int
coda_pathconf(struct vnode *vp,
	      int cmd,
	      u_long *valp,
	      struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_pathconf"));
  return fs_pathconf (vp, cmd, valp, cred);
}

static int
coda_pageio(struct vnode *vp,
	    struct page *page,
#ifdef _LARGEFILE64_SOURCE
	    u_offset_t io_off,  /* Solaris 2.6+ */
#else
	    u_int io_off,  /* Solaris 2.5 */
#endif
	    size_t io_len,
	    int flags,
	    struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_pageio"));
  return ENOSYS;
}

static int
coda_dumpctl(struct vnode *vp,
	     int flag)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_dumpctl"));
  return ENOSYS;
}

static void
coda_dispose(struct vnode *vp,
	     struct page *page,
	     int a,
	     int b,
	     struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_dispose"));
}

static int
coda_setsecattr(struct vnode *vp,
		vsecattr_t *attr,
		int flag,
		struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_setsecattr"));
  return ENOSYS;
}

static int
coda_getsecattr(struct vnode *vp,
		vsecattr_t *attr,
		int flag,
		struct cred *cred)
{
  CODADEB(CDEBVNOPS, (CE_NOTE, "coda_getsecattr"));
  return fs_fab_acl(vp, attr, flag, cred);
}

struct vnodeops coda_vnodeops = {
        coda_open,
        coda_close,
        coda_read,
        coda_write,
        coda_ioctl,
	coda_setfl,
        coda_getattr,
        coda_setattr,
        coda_access,
        coda_lookup,
        coda_create,
        coda_remove,
        coda_link,
        coda_rename,
        coda_mkdir,
        coda_rmdir,
        coda_readdir,
        coda_symlink,
        coda_readlink,
        coda_fsync,
        coda_inactive,
	coda_fid,
	coda_rwlock,
	coda_rwunlock,
	coda_seek,
	coda_cmp,
	coda_frlock,
	coda_space,
	coda_realvp,
        coda_getpage,
        coda_putpage,
        coda_map,
	coda_addmap,
	coda_delmap,
	coda_poll,
        coda_dump,
	coda_pathconf,
	coda_pageio,
	coda_dumpctl,
	coda_dispose,
	coda_setsecattr,
	coda_getsecattr
	/* coda_shrlock */
};
