/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Copyright (C) 1999  Carnegie Mellon University
 * Extensions and modifications to Arla code base for Coda by
 * Philip A. Nelson
 */

/* Started with: xfs_vfsops.c,v 1.12 1998/12/22 13:16:53 lha Exp */
/* $Id: coda_vfsops.c,v 1.3 2000/03/07 20:30:49 phil Exp $ */

#include <coda.h>
#include <coda_sys.h>
#include <coda_common.h>
#include <coda_dev.h>
#include <coda_fs.h>
#include <coda_debug.h>
#include <ncoda.h>
#include <coda_vfsops.h>
#include <coda_upcalls.h>

/* TESTING!!!! */
#include <sys/fs/ufs_inode.h>

extern int nchrdev;

static struct vnode *make_dead_vnode (struct vfs *vfsp);

/* The file system offset given by Solaris.  Used as type some places
   and as id in others.  This may not be use correctly in all places. */
int coda_fstype;

struct coda coda[NCODA];

struct vnode *coda_ctlvp;

static int
coda_mount(struct vfs *vfsp,
	   struct vnode *mvp,
	   struct mounta *uap,
	   struct cred *cred)
{
  struct vnode *devvp;
  struct coda_node *ctlcp;
  ViceFid ctlVFid;
  dev_t dev;
  int error;
  
#ifdef DEBUG
  char dir[MAXPATHLEN], spec[MAXPATHLEN];

  if (copyinstr(uap->dir, dir, sizeof(dir), NULL) ||
      copyinstr(uap->spec, spec, sizeof(spec), NULL))
      return EFAULT;
  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_mount vfsp = 0x%x path = '%s' "
		      "args = '%s'\n", (u_int) vfsp, dir, spec));
#endif

  /*
   * This is something that should be done before calling this
   * function, but it's not, so we do it here.
   */

  if (mvp->v_type != VDIR)
      return ENOTDIR;

  mutex_enter(&mvp->v_lock);
  if (mvp->v_count != 1 || (mvp->v_flag & VROOT)) {
      mutex_exit(&mvp->v_lock);
      return EBUSY;
  }
  mutex_exit(&mvp->v_lock);

  error = lookupname(uap->spec, UIO_USERSPACE, FOLLOW, 0, &devvp);
  if (error != 0)
    return error;

  if (devvp->v_type != VCHR) {
      VN_RELE(devvp);
      return ENXIO;
  }
  dev = devvp->v_rdev;
  VN_RELE(devvp);

  /* Check that this device really is an coda_dev */

  /* I'm not sure how to test this under solaris */
#if 0
  if (getmajor(dev) < 0 || nchrdev < getmajor(dev))
    return ENXIO;
#endif

  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_mount dev = %x, minor = %x, major = %x,"
		      "ops = %x, "
		      "cb_ops = %x, "
		      "open = %x, "
		      "(coda_devopen = %x)\n",
		      (unsigned)dev,
		      (unsigned)getminor(dev),
		      (unsigned)getmajor(dev),
		      (unsigned)devopsp[getmajor(dev)],
		      (unsigned) (devopsp[getmajor(dev)] ? devopsp[getmajor(dev)]->devo_cb_ops : 0),
		      (unsigned) ((devopsp[getmajor(dev)] && devopsp[getmajor(dev)]->devo_cb_ops) ? devopsp[getmajor(dev)]->devo_cb_ops->cb_open : 0),
		      (unsigned) coda_devopen));

  if (getminor(dev) < 0 || NCODA < getminor(dev))
    return ENXIO;
  if (devopsp[getmajor(dev)] == NULL ||
      devopsp[getmajor(dev)]->devo_cb_ops == NULL ||
      devopsp[getmajor(dev)]->devo_cb_ops->cb_open != coda_devopen)
      return ENXIO;

  if (coda[getminor(dev)].status & CODA_MOUNTED)
    return EBUSY;

  coda[getminor(dev)].status = CODA_MOUNTED;
  coda[getminor(dev)].vfsp   = vfsp;
  coda[getminor(dev)].root   = NULL;
  coda[getminor(dev)].nnodes = 0;
  coda[getminor(dev)].nodes  = 0;
  coda[getminor(dev)].rdev   = dev;
  coda[getminor(dev)].minor  = getminor(dev);

  VFS_TO_CODA(vfsp) = &coda[getminor(dev)];
  vfsp->vfs_fstype = coda_fstype;
  vfsp->vfs_dev    = getminor(dev);
  vfsp->vfs_bsize  = PAGESIZE;
  vfsp->vfs_flag  |= VFS_NOTRUNC;
  vfsp->vfs_fsid.val[0] = getminor(dev);
  vfsp->vfs_fsid.val[1] = getmajor(dev); /* What is this good for */

  /* The control vnode/cnode! */
  ctlVFid.Volume = CTL_VOL;
  ctlVFid.Vnode  = CTL_VNO;
  ctlVFid.Unique = CTL_UNI;
  ctlcp = new_coda_node (&coda[getminor(dev)], &ctlVFid, VREG);
  coda_ctlvp = CNODE_TO_VNODE(ctlcp);

  return 0;
}

static int
coda_unmount(struct vfs *vfsp, struct cred *cred)
{
  struct coda *codap = VFS_TO_CODA(vfsp);

  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_umount vfsp = 0x%x\n", (u_int) vfsp));

  free_all_coda_nodes(codap);
  codap->status = 0;
  return 0;			/* Always allow umount to succed */
}

static int
coda_root(struct vfs *vfsp,
	 struct vnode **vpp)     
{
  struct coda *codap = VFS_TO_CODA(vfsp);
  ViceFid Fid;
  int error;
  
  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_root vfsp = 0x%x\n", (u_int) vfsp));

  if (codap->root != 0) {
    *vpp = CNODE_TO_VNODE(codap->root);
    VN_HOLD(*vpp);
    CODADEB(CDEBVFOPS, (CE_NOTE, "coda_root: returning real vnode\n"));
    return 0;
  }

  /* Ask venus for the root. */

  error = coda_venus_root (codap->minor, &Fid);
  if (!error) {
    /* Initialize the root vnode */
    codap->root = new_coda_node (codap, &Fid, VDIR);
    codap->root->vn.v_flag |= VROOT;
    *vpp = CNODE_TO_VNODE(codap->root);
    VN_HOLD(*vpp);
    CODADEB(CDEBVFOPS, (CE_NOTE, "coda_root: returning new vnode\n"));
    return 0;
  }

  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_root: returning dead vnode\n"));

  /*
   * Failed to get message through, need to pretend that all went well
   * and return a fake dead vnode to be able to unmount.
   */
  *vpp = make_dead_vnode(vfsp);
  VN_HOLD(*vpp);
  return 0;
}

static int
coda_statvfs(struct vfs *vfsp,
#ifdef _LARGEFILE64_SOURCE
	    struct statvfs64 *sbp)
#else
	    struct statvfs *sbp)
#endif
{
    CODADEB(CDEBVFOPS, (CE_NOTE, "coda_statvfs\n"));
    sbp->f_bsize = 8192;
    sbp->f_frsize = 1024;
    sbp->f_blocks = 7272727;
    sbp->f_bfree = 7272727;
    sbp->f_bavail = 7272727;
    sbp->f_files = 7272;
    sbp->f_ffree = 7272;
    sbp->f_favail = 7272;
    sbp->f_fsid = coda_fstype;
    strcpy(sbp->f_basetype, "CODA");
    sbp->f_flag = ST_NOTRUNC;
    sbp->f_namemax = 256;


    return 0;
}

static int
coda_sync(struct vfs *vfsp, short flag, struct cred *cred)
{
  /* CODADEB(CDEBVFOPS, (CE_NOTE, "coda_sync\n")); */
  return 0;
}

static int
coda_vget(struct vfs *vfsp,
	 struct vnode **vpp,
	 struct fid *fidp)     
{
  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_vget\n"));
  return ENOSYS;
}

static int
coda_mountroot(struct vfs *vfsp,
	      enum whymountroot reason)
{
  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_mountroot\n"));
  return ENOSYS;
}

static int
coda_swapvp(struct vfs *vfsp,
	   struct vnode **vpp,
	   char *path)     
{
  CODADEB(CDEBVFOPS, (CE_NOTE, "coda_swapvp\n"));
  return ENOSYS;
}

/*
 * To be able to unmount when the CODA daemon is not
 * responding we need a root vnode, use a dead vnode!
 */

static void
dead_vnode_inactive(struct vnode *vp, struct cred *cred)
{
  CODADEB(CDEBVFOPS, (CE_NOTE, "dead_vnode_inactive\n"));
  coda_free(vp, sizeof(*vp));
}

struct vnodeops dead_vnodeops = {
        nodev,			/* coda_open */
        nodev,			/* coda_close */
        nodev,			/* coda_read */
        nodev,			/* coda_write */
        nodev,			/* coda_ioctl */
	nodev,			/* coda_setfl */
        nodev,			/* coda_getattr */
        nodev,			/* coda_setattr */
        nodev,			/* coda_access */
        nodev,			/* coda_lookup */
        nodev,			/* coda_create */
        nodev,			/* coda_remove */
        nodev,			/* coda_link */
        nodev,			/* coda_rename */
        nodev,			/* coda_mkdir */
        nodev,			/* coda_rmdir */
        nodev,			/* coda_readdir */
        nodev,			/* coda_symlink */
        nodev,			/* coda_readlink */
        nodev,			/* coda_fsync */
        dead_vnode_inactive,	/* coda_inactive */
        nodev,			/* coda_fid */
        (void*) nodev,		/* coda_rwlock */
        (void*) nodev,		/* coda_rwunlock */
        nodev,			/* coda_seek */
        nodev,			/* coda_cmp */
        nodev,			/* coda_frlock */
        nodev,			/* coda_space */
        nodev,			/* coda_realvp */
        nodev,			/* coda_getpage */
        nodev,			/* coda_putpage */
        (void*) nodev,		/* coda_map */
        (void*) nodev,		/* coda_addmap */
        nodev,			/* coda_delmap */
        (void*) nodev,		/* coda_poll */
        nodev,			/* coda_dump */
        nodev,			/* coda_pathconf */
        nodev,			/* coda_pageio */
        nodev,			/* coda_dumpctl */
        (void*) nodev,		/* coda_dispose */
        nodev,			/* coda_setsecattr */
        nodev			/* coda_getsecattr */
};

static struct vnode *
make_dead_vnode(struct vfs *vfsp)
{
  struct vnode *dead;

  CODADEB(CDEBNODE, (CE_NOTE, "make_dead_vnode vfsp = 0x%x\n", (u_int) vfsp));

  dead = coda_alloc(sizeof(*dead));
  bzero((caddr_t)dead, sizeof(*dead));
  VN_INIT(dead, vfsp, VDIR, 0);
  dead->v_flag = VROOT;
  dead->v_op = &dead_vnodeops;
  dead->v_count = 0;
  return dead;
}


/*
 * file system
 */

static int
coda_vfs_init (struct vfssw *vfssw, int offset)
{
    printf ("coda_vfs_init: offset = %d\n", offset);
    coda_fstype = offset;
    return 0;
}

static struct vfsops coda_vfsops = {
    coda_mount,			/* mount */
    coda_unmount,		/* unmount */
    coda_root,			/* root */
    coda_statvfs,		/* statvfs */
    coda_sync,			/* sync */
    coda_vget,			/* vget */
    coda_mountroot,		/* mountroot */
    coda_swapvp			/* swapvp */
};

static struct vfssw coda_vfssw = {
    "coda",			/* name */
    coda_vfs_init,		/* init */
    &coda_vfsops,		/* vfsops */
    0				/* flags */
};

struct modlfs coda_modlfs = {
    &mod_fsops,
    "coda filesystem",
    &coda_vfssw
};
