/*
 * 
 *             Coda: an Experimental Distributed File System
 *                              Release 3.1
 * 
 *           Copyright (c) 1999 Carnegie Mellon University
 *                          All Rights Reserved
 * 
 * Permission  to  use, copy, modify and distribute this software and its
 * documentation is hereby granted,  provided  that  both  the  copyright
 * notice  and  this  permission  notice  appear  in  all  copies  of the
 * software, derivative works or  modified  versions,  and  any  portions
 * thereof, and that both notices appear in supporting documentation, and
 * that credit is given to Carnegie Mellon University  in  all  documents
 * and publicity pertaining to direct or indirect use of this code or its
 * derivatives.
 * 
 * CODA IS AN EXPERIMENTAL SOFTWARE SYSTEM AND IS  KNOWN  TO  HAVE  BUGS,
 * SOME  OF  WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON ALLOWS
 * FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.   CARNEGIE  MELLON
 * DISCLAIMS  ANY  LIABILITY  OF  ANY  KIND  FOR  ANY  DAMAGES WHATSOEVER
 * RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE  OR  OF
 * ANY DERIVATIVE WORK.
 * 
 * Carnegie  Mellon  encourages  users  of  this  software  to return any
 * improvements or extensions that  they  make,  and  to  grant  Carnegie
 * Mellon the rights to redistribute these changes without encumbrance.
 * 
 */

/* $Id: coda_upcalls.c,v 1.3 2000/01/26 00:34:11 phil Exp $ */

#include <coda.h>
#include <coda_sys.h>
#include <coda_common.h>
#include <coda_fs.h>
#include <coda_dev.h>
#include <coda_debug.h>
#include <coda_upcalls.h>
#include <sys/fcntl.h>

/* Useful definitions */

#define IN_SIZE(kind) sizeof(struct coda_ ## kind ## _in)
#define OUT_SIZE(kind) sizeof(struct coda_ ## kind ## _out)
#define MSGSIZE(kind) MAX (IN_SIZE(kind), OUT_SIZE(kind))

#define INIT_IN(in, op, cred, p) 					\
	(in).opcode = op; 						\
	(in).pid = ((p) ? (p)->p_pid : -1);				\
	(in).pgid = ((p) ? (p)->p_pgrp : -1);				\
	(in).sid = ((p && (p)->p_sessp) ? (p)->p_sessp->s_sid : -1); 	\
	cred_sol2coda (cred, &((in).cred));


/* Useful functions */
inline static void
cred_sol2coda (cred_t *sol_cred, struct coda_cred *coda_cred)
{
  bzero((char *)coda_cred,sizeof(struct coda_cred));
  if (sol_cred) {
    coda_cred->cr_uid     = sol_cred->cr_ruid;
    coda_cred->cr_groupid = sol_cred->cr_rgid;
    coda_cred->cr_euid    = sol_cred->cr_uid;
    coda_cred->cr_egid    = sol_cred->cr_gid;
  } else {
    coda_cred->cr_uid     =
    coda_cred->cr_groupid =
    coda_cred->cr_euid    =
    coda_cred->cr_egid    = -1;
  }
}

static vtype_t coda2vtype[]
	= {VNON, VREG, VDIR, VBLK, VCHR, VLNK, VSOCK, VFIFO, VBAD};

static void
coda_attr2vattr(struct coda_vattr *cap, struct vattr *vap)
{
  vap->va_mask    = (AT_TYPE | AT_MODE | AT_UID | AT_GID | AT_NODEID
		     | AT_NLINK | AT_SIZE | AT_ATIME | AT_MTIME
		     | AT_CTIME | AT_BLKSIZE | AT_NBLOCKS
		     | AT_RDEV  | AT_FSID );
  vap->va_type    = coda2vtype[cap->va_type];
  vap->va_mode    = cap->va_mode;
  vap->va_uid     = cap->va_uid;
  vap->va_gid     = cap->va_gid;
  vap->va_nodeid  = cap->va_fileid;
  vap->va_nlink   = cap->va_nlink;
  vap->va_size    = cap->va_size;
  vap->va_atime   = cap->va_atime;
  vap->va_mtime   = cap->va_mtime;
  vap->va_ctime   = cap->va_ctime;
  vap->va_blksize = cap->va_blocksize;
  vap->va_nblocks = (cap->va_size+511)/512;
  vap->va_rdev    = cap->va_rdev;
  vap->va_fsid    = coda_fstype;
}


/* static enum coda_vtype vtype2coda[]
	= {C_VNON, C_VREG, C_VDIR, C_VBLK, C_VCHR, C_VLNK, C_VFIFO, C_VBAD,
	   C_VBAD, C_VSOCK, C_VBAD}; */

static timestruc_t notime = {-1, -1};

static void
vattr2coda_attr(struct vattr *vap, struct coda_vattr *cap) 
{
  /* Set fixed values ... i.e. can't change! */
  cap->va_type   = C_VNON;
  cap->va_flags  = -1;
  cap->va_fileid = -1;
  cap->va_gen    = -1;
  cap->va_rdev   = -1;
  cap->va_nlink  = -1;
  cap->va_bytes  = -1;
  cap->va_blocksize = -1;

  /* The ones to set! */
  cap->va_mode   = ( vap->va_mask & AT_MODE ? vap->va_mode : -1);
  cap->va_uid    = ( vap->va_mask & AT_UID ? vap->va_uid : -1);
  cap->va_gid    = ( vap->va_mask & AT_GID ? vap->va_gid : -1);
  cap->va_size   = ( vap->va_mask & AT_SIZE ? vap->va_size : -1);
  cap->va_atime  = ( vap->va_mask & AT_ATIME ? vap->va_atime : notime);
  cap->va_mtime  = ( vap->va_mask & AT_MTIME ? vap->va_mtime : notime);
  cap->va_ctime  = ( vap->va_mask & AT_CTIME ? vap->va_ctime : notime);

  CODADEB (CDEBUPC, (CE_NOTE, "vattr2coda_attr: mask=0x%x", vap->va_mask));
  CODADEB (CDEBUPC, (CE_NOTE, "mode 0%o, uid %d, gid %d, size %d",
		     cap->va_mode, cap->va_uid, cap->va_gid, cap->va_size));
}


/* Upcall functions */

int
coda_venus_root (int minor, ViceFid *VFidp)
{
  struct coda_in_hdr   *in;
  struct coda_root_out *out;
  struct proc *cp;
  cred_t *cred = CRED();
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_root: minor = %d", minor));

  msgsize = MAX(sizeof(struct coda_in_hdr), OUT_SIZE(root));

  in  = (struct coda_in_hdr *) coda_alloc (msgsize);
  out = (struct coda_root_out *) in;
  cp = (curthread ? curproc : NULL);

  INIT_IN(*in, CODA_ROOT, cred, cp);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  sizeof(struct coda_in_hdr), OUT_SIZE(root), UP_WAIT);

  if (!error) {
    *VFidp = out->VFid;

    CODADEB (CDEBUPC,(CE_NOTE,"coda_venus_root VFid: vol=0x%x vn=0x%x "
		      "uniq=0x%x",
		      VFidp->Volume, VFidp->Vnode, VFidp->Unique));
  }

  coda_free(in, msgsize);

  return error;
}

int
coda_venus_open (int minor, cred_t *cred, int flags, ViceFid *VFidp,
		dev_t *dev, ino_t *inode)
{
  struct coda_open_in   *in;
  struct coda_open_out *out;
  struct proc *cp;
  int msgsize;
  int codaflags;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open: minor=%d", minor));

  msgsize = MSGSIZE(open);
  in  = (struct coda_open_in *) coda_alloc (msgsize);
  out = (struct coda_open_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_OPEN, cred, cp);
  in->VFid = *VFidp;
  /* Convert from Solaris flags to coda flags. */
  codaflags = 0;
  if (flags & FREAD)   codaflags |= C_O_READ;
  if (flags & FWRITE)  codaflags |= C_O_WRITE;
  if (flags & O_CREAT) codaflags |= C_O_CREAT;
  if (flags & O_TRUNC) codaflags |= C_O_TRUNC;
  if (flags & O_EXCL)  codaflags |= C_O_EXCL;
  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open: codaflags=%d", codaflags));
  in->flags = codaflags; 


  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(open), OUT_SIZE(open), UP_WAIT);

  if (!error) {
    *dev   = out->dev;
    *inode = out->inode;
    
    CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open: dev=%d inode=%d",
		       *dev, *inode));
  }
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_open_by_path (int minor, cred_t *cred, int flags, ViceFid *VFidp,
		char *name)
{
  struct coda_open_by_path_in   *in;
  struct coda_open_by_path_out *out;
  struct proc *cp;
  int codaflags;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open_by_path: minor = %d", minor));

  msgsize = MAX(IN_SIZE(open_by_path), OUT_SIZE(open_by_path)+CODA_MAXNAMLEN);
  in  = (struct coda_open_by_path_in *) coda_alloc (msgsize);
  out = (struct coda_open_by_path_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_OPEN_BY_PATH, cred, cp);
  in->VFid = *VFidp;
  /* Convert from Solaris flags to coda flags. */
  codaflags = 0;
  if (flags & FREAD)   codaflags |= C_O_READ;
  if (flags & FWRITE)  codaflags |= C_O_WRITE;
  if (flags & O_CREAT) codaflags |= C_O_CREAT;
  if (flags & O_TRUNC) codaflags |= C_O_TRUNC;
  if (flags & O_EXCL)  codaflags |= C_O_EXCL;
  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open_by_path: codaflags=%d",
		     codaflags));
  in->flags = codaflags; 

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(open_by_path), OUT_SIZE(open_by_path), UP_WAIT);

  if (!error) {
    strcpy(name, (char *)out + OUT_SIZE(open_by_path));
    
    CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_open_by_path: name = %s",
		       name));
  }
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_getattr (int minor, cred_t *cred, ViceFid *VFidp, struct vattr *vap)
{
  struct coda_getattr_in   *in;
  struct coda_getattr_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_getattr: minor = %d", minor));

  msgsize = MSGSIZE(getattr);
  in  = (struct coda_getattr_in *) coda_alloc (msgsize);
  out = (struct coda_getattr_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_GETATTR, cred, cp);
  in->VFid = *VFidp;

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(getattr), OUT_SIZE(getattr), UP_WAIT);

  if (!error) {

    coda_attr2vattr(&out->attr, vap);
    
    CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_getattr: success"));
  }
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_setattr (int minor, cred_t *cred, ViceFid *VFidp, struct vattr *vap)
{
  struct coda_setattr_in   *in;
  struct coda_setattr_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_setattr: minor = %d, "
		     "VFid=(%d.%d.%d)",
		     minor, VFidp->Volume, VFidp->Vnode, VFidp->Unique));

  msgsize = MSGSIZE(setattr);
  in  = (struct coda_setattr_in *) coda_alloc (msgsize);
  out = (struct coda_setattr_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_SETATTR, cred, cp);
  in->VFid = *VFidp;
  vattr2coda_attr (vap, &in->attr);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(setattr), OUT_SIZE(setattr), UP_WAIT);

    
  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_setattr: error = %d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_access (int minor, cred_t *cred, ViceFid *VFidp, int mode)
{
  struct coda_access_in   *in;
  struct coda_access_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_access: minor = %d", minor));

  msgsize = MSGSIZE(access);
  in  = (struct coda_access_in *) coda_alloc (msgsize);
  out = (struct coda_access_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_ACCESS, cred, cp);
  in->VFid = *VFidp;
  in->flags = mode >> 6;  /* XXXX Does this work in all cases? */

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(access), OUT_SIZE(access), UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_access: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_lookup (int minor, cred_t *cred, ViceFid *inVFidp, char *name,
		  ViceFid *outVFidp,  vtype_t *vtypep, int *nocache)
{
  struct coda_lookup_in   *in;
  struct coda_lookup_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_lookup: minor = %d, "
		     "name=\"%s\" len=%d", minor, name, namelen));

  msgsize = MAX(IN_SIZE(lookup)+namelen, OUT_SIZE(lookup));
  in  = (struct coda_lookup_in *) coda_alloc (msgsize);
  out = (struct coda_lookup_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_LOOKUP, cred, cp);
  in->VFid = *inVFidp;
  in->name = IN_SIZE(lookup);
  in->flags = CLU_CASE_SENSITIVE;
  bcopy (name, ((char *)in)+in->name, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(lookup)+namelen, OUT_SIZE(lookup), UP_WAIT);

  if (!error) {
    /* Post processing */
    *outVFidp = out->VFid;
    *nocache = (out->vtype | CODA_NOCACHE) != 0;
    out->vtype &= ~CODA_NOCACHE;
    *vtypep = coda2vtype[out->vtype];
  }

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_lookup: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}
		

int
coda_venus_create (int minor, cred_t *cred, ViceFid *inVFidp, char *name,
		  int mode, struct vattr *va, vcexcl_t exclusive,
		  ViceFid *outVFidp)
{
  struct coda_create_in   *in;
  struct coda_create_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_create: minor = %d, "
		     "name=(%s) len=(%d)",
		     minor, name, namelen));

  msgsize = MAX(IN_SIZE(create)+namelen, OUT_SIZE(create));
  in  = (struct coda_create_in *) coda_alloc (msgsize);
  out = (struct coda_create_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_CREATE, cred, cp);
  in->VFid = *inVFidp;
  in->excl = exclusive; 		/* XXXX correct ? */
  in->mode = va->va_mode;		/* ignore mode parameter! */
  vattr2coda_attr(va, &in->attr);
  in->name = IN_SIZE(create);
  bcopy (name, ((char *)in)+in->name, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(create)+namelen, OUT_SIZE(create), UP_WAIT);

  if (!error) {
    /* Post processing */
    *outVFidp = out->VFid;
    coda_attr2vattr(&out->attr, va);
  }

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_create: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_mkdir (int minor, cred_t *cred, ViceFid *inVFidp, char *name,
		 struct vattr *va, ViceFid *outVFidp)
{
  struct coda_mkdir_in   *in;
  struct coda_mkdir_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_mkdir: minor = %d, "
		     "name=(%s) len=(%d)",
		     minor, name, namelen));

  msgsize = MAX(IN_SIZE(mkdir)+namelen, OUT_SIZE(mkdir));
  in  = (struct coda_mkdir_in *) coda_alloc (msgsize);
  out = (struct coda_mkdir_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_MKDIR, cred, cp);
  in->VFid = *inVFidp;
  vattr2coda_attr(va, &in->attr);
  in->name = IN_SIZE(mkdir);
  bcopy (name, ((char *)in)+in->name, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(mkdir)+namelen, OUT_SIZE(mkdir), UP_WAIT);

  if (!error) {
    /* Post processing */
    *outVFidp = out->VFid;
    coda_attr2vattr(&out->attr, va);
  }

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_mkdir: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_link (int minor, cred_t *cred, ViceFid *fileVFidp,
		 ViceFid *dirVFidp, char *name)
{
  struct coda_link_in   *in;
  struct coda_link_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_link: minor = %d, "
		     "name=(%s) len=(%d)",
		     minor, name, namelen));

  msgsize = MAX(IN_SIZE(link)+namelen, OUT_SIZE(link));
  in  = (struct coda_link_in *) coda_alloc (msgsize);
  out = (struct coda_link_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_LINK, cred, cp);
  in->sourceFid = *fileVFidp;
  in->destFid = *dirVFidp;
  in->tname = IN_SIZE(link);
  bcopy (name, ((char *)in)+in->tname, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(link)+namelen, OUT_SIZE(link), UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_link: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_symlink (int minor, cred_t *cred, ViceFid *VFidp,
		    char *linkname, struct vattr *tva, char *tname)
{
  struct coda_symlink_in   *in;
  struct coda_symlink_out *out;
  struct proc *cp;
  int msgsize;
  int linklen = strlen(linkname)+1;
  int tlen = strlen(tname)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_symlink: minor = %d, "
		     "linkname=(%s) len=(%d) entry=(%s)",
		     minor, linkname, linklen, tname));

  msgsize = MAX(IN_SIZE(symlink)+linklen+tlen, OUT_SIZE(symlink));
  in  = (struct coda_symlink_in *) coda_alloc (msgsize);
  out = (struct coda_symlink_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_SYMLINK, cred, cp);
  in->VFid = *VFidp;
  vattr2coda_attr(tva, &in->attr);
  in->srcname = IN_SIZE(symlink);
  in->tname = IN_SIZE(symlink)+linklen;
  bcopy (linkname, ((char *)in)+in->srcname, linklen);
  bcopy (tname, ((char *)in)+in->tname, tlen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(symlink)+linklen+tlen, OUT_SIZE(symlink),
			  UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_symlink: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_remove (int minor, cred_t *cred, ViceFid *VFidp, char *name)
{
  struct coda_remove_in   *in;
  struct coda_remove_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_remove: minor = %d, "
		     "name=(%s) len=(%d)",
		     minor, name, namelen));

  msgsize = MAX(IN_SIZE(remove)+namelen, OUT_SIZE(remove));
  in  = (struct coda_remove_in *) coda_alloc (msgsize);
  out = (struct coda_remove_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_REMOVE, cred, cp);
  in->VFid = *VFidp;
  in->name = IN_SIZE(remove);
  bcopy (name, ((char *)in)+in->name, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(remove)+namelen, OUT_SIZE(remove), UP_WAIT);
  
  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_remove: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_rmdir (int minor, cred_t *cred, ViceFid *VFidp, char *name)
{
  struct coda_rmdir_in   *in;
  struct coda_rmdir_out *out;
  struct proc *cp;
  int msgsize;
  int namelen = strlen(name)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_rmdir: minor = %d, "
		     "name=(%s) len=(%d)",
		     minor, name, namelen));

  msgsize = MAX(IN_SIZE(rmdir)+namelen, OUT_SIZE(rmdir));
  in  = (struct coda_rmdir_in *) coda_alloc (msgsize);
  out = (struct coda_rmdir_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_RMDIR, cred, cp);
  in->VFid = *VFidp;
  in->name = IN_SIZE(rmdir);
  bcopy (name, ((char *)in)+in->name, namelen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(rmdir)+namelen, OUT_SIZE(rmdir), UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_rmdir: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_readlink (int minor, cred_t *cred, ViceFid *VFidp, struct uio *uiop)
{
  struct coda_readlink_in   *in;
  struct coda_readlink_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_readlink: minor = %d, ", minor));

  msgsize = MAX(IN_SIZE(readlink), OUT_SIZE(readlink)+CODA_MAXNAMLEN);
  in  = (struct coda_readlink_in *) coda_alloc (msgsize);
  out = (struct coda_readlink_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_READLINK, cred, cp);
  in->VFid = *VFidp;

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(readlink), OUT_SIZE(readlink)+CODA_MAXNAMLEN,
			  UP_WAIT);
  if (!error) 
    /* Transfer data! */
    error = uiomove((caddr_t) ((char *)out+(int)out->data),
		    out->count, UIO_READ, uiop);
    
  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_readlink: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}


int
coda_venus_rename (int minor, cred_t *cred, ViceFid *srcFidp,
		   char *srcname, ViceFid *dstFidp, char *dstname)
{
  struct coda_rename_in   *in;
  struct coda_rename_out *out;
  struct proc *cp;
  int msgsize;
  int srclen = strlen(srcname)+1;
  int dstlen = strlen(dstname)+1;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_rename: minor = %d, "
		     "srcname=(%s) dstname=(%s)",
		     minor, srcname, dstname));

  msgsize = MAX(IN_SIZE(rename)+srclen+dstlen, OUT_SIZE(rename));
  in  = (struct coda_rename_in *) coda_alloc (msgsize);
  out = (struct coda_rename_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_RENAME, cred, cp);
  in->sourceFid = *srcFidp;
  in->destFid = *dstFidp;
  in->srcname = IN_SIZE(rename);
  in->destname = IN_SIZE(rename)+srclen;
  bcopy (srcname, ((char *)in)+in->srcname, srclen);
  bcopy (dstname, ((char *)in)+in->destname, dstlen);

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(rename)+srclen+dstlen, OUT_SIZE(rename),
			  UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_rename: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_fsync (int minor, cred_t *cred, ViceFid *VFidp)
{
  struct coda_fsync_in   *in;
  struct coda_fsync_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_fsync: minor = %d", minor));

  msgsize = MSGSIZE(fsync);
  in  = (struct coda_fsync_in *) coda_alloc (msgsize);
  out = (struct coda_fsync_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_FSYNC, cred, cp);
  in->VFid = *VFidp;

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(fsync), OUT_SIZE(fsync), UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_fsync: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_close (int minor, cred_t *cred, ViceFid *VFidp, int flag)
{
  struct coda_close_in   *in;
  struct coda_close_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_close: minor = %d", minor));

  msgsize = MSGSIZE(close);
  in  = (struct coda_close_in *) coda_alloc (msgsize);
  out = (struct coda_close_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_CLOSE, cred, cp);
  in->VFid = *VFidp;
  in->flags = 0;
  if (flag & FWRITE)  in->flags |= C_O_WRITE;
  if (flag & O_TRUNC) in->flags |= C_O_TRUNC;

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(close), OUT_SIZE(close), UP_WAIT);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_close: error=%d", error));
 
  coda_free(in, msgsize);

  return error;
}

int
coda_venus_ioctl (int minor, cred_t *cred, ViceFid *VFidp, int cmd,
		  struct PioctlData *ioargp)
{
  struct coda_ioctl_in   *in;
  struct coda_ioctl_out *out;
  struct proc *cp;
  int msgsize;
  int error;

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_ioctl: minor = %d", minor));

  msgsize = MAX(IN_SIZE(ioctl)+ioargp->vi.in_size,
		OUT_SIZE(ioctl)+ioargp->vi.out_size);
  in  = (struct coda_ioctl_in *) coda_alloc (msgsize);
  out = (struct coda_ioctl_out *) in;
  cp = (curthread ? curproc : NULL);

  /* Prepare message */
  INIT_IN(in->ih, CODA_IOCTL, cred, cp);
  in->VFid = *VFidp;
  in->cmd  = cmd;
  in->len  = ioargp->vi.in_size;
  in->rwflag = 0;
  in->data = (char *)IN_SIZE(ioctl);
  error = copyin (ioargp->vi.in, (char *)in+(int)in->data, in->len);

  if (error) {
    coda_free(in, msgsize);
    return error;
  }

  error = coda_do_upcall (minor, (union inputArgs *)in,
			  IN_SIZE(ioctl)+ioargp->vi.in_size,
			  OUT_SIZE(ioctl)+ioargp->vi.out_size, UP_WAIT);

  if (error) {
    coda_free(in, msgsize);
    CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_ioctl: error=%d", error));
    return error;
  }

  if (out->len > ioargp->vi.out_size) {
    coda_free(in, msgsize);
    CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_ioctl: bad out size"));
    return EINVAL;
  }

  error = copyout ((char *)out+(int)out->data, ioargp->vi.out, out->len);

  CODADEB (CDEBUPC, (CE_NOTE, "coda_venus_ioctl: error=%d", error));

  coda_free(in, msgsize);

  return error;
}
