/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Copyright (C) 1999  Carnegie Mellon University
 * Extensions and modifications to Arla code base for Coda by
 * Philip A. Nelson
 */

/* Started with: xfs_dev.c,v 1.14 1999/01/10 16:11:33 rb Exp */
/* $Id: coda_dev.c,v 1.2 2000/03/07 20:30:49 phil Exp $ */

#include <coda.h>
#include <coda_sys.h>
#include <coda_dev.h>
#include <coda_debug.h>
#include <coda_downcall.h>
#include <coda_fs.h>
#include <ncoda.h>

/* For access to the kernel device information. */
static void *coda_dev_state;

/* A mutex for the dev driver. */
static kmutex_t coda_dev_mutex;

/* For the rest of the kernel module so errors can be generated
   if a coda operation is attempted before open or after close. */
int coda_dev_opened;

static void
coda_initq(struct coda_link *q)
{
  q->next = q;
  q->prev = q;
}

/* Is this queue empty? */
#define coda_emptyq(q) ((q)->next == (q))

/* Is this link on any queue? Link *must* be inited! */
#define coda_onq(link) ((link)->next != 0 || (link)->prev != 0)

/* Append q with p */
static void
coda_qinsert(struct coda_link *head, struct coda_link *item)     
{
  item->next = head;
  item->prev = head->prev;
  item->prev->next = item;
  head->prev = item;
}

static void
coda_qdelete(struct coda_link *item)     
{
  item->next->prev = item->prev;
  item->prev->next = item->next;
  item->next = item->prev = 0;
}

/* Called with a mutex in force. */
static void
coda_link_free (struct coda_link *item)
{
  if (!item->is_sleeping)
    coda_free(item->message, MAX(item->insize, item->outsize));
  cv_destroy (&item->cv);
  coda_free(item, sizeof(struct coda_link));
}

/*
 * Only allow one open.
 */
int
coda_devopen(dev_t *devp, int flags, int otyp, cred_t *credp)
{
  struct coda_channel *chan;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devopen dev = %ld, flags = %d, otyp = %d\n",
		    *devp, flags, otyp));

  /* XXXX TESTING ONLY XXXX */

  if (otyp != OTYP_CHR)
      return EINVAL;

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
						   getminor(*devp));
  if (chan == NULL)
      return ENXIO;

  /* Only allow one reader/writer */
  if (chan->status & CHANNEL_OPENED)
    return EBUSY;
  else
    chan->status |= CHANNEL_OPENED;

  chan->message_buffer = coda_alloc(BUFFER_SIZE);
  coda_dev_opened = 1;

  ASSERT(chan->message_buffer != NULL);

  return 0;
}

/*
 * Wakeup all sleepers and cleanup.
 */
int
coda_devclose(dev_t dev, int flags, int otyp, cred_t *credp)
{
  struct coda_channel *chan;
  struct coda_link *first;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devclose dev = %ld, flags = %d, "
		    "otyp = %d\n", dev, flags, otyp));

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
						   getminor(dev));
  if (chan == NULL)
      return ENXIO;

  /* Sanity check, paranoia? */
  if (!(chan->status & CHANNEL_OPENED))
    panic("coda_devclose never opened?");

  /* First, change status! */
  chan->status &= ~CHANNEL_OPENED;
  coda_dev_opened = 0;

  /* No one is going to read those messages so empty queue and
     wakeup process sleeping on those messages! */
  mutex_enter(&coda_dev_mutex);
  while (!coda_emptyq(&chan->messageq)) {
      CODADEB(CDEBDEV, (CE_NOTE, "coda_devclose: before qdelete(messageq)\n"));
      first = chan->messageq.next;
      coda_qdelete(first);
      if (first->is_sleeping) {
	first->error = ENODEV;
	cv_signal(&first->cv);
      } else {
	coda_link_free(first);
      }
      CODADEB(CDEBDEV, (CE_NOTE, "coda_devclose: after qdelete(messageq)\n"));
  }

  /* Wakeup those waiting for replies that will never arrive. */
  while (!coda_emptyq(&chan->sleepq)) {
      CODADEB(CDEBDEV, (CE_NOTE, "coda_devclose: before qdelete(sleepq)\n"));
      first = chan->sleepq.next;
      coda_qdelete(first);
      first->error = ENODEV;
      cv_signal(&first->cv);
      CODADEB(CDEBDEV, (CE_NOTE, "coda_devclose: after qdelete(sleepq)\n"));
  }
  mutex_exit(&coda_dev_mutex);
  
  if (chan->message_buffer) {
      coda_free(chan->message_buffer, BUFFER_SIZE);
      chan->message_buffer = 0;
  }
      
  /* Free all coda_nodes. */
  mutex_enter(&coda_dev_mutex);
  free_all_coda_nodes(&coda[getminor(dev)]);
  mutex_exit(&coda_dev_mutex);

  return 0;
}

/*
 * Move messages from kernel to user space.
 */
int
coda_devread(dev_t dev, struct uio *uiop, cred_t *credp)
{
  struct coda_channel *chan;
  struct coda_link *first;
  int error = 0;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devread dev = %ld\n", dev));

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
						  getminor(dev));
  if (chan == NULL)
      return ENXIO;

  mutex_enter(&coda_dev_mutex);
  if (!coda_emptyq (&chan->messageq)) {
      /* Remove message */
      first = chan->messageq.next;
      coda_qdelete(first);
      mutex_exit(&coda_dev_mutex);

      ASSERT(first->insize != 0);

      CODADEB(CDEBDEV, (CE_NOTE, "coda_devread sending message, size = %ld, "
			"resid = %ld\n", first->insize, uiop->uio_resid));

      error = uiomove((caddr_t) first->message,
		      first->insize, UIO_READ, uiop);

      CODADEB(CDEBDEV, (CE_NOTE, "coda_devread uiomove error = %ld, "
			"resid = %ld\n", error, uiop->uio_resid));

      if (!error) {
	  /* Deal properly with the queue message. */
	  if (first->is_sleeping) {
	      mutex_enter(&coda_dev_mutex);
	      coda_qinsert(&chan->sleepq, first);
	      mutex_exit(&coda_dev_mutex);
	  } else {
	      coda_link_free(first);
	  }
      } else {
	  /* Put it back on the queue for a retry. XXXX limited number? */
	  mutex_enter(&coda_dev_mutex);
	  coda_qinsert(&chan->messageq, first);
	  mutex_exit(&coda_dev_mutex);
      }
  } else
      mutex_exit(&coda_dev_mutex);   

  return error;
}

/*
 * Move messages from user space to kernel space,
 * wakeup sleepers, insert new data in VFS.
 */
int
coda_devwrite(dev_t dev, struct uio *uiop, cred_t *credp)
{
  struct coda_channel *chan;
  int error;
  u_int cnt;
  union outputArgs *msg;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devwrite dev = %ld\n", dev));

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
						   getminor(dev));
  if (chan == NULL)
      return ENXIO;

  ASSERT(chan->message_buffer != NULL);

  cnt = uiop->uio_resid;
  error = uiomove((caddr_t) chan->message_buffer, BUFFER_SIZE,
		  UIO_WRITE, uiop);
  if (error != 0)
    return error;
  
  cnt -= uiop->uio_resid;

  if (cnt < sizeof(struct coda_out_hdr)) {
      cmn_err (CE_NOTE, "coda_devwrite: message too small, cnt = %d\n", cnt);
      return EINVAL;
  }

  /*
   * Do the proper thing with the received message.
   */

  msg = (union outputArgs *) chan->message_buffer;
  if (DOWNCALL(msg->oh.opcode)) {
      error = coda_do_downcall (msg, cnt, chan);
  } else {
      error = coda_finish_upcall (msg, cnt, chan);
  }

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devwrite error = %d\n", error));
  return error;
}

/*
 * Device ioctl.  Used?
 */
int
coda_devioctl(dev_t dev, int cmd, intptr_t arg, int mode, cred_t *credp,
	     int *rvalp)
{
  CODADEB(CDEBDEV, (CE_NOTE, "coda_devioctl dev = %ld, cmd = %d\n", dev, cmd));
  return EINVAL;
}

/*
 * Are there any messages on this filesystem?
 */
int
coda_chpoll(dev_t dev, short events, int anyyet,
	   short *reventsp, struct pollhead **phpp)
{
  struct coda_channel *chan;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_devselect dev = %ld, events = %d, "
		    "anyyet = %d\n", dev, events, anyyet));

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
						   getminor(dev));
  if (chan == NULL)
      return ENXIO;

  if (!(events & POLLRDNORM))
      return 0;

  mutex_enter(&coda_dev_mutex);
  if (!coda_emptyq(&chan->messageq)) {
      *reventsp |= POLLRDNORM;
  } else {
      *reventsp = 0;
      if (!anyyet)
	  *phpp = &chan->pollhead;
  }
  mutex_exit(&coda_dev_mutex);
  return 0;
}


/*
 * Send a message to venus (user space) and wait for reply (if requested).
 */
int
coda_do_upcall(int minor, union inputArgs *message, u_int insize,
	      u_int outsize, u_int waitreply)
{
  struct coda_channel *chan;
  struct coda_link *this_message;
  union  inputArgs *msg;
  int ret;

  ASSERT(message != NULL);

  chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state, minor);
  if (chan == NULL)
      return ENXIO;

  CODADEB(CDEBDEV, (CE_NOTE, "coda_do_upcall opcode = %d\n",
		    message->ih.opcode));

  if (!(chan->status & CHANNEL_OPENED))	/* No receiver? */
    return ENODEV;
  
  if (outsize < sizeof(struct coda_out_hdr)) {
      cmn_err(CE_WARN, "CODA Error: Message buffer to small for result, "
	      "opcode = %d\n", message->ih.opcode);
      return ENOMEM;
  }

  this_message = (struct coda_link *) coda_alloc(sizeof(struct coda_link));
  bzero (this_message, sizeof(struct coda_link));

  if (!waitreply) {
    msg = (union inputArgs *) coda_alloc(insize);
    bcopy((caddr_t)message, (caddr_t)msg, insize);
  } else {
    msg = message;
  }

  cv_init(&this_message->cv, "this_message", CV_DRIVER, NULL);

  ASSERT(insize != 0);

  this_message->error = 0;
  this_message->is_sleeping = waitreply;
  this_message->message = msg;
  this_message->insize = insize;
  this_message->outsize = outsize;
  this_message->unique = ++chan->nsequence;
  msg->ih.unique = this_message->unique;
  mutex_enter(&coda_dev_mutex);
  coda_qinsert(&chan->messageq, this_message);
  mutex_exit(&coda_dev_mutex);
  pollwakeup(&chan->pollhead, POLLRDNORM);
  CODADEB(CDEBDEV, (CE_NOTE, "messageq = %x, next = %x first: %d:%u\n",
		    (int)&chan->messageq, (int)&chan->messageq.next,
		    chan->messageq.next->message->ih.opcode,
		    chan->messageq.next->insize));
  if (!waitreply)
      return 0;
  CODADEB(CDEBDEV, (CE_NOTE, "coda_do_upcall before sleep\n"));
  mutex_enter(&coda_dev_mutex);
  if(cv_wait_sig(&this_message->cv, &coda_dev_mutex) == 0) {
      CODADEB(CDEBDEV, (CE_NOTE, "caught signal\n"));
      this_message->error = EINTR;
  }
  CODADEB(CDEBDEV, (CE_NOTE, "coda_do_upcall after sleep\n"));
  /*
   * Caught signal, got reply message or device was closed.
   * Need to clean up both messageq and sleepq.
   */
  if (coda_onq(this_message)) {
      coda_qdelete(this_message);
  }

  ret = this_message->error;

  mutex_exit(&coda_dev_mutex);
  coda_link_free(this_message);

  return ret;
}

int
coda_finish_upcall(union outputArgs *message, u_int size,
		       struct coda_channel *chan)
{
  struct coda_link *sleepq;
  struct coda_link *t;

  ASSERT(message != NULL);

  mutex_enter(&coda_dev_mutex);
  sleepq = &chan->sleepq;
  t = chan->sleepq.next; /* Really first in q */
  CODADEB(CDEBDEV, (CE_NOTE, "coda_finish_upcall\n"));

  for (; t != sleepq; t = t->next) {
    ASSERT(t->message != NULL);
    if (t->message->ih.unique == message->oh.unique)
      {
	/* Check message->outsize vs size. */

	bcopy((caddr_t)message, (caddr_t)t->message, size);
	t->error = message->oh.result;
	cv_signal (&t->cv);
	break;
      }
  }
  mutex_exit(&coda_dev_mutex);

#ifdef DEBUG
  if (t == sleepq) {
    CODADEB(CDEBDEV, (CE_WARN, "coda_finish_upcall: no message on queue.\n"));
  }
#endif

  return 0;
}


int
coda_dev_init(void)
{
    int ret;

    CODADEB(CDEBDEV, (CE_NOTE, "coda_dev_init\n"));
    ret = ddi_soft_state_init(&coda_dev_state,
			      sizeof(struct coda_channel), NCODA);
    coda_dev_opened = 0;
    mutex_init(&coda_dev_mutex, "coda_dev_mutex", MUTEX_DRIVER, NULL);

    return ret;
}

int
coda_dev_fini(void)
{
    CODADEB(CDEBDEV, (CE_NOTE, "coda_dev_fini\n"));
    ddi_soft_state_fini(&coda_dev_state);
    mutex_destroy(&coda_dev_mutex);
    return 0;
}

static int
coda_dev_getinfo(dev_info_t *dip,
		ddi_info_cmd_t infocmd,
		void *arg,
		void **result)
{
    int ret;

    CODADEB(CDEBDEV, (CE_NOTE, "coda_dev_getinfo\n"));

    switch(infocmd) {
    case DDI_INFO_DEVT2INSTANCE : {
	dev_t dev = (dev_t)arg;
	*result = (void *)getminor(dev);
	ret = DDI_SUCCESS;
	break;
    }
    case DDI_INFO_DEVT2DEVINFO : {
	dev_t dev = (dev_t)arg;
	struct coda_channel *chan;

	chan = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
							getminor(dev));

	if (chan == NULL) {
	    *result = NULL;
	    ret = DDI_FAILURE;
	} else {
	    *result = chan->dip;
	    ret = DDI_SUCCESS;
	}
	break;
    }
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static int
coda_dev_attach(dev_info_t *dip,
	       ddi_attach_cmd_t cmd)
{
    int ret;

    CODADEB(CDEBDEV, (CE_NOTE, "coda_dev_attach\n"));

    switch(cmd) {
    case DDI_ATTACH : {
	int instance = ddi_get_instance(dip);
	struct coda_channel *state;

	ret = ddi_soft_state_zalloc(coda_dev_state, instance);
	if (ret != DDI_SUCCESS)
	    break;
	state = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
							  instance);

	ret = ddi_create_minor_node(dip, "coda", S_IFCHR, instance,
				    DDI_PSEUDO, 0);
	if (ret != DDI_SUCCESS) {
	    ddi_soft_state_free(coda_dev_state, instance);
	    break;
	}
	
	state->dip = dip;
	coda_initq(&state->messageq);
	coda_initq(&state->sleepq);
	state->nsequence = 0;
	state->message_buffer = 0;
	state->status = 0;
	/* how is the pollhead supposed to be initialized? */
	bzero ((caddr_t)&state->pollhead, sizeof(state->pollhead));

	ddi_report_dev(dip);
	ret = DDI_SUCCESS;
	break;
    }	
#ifdef DDI_PM_RESUME
    case DDI_PM_RESUME :
#endif
    case DDI_RESUME :
	ret = DDI_SUCCESS;
	break;
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static int
coda_dev_detach(dev_info_t *dip, ddi_detach_cmd_t cmd)
{
    int ret;

    CODADEB(CDEBDEV, (CE_NOTE, "coda_dev_detach\n"));

    switch (cmd) {
    case DDI_DETACH : {
	int instance = ddi_get_instance(dip);
	struct coda_channel *state;

	state = (struct coda_channel *)ddi_get_soft_state(coda_dev_state,
							  instance);
	ddi_remove_minor_node(dip, NULL);
	ddi_soft_state_free(coda_dev_state, instance);
	ret = DDI_SUCCESS;
	break;
    }
    case DDI_PM_SUSPEND :
    case DDI_SUSPEND :
	ret = DDI_SUCCESS;
	break;
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static struct cb_ops coda_cb_ops = {
    coda_devopen,		/* open */
    coda_devclose,		/* close */
    nodev,			/* strategy */
    nodev,			/* print */
    nodev,			/* dump */
    coda_devread,		/* read */
    coda_devwrite,		/* write */
    coda_devioctl,		/* ioctl */
    nodev,			/* devmap */
    nodev,			/* mmap */
    nodev,			/* segmap */
    coda_chpoll,			/* chpoll */
    nodev,			/* prop_op */
    NULL,			/* cb_str */
    D_NEW | D_MP,		/* flag */
    CB_REV,			/* rev */
    nodev,			/* aread */
    nodev			/* awrite */
};

static struct dev_ops coda_dev_ops = {
    DEVO_REV,			/* rev */
    0,				/* refcnt */
    coda_dev_getinfo,		/* getinfo */
    nulldev,			/* identify */
    nulldev,			/* probe */
    coda_dev_attach,		/* attach */
    coda_dev_detach,		/* detach */
    nodev,			/* reset */
    &coda_cb_ops,		/* cb_ops */
    NULL,			/* bus_ops */
    NULL			/* power */
};

struct modldrv coda_modldrv = {
    &mod_driverops,
    "coda filesystem (cdev driver)",
    &coda_dev_ops
};
