(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: setting.ml,v 1.14 2011-05-23 18:38:12 weis Exp $ *)

(** {3 A module to define and record settings of the compiler} *)

type file_permission = int;;

type setting = {

  mutable marker_char : char;

  mutable debug : bool;
  mutable print_env : bool;
  mutable print_setting : bool;

  mutable set_writable_target : bool;
  mutable target_perms : file_permission option;
  mutable force_output_file_name : bool;

  mutable honor_ssi_directive : bool;
  mutable honor_ssi_include : bool;
  mutable honor_ssi_binding : bool;
  mutable delimit_ssi_include : bool;

  mutable honor_non_ssi_directive : bool;
  mutable honor_non_ssi_include : bool;
  mutable honor_non_ssi_binding : bool;
  mutable delimit_non_ssi_include : bool;
  mutable keep_non_ssi_include : bool;
  mutable keep_non_ssi_binding : bool;

  mutable quote_marker : bool;
  mutable honor_substitution_in_text : bool;

  mutable expand_mode : bool;
  mutable ssi_mode : bool;
  mutable non_ssi_mode : bool;

  mutable honor_line_continuation : bool;

  mutable htmlc_definitions : Types.htmlc_definitions list;
  mutable source_preprocessors : Types.command_name list;
  mutable ssi_preprocessors : Types.command_name list;
  mutable non_ssi_preprocessors : Types.command_name list;

  mutable default_language : Lang.t;
  mutable language : Lang.t;
  mutable supported_languages : Lang.t list;

  mutable search_path : File.dir_name list;

}
;;

(** In addition to the file permission modes, the following mode bits are
   available:

   4000 Set-user-ID on execution.
   2000 Set-group-ID on execution.
   1000 Enable sticky bit. *)
let default = {

  marker_char = '$';

  debug = false;
  print_env = false;
  print_setting = false;

  set_writable_target = false;
  target_perms = None;
  force_output_file_name = false;

  honor_ssi_directive = true;
  honor_ssi_include = true;
  honor_ssi_binding = true;
  delimit_ssi_include = false;

  honor_non_ssi_directive = false;
  honor_non_ssi_include = false;
  honor_non_ssi_binding = false;
  delimit_non_ssi_include = false;
  keep_non_ssi_include = false;
  keep_non_ssi_binding = false;

  quote_marker = false;
  honor_substitution_in_text = true;

  expand_mode = false;
  ssi_mode = true;
  non_ssi_mode = false;

  honor_line_continuation = false;

  htmlc_definitions = [];
  source_preprocessors = [];
  ssi_preprocessors = [];
  non_ssi_preprocessors = [];

  default_language = Lang.get_default ();
  language = Lang.get ();
  supported_languages = [ Lang.Fr; Lang.Uk; Lang.Es; Lang.De; ];

  search_path = [ "." ];

}
;;

let print_target_perms ppf = function
  | None -> Format.fprintf ppf "None"
  | Some perms -> Format.fprintf ppf "Some %d" perms
;;

let print_htmlc_definitions ppf defs =
  let pdef ppf = function
    | Types.Htmlc_environment fname ->
        Format.fprintf ppf "Environment from file `%s'" fname
    | Types.Htmlc_bindings fname ->
        Format.fprintf ppf "List of bindings from file `%s'" fname
    | Types.Htmlc_definition (ident, v) ->
        Format.fprintf ppf "let %s = %S;;" ident v in
  let pdefs ppf pdefs = List.iter (pdef ppf) pdefs in
  Format.fprintf ppf "[@ @[<hov 1>%a@ @]]" pdefs defs
;;

let print_preprocessors ppf prepros =
  let ppp ppf p = Format.fprintf ppf "%s;@ " p in
  let ppps ppf prepros = List.iter (ppp ppf) prepros in
  Format.fprintf ppf "[@ @[<hov 1>%a@ @]]" ppps prepros
;;

let print_language = Lang.print;;
let print_supported_languages ppf langs =
  let rec loop ppf = function
    | [] -> ()
    | x :: xs -> Format.fprintf ppf "%a%a" print_language x rest xs
  and rest ppf = function
    | [] -> ()
    | x :: xs -> Format.fprintf ppf ";@ %a%a" print_language x rest xs in
  Format.fprintf ppf "[@ @[<hov 1>%a@ @]]" loop langs;;

let print_search_path ppf sp = Path.print ppf sp;;

let print_setting ppf = function {

  marker_char = marker;

  debug = debug;
  print_env = print_env;
  print_setting = print_setting;

  set_writable_target = set_writable_target;
  target_perms = target_perms;
  force_output_file_name = force_output_file_name;

  honor_ssi_directive = honor_ssi_directive;
  honor_ssi_include = honor_ssi_include;
  honor_ssi_binding = honor_ssi_binding;
  delimit_ssi_include = delimit_ssi_include;

  honor_non_ssi_directive = honor_non_ssi_directive;
  honor_non_ssi_include = honor_non_ssi_include;
  honor_non_ssi_binding = honor_non_ssi_binding;
  delimit_non_ssi_include = delimit_non_ssi_include;
  keep_non_ssi_include = keep_non_ssi_include;
  keep_non_ssi_binding = keep_non_ssi_binding;

  quote_marker = quote_marker;
  honor_substitution_in_text = honor_substitution_in_text;

  expand_mode = expand_mode;
  ssi_mode = ssi_mode;
  non_ssi_mode = non_ssi_mode;

  honor_line_continuation = honor_line_continuation;

  htmlc_definitions = htmlc_definitions;
  source_preprocessors = source_preprocessors;
  ssi_preprocessors = ssi_preprocessors;
  non_ssi_preprocessors = non_ssi_preprocessors;

  default_language = default_language;
  language = language;
  supported_languages = supported_languages;

  search_path = search_path;

  } ->
   Format.fprintf ppf "@[\
    marker_char = %C;@ \
    debug = %B;@ \
    print_env = %B;@ \
    print_setting = %B;@ \
    @ \
    set_writable_target = %B;@ \
    target_perms = %a;@ @ \
    force_output_file_name = %B;@ \
    @ \
    honor_ssi_directive = %B;@ \
    honor_ssi_include = %B;@ \
    honor_ssi_binding = %B;@ \
    delimit_ssi_include = %B;@ \
    @ \
    honor_non_ssi_directive = %B;@ \
    honor_non_ssi_include = %B;@ \
    honor_non_ssi_binding = %B;@ \
    delimit_non_ssi_include = %B;@ \
    keep_non_ssi_include = %B;@ \
    keep_non_ssi_binding = %B;@ \
    @ \
    quote_marker = %B;@ \
    honor_substitution_in_text = %B;@ \
    @ \
    expand_mode = %B;@ \
    ssi_mode = %B;@ \
    non_ssi_mode = %B;@ \
    @ \
    honor_line_continuation = %B;@ \
    @ \
    htmlc_definitions = %a;@ \
    source_preprocessors = %a;@ \
    ssi_preprocessors = %a;@ \
    non_ssi_preprocessors = %a;@ \
    @ \
    default_language = %a;@ \
    language = %a;@ \
    supported_languages = %a;@ \
    @ \
    search_path = %a;@ \
  @]"
  marker
  debug
  print_env
  print_setting

  set_writable_target print_target_perms target_perms
  force_output_file_name

  honor_ssi_directive
  honor_ssi_include
  honor_ssi_binding
  delimit_ssi_include

  honor_non_ssi_directive
  honor_non_ssi_include
  honor_non_ssi_binding
  delimit_non_ssi_include
  keep_non_ssi_include
  keep_non_ssi_binding

  quote_marker
  honor_substitution_in_text

  expand_mode
  ssi_mode
  non_ssi_mode

  honor_line_continuation

  print_htmlc_definitions htmlc_definitions
  print_preprocessors source_preprocessors
  print_preprocessors ssi_preprocessors
  print_preprocessors non_ssi_preprocessors

  print_language default_language
  print_language language
  print_supported_languages supported_languages

  print_search_path search_path
;;
