(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: primitives.ml,v 1.27 2013-02-01 11:57:56 weis Exp $ *)

(** {3 Defining primitives for the Htmlc functional environment} *)

(** Defining a few primitives:
   - [scan_lines] expands its string argument as if it were a filename
     argument to htmlc (i.e. the string is scanned and all SSI are executed).
   - [use] loads an environment from within an HTML template file.
 *)
let htmlc_scan_lines_prim =
  let ob = Buffer.create Configuration.file_buffer_length in
  let scan_line s =
    let ib = Scanf.Scanning.from_string s in
    Htmlc.treat_buffer "htmlc_scan_lines_prim" ob ib in
  (fun ss ->
   Buffer.reset ob;
   List.iter scan_line ss;
   Buffer.contents ob)
;;

let htmlc_use_envs_prim fnames =
  Htmlc.load_env_file_names fnames;
  Printf.sprintf "<!-- %s loaded -->" (String.concat " " fnames)
;;

let htmlc_include_file_prim =
  let ob = Buffer.create Configuration.file_buffer_length in
  (fun fnames ->
   match fnames with
   | [fname] ->
     Buffer.reset ob;
     let ib = Scanf.Scanning.from_file fname in
     Htmlc.treat_buffer fname ob ib;
     let s = Buffer.contents ob in
     s
   | _ ->
     Debug.failwith
       (Printf.sprintf "Include_file_prim: \
          needs one file name as argument.")
  )
;;

let check_string_length primitive_name s n =
  if String.length s != n then
    Debug.failwith
      (Printf.sprintf "%s: \
         the string argument \"%s\" must be a string of length %i."
         primitive_name s n)
;;

let check_string_is_one_char primitive_name s =
  if String.length s != 1 then
    Debug.failwith
      (Printf.sprintf "%s: \
         the first string argument must be a one-character long string."
         primitive_name)
;;

type alpha = String | Integer | Float;;

let string_of_alpha = function
  | String -> "string"
  | Integer -> "integer"
  | Float -> "float"
;;

let need_alpha_arguments typ count primitive_name =
  let styp = string_of_alpha typ in
  let scount, plural_mark =
    match count with
    | 1 -> "one", ""
    | n when n > 1 -> Printf.sprintf "%i" n, "s"
    | _ -> assert false in
  Debug.failwith
    (Printf.sprintf "%s: needs %s %s argument%s."
       primitive_name scount styp plural_mark)
;;

let need_string_arguments = need_alpha_arguments String
and need_integer_arguments = need_alpha_arguments Integer
and need_float_arguments = need_alpha_arguments Float
;;

(*
let need_string_arguments count primitive_name =
  match count with
  | 1 ->
    Debug.failwith
     (Printf.sprintf "%s: needs one string argument." primitive_name)
  | n ->
    Debug.failwith
     (Printf.sprintf "%s: needs %i string arguments." primitive_name n)
;;
*)

let need_one_string_argument = need_string_arguments 1
;;

let htmlc_include_virtual_prim fnames =
  match fnames with
  | [fname] -> Printf.sprintf "<!--#include virtual=\"%s\"-->" fname
  | _ -> need_one_string_argument "htmlc_include_virtual_prim"
;;

let htmlc_year_range_prim sl =
  match sl with
  | [first; last] ->
    (* We directly compare the date arguments as string values: it is
       equivalent to comparing the integer values they represent ... *)
    begin match compare first last with
    | 0 -> first
    | -1 -> Printf.sprintf "%s-%s" first last
    | _ ->
      Debug.failwith
        "htmlc_year_range_prim: \
         needs 2 integer arguments \
         and the first should be lesser than the second." end
  | _ -> need_integer_arguments 2 "htmlc_year_range_prim"
;;

let htmlc_center_gen_prim sl =
  match sl with
  | [c; s; len] ->
    check_string_is_one_char "htmlc_center_gen_prim" c;
    Lib_strings.center_gen c.[0] s (int_of_string len)
  | _ ->
    Debug.failwith
      "htmlc_center_gen_prim: \
       needs 3 arguments, i.e. \
       (1) a one-character long string used for padding, \
       (2) a string to center, \
       (3) an integer, the length of the resulting string."
;;

let htmlc_center_prim sl = htmlc_center_gen_prim (" " :: sl);;

let htmlc_flush_left_gen_prim sl =
  match sl with
  | [c; s; len] ->
    check_string_is_one_char "htmlc_flush_left_gen_prim" c;
    Lib_strings.flush_left_gen c.[0] s (int_of_string len)
  | _ ->
    Debug.failwith
      "htmlc_flush_left_gen_prim: \
       needs 3 arguments, i.e. \
       (1) a one-character long string used for padding, \
       (2) a string to flush left, \
       (3) an integer, the length of the resulting string."
;;

let htmlc_flush_left_prim sl = htmlc_flush_left_gen_prim (" " :: sl);;

let htmlc_flush_right_gen_prim sl =
  match sl with
  | [c; s; len] ->
    check_string_is_one_char "htmlc_flush_right_gen_prim" c;
    Lib_strings.flush_right_gen c.[0] s (int_of_string len)
  | _ ->
    Debug.failwith
      "htmlc_flush_right_gen_prim: \
       needs 3 arguments, i.e. \
       (1) a one-character long string used for padding, \
       (2) a string to flush right, \
       (3) an integer, the length of the resulting string."
;;

let htmlc_flush_right_prim sl = htmlc_flush_right_gen_prim (" " :: sl);;

let htmlc_capitalize_prim sl =
  match sl with
  | [s] -> String.capitalize s
  | _ -> need_one_string_argument "htmlc_capitalize_prim"
;;

let htmlc_lowercase_prim sl =
  match sl with
  | [s] -> String.lowercase s
  | _ -> need_one_string_argument "htmlc_lowercase_prim"
;;

let htmlc_uppercase_prim sl =
  match sl with
  | [s] -> String.uppercase s
  | _ -> need_one_string_argument "htmlc_uppercase_prim"
;;

let htmlc_string_make_prim sl =
  match sl with
  | [s; len] ->
    check_string_is_one_char "htmlc_string_make_prim" s;
    String.make (int_of_string len) s.[0]
  | _ ->
    Debug.failwith
      "htmlc_string_make_prim: \
       needs 2 arguments: \
       (1) a one-character long string used for padding, \
       (2) an integer, the length of the resulting string."
;;

let htmlc_string_length_prim sl =
  match sl with
  | [s] -> string_of_int (String.length s)
  | _ -> need_one_string_argument "htmlc_string_length_prim"
;;

let htmlc_format_date_prim_gen format_date sl =
  match sl with
  | [month_day; month; week_day; year;] ->
    let md = int_of_string month_day in
    let m = int_of_string month in
    let wd =
      match week_day with
      | "" -> None
      | week_day -> Some (int_of_string week_day) in
    let y = int_of_string year in
    format_date md m wd y
  | _ -> need_string_arguments 4 "htmlc_format_date_prim_gen"
;;

let htmlc_format_date_uk_prim =
  htmlc_format_date_prim_gen Date.format_date_Uk;;

let htmlc_format_date_fr_prim =
  htmlc_format_date_prim_gen Date.format_date_Fr;;

let htmlc_release_prim_gen separator sl =
  match sl with
  | [major; minor;] ->
    let imajor = int_of_string major in
    let iminor = int_of_string minor in
    if imajor < 0 || iminor < 0 then
      Debug.failwith
        "htmlc_release_prim_gen: \
         each string argument must represent a nonnegative integer" else
    Printf.sprintf "%s%c%s" major separator minor
  | _ -> need_string_arguments 2 "htmlc_release_prim_gen"
;;

let htmlc_release_ident_prim = htmlc_release_prim_gen '.'
(** Release_ident is
      Major.Minor
    where Major and Minor are naturals. *)
;;

type level_type = Development | Alpha | Beta | Release_candidate | Patch;;

let level_type_of_string = function
| "development" -> Development
| "alpha" -> Alpha
| "beta" -> Beta
| "release_candidate" -> Release_candidate
| "patch" -> Patch
| s ->
  Debug.failwith
    (Printf.sprintf "level_of_string: unknown level type %S" s)
;;

let abbrev_of_level_type = function
| Development -> "dev"
| Alpha -> "alpha"
| Beta -> "beta"
| Release_candidate -> "rc"
| Patch -> "pl"
;;

let htmlc_level_ident = function
| [ level_type; level_value; ] ->
  let ilevel_value = int_of_string level_value in
  if ilevel_value < 0 then
    Debug.failwith
      "htmlc_level_ident: \
       level_value argument must represent a nonnegative integer" else
  let level_type_abbrev =
    abbrev_of_level_type (level_type_of_string level_type) in
  Printf.sprintf "%s%s" level_type_abbrev level_value
| _ -> need_string_arguments 2 "htmlc_level_ident"
;;

let htmlc_distribution_prim_gen version_separator level_separator = function
| [ major; minor; level_type; level_value; ] ->
  let release_ident =
    htmlc_release_prim_gen version_separator [major; minor;] in
  let level_ident = htmlc_level_ident [level_type; level_value;] in
  Printf.sprintf "%s%c%s" release_ident level_separator level_ident
| _ -> need_string_arguments 4 "htmlc_distribution_prim_gen"
;;

let htmlc_distribution_ident_prim = htmlc_distribution_prim_gen '.' '+'
(** Distribution_ident is
      Major.Minor+Level
    where Major and Minor are naturals and Level is level_type+level_value.... *)
;;

let htmlc_version_full_info_prim = function
| [major; minor; level_type; level_value; version_info; version_date] ->
  let v =
    htmlc_distribution_ident_prim
      [major; minor; level_type; level_value;] in
  Printf.sprintf "%s (%s, %s)" v version_date version_info
| _ -> need_string_arguments 6 "htmlc_version_full_info_prim"
(** Version_full_info is
      Distribution_ident (Version_date, Version_info)
    where Version_date is a string (supposed to be a full iso date)
    and Version_info is a string. *)
;;

let htmlc_release_tag_base_prim = htmlc_release_prim_gen '_'
(** Release_tag_base is similar to Release_ident with underscores '_'
    instead of dots '.'. *)
;;

let htmlc_distribution_tag_base_prim = htmlc_distribution_prim_gen '_' '_'
(** Distribution_tag_base is similar to Distribution_ident with underscores '_'
    instead of dots '.' and plus signs '+'. *)
;;

let htmlc_arithmetic_operation_two_args_gen_prim op name = function
  | [ x; y; ] ->
    let fx = float_of_string x
    and fy = float_of_string y in
    let res = op fx fy in
    string_of_float res
  | _ -> need_float_arguments 2 name
;;

let htmlc_arithmetic_operation_one_arg_gen_prim op name = function
  | [ x; ] ->
    let fx = float_of_string x in
    let res = op fx in
    string_of_float res
  | _ -> need_float_arguments 1 name
;;

let htmlc_opp_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim ( ~-. ) "htmlc_opp_prim"
and htmlc_plus_prim =
  htmlc_arithmetic_operation_two_args_gen_prim ( +. ) "htmlc_plus_prim"
and htmlc_minus_prim =
  htmlc_arithmetic_operation_two_args_gen_prim ( -. ) "htmlc_minus_prim"
and htmlc_multiply_prim =
  htmlc_arithmetic_operation_two_args_gen_prim ( *. ) "htmlc_multiply_prim"
and htmlc_divide_prim =
  htmlc_arithmetic_operation_two_args_gen_prim ( /. ) "htmlc_divide_prim"
and htmlc_power_prim =
  htmlc_arithmetic_operation_two_args_gen_prim ( ** ) "htmlc_power_prim"
and htmlc_sqrt_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim sqrt "htmlc_sqrt_prim"
and htmlc_exp_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim exp "htmlc_exp_prim"
and htmlc_log_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim log "htmlc_log_prim"
and htmlc_cos_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim cos "htmlc_cos_prim"
and htmlc_sin_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim sin "htmlc_sin_prim"
and htmlc_tan_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim tan "htmlc_tan_prim"
and htmlc_acos_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim acos "htmlc_acos_prim"
and htmlc_asin_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim asin "htmlc_asin_prim"
and htmlc_atan_prim =
  htmlc_arithmetic_operation_one_arg_gen_prim atan "htmlc_atan_prim"
;;

let output_two_decimals = Printf.sprintf "%.2f"
and output_no_decimals = Printf.sprintf "%.0f"
;;

let htmlc_decimals_gen_prim output_decimals name = function
  | [ x; ] ->
    let fx = float_of_string x in
    output_decimals fx
  | _ -> need_float_arguments 1 name
;;

let htmlc_two_decimals_prim =
  htmlc_decimals_gen_prim output_two_decimals "htmlc_two_decimals_prim"
and htmlc_no_decimals_prim =
  htmlc_decimals_gen_prim output_no_decimals "htmlc_no_decimals_prim"
;;

let htmlc_conditional_prim = function
  | [ bool; true_expr; false_expr; ] ->
    let b = bool_of_string bool in
    let res = if b then true_expr else false_expr in
    Printf.sprintf "%s" res
  | _ -> need_string_arguments 3 "htmlc_conditional_prim"
;;

let htmlc_is_empty_prim = function
  | [ s; ] -> string_of_bool (String.length s = 0)
  | _ -> need_one_string_argument "htmlc_is_empty_prim"
;;

let htmlc_file_digest_prim sl =
  match sl with
  | [fname] ->
    let fname =
      try Path.find fname with
      | Sys_error _ ->
        Debug.failwith
         (Printf.sprintf "htmlc_file_digest_prim: \
            cannot find file \"%s\"."
         fname) in
    Digest.file fname
  | _ ->
    Debug.failwith
      "htmlc_file_digest_prim: \
       needs 1 file name argument."
;;

let set_primitive (primitive_name, args, htmlc_prim) =
  Env.htmlc_set primitive_name (Env.Primitive (args, htmlc_prim))
;;

let htmlc_init_primitives () =
  List.iter set_primitive [
    "use", [ "fname"; ], htmlc_use_envs_prim;
    "include", [ "fname"; ], htmlc_include_file_prim;
    "include_virtual", [ "fname"; ], htmlc_include_virtual_prim;
    "scan_lines", [ "s"; ], htmlc_scan_lines_prim;
    "center_gen", [ "c"; "s"; "len"; ], htmlc_center_gen_prim;
    "center", [ "s"; "len"; ], htmlc_center_prim;
    "flush_left_gen", [ "c"; "s"; "len"; ], htmlc_flush_left_gen_prim;
    "flush_left", [ "s"; "len"; ], htmlc_flush_left_prim;
    "flush_right_gen", [ "c"; "s"; "len"; ], htmlc_flush_right_gen_prim;
    "flush_right", [ "s"; "len"; ], htmlc_flush_right_prim;
    "capitalize", [ "s"; ], htmlc_capitalize_prim;
    "lowercase", [ "s"; ], htmlc_lowercase_prim;
    "uppercase", [ "s"; ], htmlc_uppercase_prim;
    "string_make", [ "c"; "len"; ], htmlc_string_make_prim;
    "string_length", [ "s"; ], htmlc_string_length_prim;
    "format_date_uk",
      [ "month_day"; "month"; "week_day"; "year"; ],
      htmlc_format_date_uk_prim;
    "format_date_fr",
      [ "month_day"; "month"; "week_day"; "year"; ],
      htmlc_format_date_fr_prim;
    "release_ident",
      [ "major"; "minor"; ],
      htmlc_release_ident_prim;
    "distribution_ident",
      [ "major"; "minor"; "level_type"; "level_value"; ],
      htmlc_distribution_ident_prim;
    "version_full_info",
      [ "major"; "minor"; "level_type"; "level_value";
        "version_info"; "version_date"; ],
      htmlc_version_full_info_prim;
    "release_tag_base",
      [ "major"; "minor"; ],
      htmlc_release_tag_base_prim;
    "distribution_tag_base",
      [ "major"; "minor"; "level_type"; "level_value"; ],
      htmlc_distribution_tag_base_prim;
    "year_range",
      [ "first"; "last"; ],
      htmlc_year_range_prim;
    "opp", [ "x"; ], htmlc_opp_prim;
    "plus", [ "x"; "y"; ], htmlc_plus_prim;
    "minus", [ "x"; "y"; ], htmlc_minus_prim;
    "multiply", [ "x"; "y"; ], htmlc_multiply_prim;
    "divide", [ "x"; "y"; ], htmlc_divide_prim;
    "power", [ "x"; "y"; ], htmlc_power_prim;
    "sqrt", [ "x"; ], htmlc_sqrt_prim;
    "exp", [ "x"; ], htmlc_exp_prim;
    "log", [ "x"; ], htmlc_log_prim;
    "cos", [ "x"; ], htmlc_cos_prim;
    "sin", [ "x"; ], htmlc_sin_prim;
    "tan", [ "x"; ], htmlc_tan_prim;
    "acos", [ "x"; ], htmlc_acos_prim;
    "asin", [ "x"; ], htmlc_asin_prim;
    "atan", [ "x"; ], htmlc_atan_prim;
    "two_decimals", [ "x"; ], htmlc_two_decimals_prim;
    "no_decimals", [ "x"; ], htmlc_no_decimals_prim;
    "conditional",
      [ "bool"; "true_expr"; "false_expr"; ],
      htmlc_conditional_prim;
    "is_empty", [ "s"; ], htmlc_is_empty_prim;
    "file_digest", [ "fname"; ], htmlc_file_digest_prim;
  ]
;;
