(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2009,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: preprocessor.ml,v 1.4 2012-03-07 15:10:39 weis Exp $ *)

(** {6 Preprocessors} *)

let htmlc_setting = Setting.default;;

open Setting;;

(** A -pp_source option was found on the command line:
   a new pre-processor will be called. *)
let get_source_preprocessors, add_source_preprocessor_command =
  (fun () ->
   let prepros = htmlc_setting.source_preprocessors in
   List.rev prepros),
  (fun cmd ->
   htmlc_setting.source_preprocessors <- cmd :: htmlc_setting.source_preprocessors)
;;

let get_ssi_preprocessors, add_ssi_preprocessor_command =
  (fun () ->
   let prepros = htmlc_setting.ssi_preprocessors in
   List.rev prepros),
  (fun cmd ->
   htmlc_setting.ssi_preprocessors <- cmd :: htmlc_setting.ssi_preprocessors)
;;

let get_non_ssi_preprocessors, add_non_ssi_preprocessor_command =
  (fun () ->
   let prepros = htmlc_setting.non_ssi_preprocessors in
   List.rev prepros),
  (fun cmd ->
   htmlc_setting.non_ssi_preprocessors <- cmd :: htmlc_setting.non_ssi_preprocessors)
;;

let get_preprocessors = function
  | Types.Mode_verbatim -> []
  | Types.Mode_source -> get_source_preprocessors ()
  | Types.Mode_ssi -> get_ssi_preprocessors ()
  | Types.Mode_non_ssi -> get_non_ssi_preprocessors ()
;;

(** {6 Running pre-processors} *)

let htmlc_magic_number = "MD5 (\"htmlc\") = 163156a8373f8007e5e9c383026d0729";;
(** [htmlc_magic_number] is used as a marker for the end of file condition. *)

let call_source_preprocessor command_name src_name ob sb =

  Debug.printf "Pre-processor command %s launched with source %s.\n%!"
    command_name src_name;

  let from_command_ic, to_command_oc = Unix.open_process command_name in

  let write_to_source_preprocessor s =
    Debug.printf "Call_source_preprocessor: writing to pre-processor %S\n%!" s;
    Pervasives.output_string to_command_oc s;
    Pervasives.flush to_command_oc in

  let start_preprocessing () =
    write_to_source_preprocessor (Printf.sprintf "%s\n" htmlc_magic_number) in
  let stop_preprocessing () =
    write_to_source_preprocessor (Printf.sprintf "%s\n" htmlc_magic_number) in

  let status =

    let rec read_result () =
      Debug.printf "Call_source_preprocessor: reading result\n%!";
      (* Read some part of the result of calling the command command_name,
         and write this (partial) result into buffer ob. *)
      let l = Pervasives.input_line from_command_ic in
      Debug.printf
        "Call_source_preprocessor: \
         from command %s result of length %i obtained:\n\t%S\n%!"
        command_name (String.length l) l;
      (* If the line we read is the htmlc end of preprocessing marker, then
         we simply exit. *)
      if l = htmlc_magic_number then raise End_of_file else begin
        (* Otherwise, we write the partial result and go on reading.*)
        Buffer.add_string ob l;
        Buffer.add_char ob '\n';
        (* Continue reading. *)
        read_result ();
      end

    and write () =
      Debug.printf "Call_source_preprocessor: writing to pre-processor\n%!";
      if Scanf.Scanning.end_of_input sb then begin
        stop_preprocessing ();
        read_result ();
      end else
      (* Read (at most) 1024 characters from source scan buffer sb. *)
      Scanf.bscanf sb "%1024[\000-\255]" (fun s ->
        (* Output those characters to the command input stream. *)
        write_to_source_preprocessor s;
        (* Continue writing. *)
        write ()) in

    try start_preprocessing (); write () with
    | End_of_file ->
      Debug.printf "Pre-processor %s ended.\n%!" command_name;
      (match Unix.close_process (from_command_ic, to_command_oc) with
       | Unix.WEXITED n -> n
       | Unix.WSIGNALED _ | Unix.WSTOPPED _ -> 2) in

  if status <> 0 then
    Debug.failwith
      (Printf.sprintf
        "file %s: pre processor command \"%s\" failed to execute properly."
        src_name command_name)
(** [call_source_preprocessor command_name src_name ob sb] calls
    [command_name]. *)
;;

let fold prepros =

  let ob = Buffer.create Configuration.file_buffer_length in

  fun src_name sb ->

    List.fold_left
      (fun sb command_name ->
         Buffer.clear ob;
         call_source_preprocessor command_name src_name ob sb;
         let s = Buffer.contents ob in
         Debug.printf
           "Call_source_preprocessors: \"%s\" returned\n%s\n" command_name s;
         Scanf.Scanning.from_string s)
      sb prepros
(** Call in turn each command of the list of preprocesors on the scan
    buffer argument [sb].
    The result is collected in the output buffer [ob], then extracted from
    [ob] and turned into a new scan buffer to call the rest of the command
    list.

    The result is a new scan buffer containing the pre-processed file. *)
;;

let call_preprocessors hmode = fold (get_preprocessors hmode)
(** [call_preprocessors hmode src_name sb] returns a new scanning buffer
    containing the preprocessed contents of the scanning buffer [sb],
    according to the pre processing commands associated to the Htmlc mode
    [hmode]. *)
;;

let call_source_preprocessors = call_preprocessors Types.Mode_source
(** Call the list of source preprocessors on a [scanbuf] argument. *)
;;

let call_ssi_preprocessors = call_preprocessors Types.Mode_ssi
;;

let call_non_ssi_preprocessors = call_preprocessors Types.Mode_non_ssi
;;
