(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: path.mli,v 1.12 2013-02-01 11:57:37 weis Exp $ *)

(** {3 Defining a path machinery to find files in the file system} *)

exception Empty;;
(** This exception is raised when [pop] is called and the current
   directory path is empty. *)

type t = File.dir_name list;;
(** A directory path is a list of directory names. *)

(** {6 The path machinery:
   define a current list of directories to search files into} *)
val get : unit -> t;;
val push : File.dir_name -> unit;;
(** Raises [Failure] if the directory is not readable or non existent. *)
val pop : unit -> unit;;
val init : t -> unit;;

val find : File.file_name -> File.explicit_file_name;;
(** Return the explicit file name of the first file with the given name
    in the current directory path.
    Raise [Sys_error] when the file argument cannot be found. *)

val print : Format.formatter -> t -> unit;;
(** Print the given [Path.t] into the [Formatter.formatter]. *)

(** {6 In path channel manipulation} *)
val open_in : File.file_name -> in_channel;;
val open_in_bin : File.file_name -> in_channel;;
val open_out : File.file_name -> out_channel;;
val open_out_bin : File.file_name -> out_channel;;
(** Opening files in the current path. Equivalent to the regular open
   primitives if the file name argument is not implicit. *)
