(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: path.ml,v 1.14 2012-03-25 01:03:31 weis Exp $ *)

(** {3 The path machinery} *)

(** We define a current list of directories to look up for finding files. *)

exception Empty;;

type t = File.dir_name list;;
(** A directory path is a list of directory names. *)

let is_readable_directory d =
  try ignore (Sys.readdir d); true with
  | _ -> false
;;

let push, pop, init, get =

  let path = ref [] in

  let push d =
   if not (Sys.file_exists d) then
     failwith (Printf.sprintf "Path.push: %s no such directory" d) else
   if not (Sys.is_directory d) then
     failwith (Printf.sprintf "Path.push: %s is not a directory" d) else
   if not (is_readable_directory d) then
     failwith (Printf.sprintf "Path.push: directory %s can not be read" d) else
   path := d :: !path in

  push,
  (fun () ->
   match !path with
   | [] -> raise Empty
   | _ :: rest -> path := rest),
  (fun l -> path := l),
  (fun () -> !path)
;;

let kfind_path f path s =
 try f s with
 | Sys_error _ as x ->

   let rec do_in = function
   | [] -> raise x
   | p :: pts ->
     let fname = Filename.concat p s in
     (* In case of debugging:
     Printf.eprintf "find_path %s\n" fname; *)
     try f fname with
     | Sys_error _ -> do_in pts in
   do_in path
(** val kfind_path : (explicit_file_name -> 'a) -> t -> file_name -> 'a;;
  [kfind_path f path s] applies [f] on any files named [s] in
   the path, and returns the result of the first call that does not
   fail by raising the exception [Sys_error]. For instance,
   [kfind_path open_in (get ()) "foo"] searches the current path for a file
   ["foo"] and returns an input channel to that file. *)
;;

let kfind f fname =
  if Filename.is_implicit fname
  then kfind_path f (get ()) fname
  else f fname
;;

let open_in = kfind Pervasives.open_in;;
let open_in_bin = kfind Pervasives.open_in_bin;;
let open_out = kfind Pervasives.open_out;;
let open_out_bin = kfind Pervasives.open_out_bin;;

let find = kfind (fun fname -> fname);;

let print ppf path =
  let rec print_path_elems ppf = function
  | [] -> ()
  | dir :: dirs ->
    Format.fprintf ppf "%S;@ " dir; print_path_elems ppf dirs in
  Format.fprintf ppf "@[[ %a]@]" print_path_elems path
;;
