(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: main.ml,v 1.48 2012-03-28 13:11:43 weis Exp $ *)

(** {3 The main routine for Htmlc.} *)

open Htmlc_options;;

(* Reporting supported languages. *)
let print_supported_languages () =
  let languages = Htmlc_options.get_supported_languages () in
  prerr_endline
   (Printf.sprintf
      "Htmlc currently supports the following languages: %s.\n\
       To volunteer for a new translation, \
       please contact Pierre.Weis@inria.fr."
      (String.concat ", "
         (List.map Lang.string_of_lang languages)))
;;

let treat_non_file_options () =
  if Htmlc_options.get_print_supported_languages () then begin
    print_supported_languages ();
    flush Pervasives.stderr;
  end;
  if Htmlc_options.get_print_setting () then begin
    Setting.print_setting Format.err_formatter Htmlc_options.htmlc_setting;
    Format.eprintf "@.";
  end
;;

let get_usage () =
  "Usage: htmlc <options> <files>\
 \n       htmlc -c <source.html>\
 \n         (input is file <source.html>, output is file <source.htm>) or\
 \n       htmlc -s <source.shtml>\
 \n         (input is file <source.shtml>, output is file <source.html>) or\
 \n       htmlc -tpl <source>\
 \n         (input is file <source>, output is file <source.html>) or\
 \n       htmlc -str <string source>\
 \n         (input is string <string source>, output is stdout) or\
 \n       htmlc <source>\
 \n         (input is file <source>, output is stdout) or\
 \n       htmlc <source> <destination>\
 \n         (input is file <source>, output is file <destination>) or\
 \n       htmlc [-i | -f | -from] (<source> | -)\
 \n             [-o | -t | -to] (<destination> | -)\
 \n         (input is file <source>, output is file <destination>)\
 \n"
;;

let input_output_domains_collision io_e =
  match io_e with
  | Io_domain.Io_file fname ->
    Debug.failwith
      (Printf.sprintf
         "output file %s collides with the input file: \
          use option [-force_output_file_name] \
          if you want to overwrite the input file."
         fname)
  | Io_domain.Io_standard -> ()
  | Io_domain.Io_string _s -> ()
;;

let check_input_output_domains (source, target) =
  (* Do not delete source file with target file. *)
  if source = target &&
     (* Except for the special convention stdout/stdin. *)
     source <> Io_domain.Io_standard (* not "-"*) &&
     (* Except when explicit source file overwriting permission
        has been granted. *)
     not (Htmlc_options.get_force_output_file_name ())
  then input_output_domains_collision source
;;
let check_io_domains = List.iter check_input_output_domains
;;

let get_io_domains, add_domain, add_o_domain, add_i_domain =
  let domains = ref []
  and source = ref None
  and target = ref None in

  let get_domain dref =
    match !dref with
    | None -> Io_domain.Io_standard
    | Some domain -> domain in

  let add_to_domains (src, tgt) =
    check_input_output_domains (src, tgt);
    domains := (src, tgt) :: !domains in

  (* get_domains *)
  (fun () ->
     begin match !domains with
     (* If there is no file argument, use stdin/stdout as input/output
        domains. *)
     | [] -> add_to_domains (get_domain source, get_domain target)
      (* Otherwise, something has been specified, we must add it,
         if something is still pending. *)
     | _ ->
       match !source, !target with
       | None, None -> ()
       | _, _ -> add_to_domains (get_domain source, get_domain target)
     end;
     let io_domains = List.rev !domains in
     check_io_domains io_domains;
     io_domains),
  (* add_domain *)
  (fun domain ->
     match !source with
     | None -> source := Some domain
     | Some src ->
       let tgt = get_domain target in
       add_to_domains (src, tgt)),
  (* add_o_domain *)
  (fun domain ->
     match !target with
     | None -> target := Some domain
     | Some tgt ->
       let src = get_domain source in
       add_to_domains (src, tgt);
       source := None;
       target := Some domain),
  (* add_i_domain *)
  (fun domain ->
     match !source with
     | None -> source := Some domain
     | Some src ->
       let tgt = get_domain target in
       add_to_domains (src, tgt);
       target := None;
       source := Some domain)
;;

let add_o_file_name fname = add_o_domain (Io_domain.of_filename fname);;
let add_i_file_name fname = add_i_domain (Io_domain.of_filename fname);;
let add_file_name fname = add_domain (Io_domain.of_filename fname);;

(** [add_file_input_output suffix fname.xxx] adds
   - [fname.xxx] as input
   - [fname.suffix] as output
   [fname] is not even supposed to have an extension,
   in this case [fname] is added as input
   and [fname.suffix] as output. *)
let add_file_input_output suffix fname =
  add_i_file_name fname;
  let fname = File.chop_extension fname in
  add_o_file_name (fname ^ suffix)
;;
(* Given a file name [fname] of the form [name.xxx]
   [add_html_file_input_output fname]
      adds [fname.xxx] as input file and [fname.htm] as output file
   [add_shtml_file_input_output fname]
      adds [fname.xxx] as input file and [fname.html] as output file. *)
let add_html_file_input_output, add_shtml_file_input_output =
  (add_file_input_output ".htm"),
  (add_file_input_output ".html")
;;

(** [add_file_name_input_output add_file suffix fname]
   given [fname] with suffix [suffix] as [name.suffix]
   adds relevant input and output files for [htmlc]
   according to function [add_files]. *)
let add_file_name_input_output add_files suffix fname =
  if Filename.check_suffix fname suffix then add_files fname else
  Debug.failwith
    (Printf.sprintf
       "file name %s has not the required %s suffix." fname suffix)
;;

let add_html_file_name =
  add_file_name_input_output add_html_file_input_output ".html"
;;
let add_shtml_file_name =
  add_file_name_input_output add_shtml_file_input_output ".shtml"
;;

let add_str_file s =
  add_i_domain (Io_domain.Io_string s);
  add_o_domain (Io_domain.Io_standard)
;;

let add_tpl_file_name = add_file_input_output ".html";;

(** Initialize Htmlc predefined environment. *)

let add_to_search_path, get_search_path =
  let search_path = ref [] in
  (function dirname ->
   if not (File.is_readable_directory dirname) then
     failwith
      (Printf.sprintf
         "htmlc: directory %S is unreadable; \
          cannot add it to the search path" dirname) else
   begin
     search_path := dirname :: !search_path;
   end),
  (fun () -> List.rev !search_path)
;;

let htmlc_init () =
  Env.htmlc_init_env ();
  Primitives.htmlc_init_primitives ();
;;

open Arg;;

let parse_args () =
  Arg.parse [
    ("-I", Arg.String add_to_search_path,
     "<dirname>:\
    \n    add directory <dirname> in front of the list of\
    \n    directories searched for include files.");
    ("-w", Arg.Bool set_writable_target,
     "<boolean>:\
    \n    triggers if the resulting file should be writable\
    \n         (default is false, hence the resulting file is read only).");
    ("-p", Arg.Int set_target_perms,
     "<int>:\
    \n    the resulting file should have the given permissions.");
    ("-env", String Htmlc.add_env_file_name,
     "<file_name>:\
    \n    read <file_name> to define variables for substitution.");
    ("-bindings", String Htmlc.add_bindings_file_name,
     "<file_name>:\
    \n    read <file_name> to define variables (make file style bindings).");
    ("-D",
     Arg.Tuple [
       String Htmlc.add_definition_ident;
       String Htmlc.add_definition_value;
     ],
     "<ident> <value>:\
    \n    bind <ident> to <value> for substitution.");

    ("-pp_source", String Preprocessor.add_source_preprocessor_command,
     "<command>:\
    \n    use <command> as a preprocessor for each source file argument.");

    ("-pp_ssi_include", String Preprocessor.add_ssi_preprocessor_command,
     "<command>:\
    \n    use <command> as a preprocessor for each SSI included file.");

    ("-pp_non_ssi_include",
       String Preprocessor.add_non_ssi_preprocessor_command,
     "<command>:\
    \n    use <command> as a preprocessor for each non-SSI included file.");

    ("-honor_line_continuation", Bool set_honor_line_continuation,
     "<boolean>:\
    \n    triggers treatment of line continuations\
    \n    (an escaped newline at end of line is considered a line continuation:\
    \n     it is omitted together with the leading blanks of next line).\
    \n       (default is false, hence line continuations are kept\
    \n        verbatim in the compiled file).");

    ("-lang", String set_language,
     "<language>:\
    \n    language is set to <language>\
    \n    (no more guess from file-name extension).");

    ("-default-lang", String set_default_language,
     "<language>:\
    \n    default language is set to <language>.");

    ("-languages", Arg.Unit set_print_supported_languages,
     ":\
    \n    prints the list of htmlc supported languages.");

    ("-c", String add_html_file_name,
     "<file_name.html>:\
    \n    input file is set to <file_name.html>,\
    \n    output file is set to <file_name.htm>.");

    ("-s", String add_shtml_file_name,
     "<file_name.shtml>:\
    \n    input file is set to <file_name.shtml>,\
    \n    output file is set to <file_name.html>.");

    ("-str", String add_str_file,
     "<source string>:\
    \n    input is <source string>,\
    \n    output file is standard output.");

    ("-tpl", String add_tpl_file_name,
     "<file_name>:\
    \n    input file is set to <file_name>,\
    \n    output file is set to <file.html>,\
    \n      where <file> designates <file_name> without \
            its extension (if any).");

    ("-o", String add_o_file_name,
     "<file_name>:\
    \n    output file is set to <file_name>.");

    ("-t", String add_o_file_name,
     "<file_name>:\
    \n    output file is set to <file_name>.");

    ("-to", String add_o_file_name,
     "<file_name>:\
    \n    output file is set to <file_name>.");

    ("-force_output_file_name", Unit set_force_output_file_name,
     ":\
    \n    force the output file destination.\
    \n    Output file can overwrite the input file.");

    ("-i", String add_i_file_name,
     "<file_name>:\
    \n    input file is set to <file_name>.");

    ("-f", String add_i_file_name,
     "<file_name>:\
    \n    input file is set to <file_name>.");

    ("-from", String add_i_file_name,
     "<file_name>:\
    \n    input file is set to <file_name>.");

    ("-v", Unit Version.print_version,
     ":\
    \n    prints the htmlc version number.");

    ("-version", Unit Version.print_long_version,
     ":\
    \n    prints the htmlc long version number.");

    ("-debug", Unit set_debug,
     ":\
    \n    triggers internal debugging.");

    ("-print_env", Unit set_print_env,
     ":\
    \n    prints the initial environment to stderr before processing.");

    ("-print_setting", Unit set_print_setting,
     ":\
    \n    prints the Htmlc setting to stderr before processing.");

    ("-marker_char", String set_marker_char,
     "<string>:\
    \n    set the character marker for htmlc variables and expressions\
    \n          (default is the character \'$\').");

    ("-quote_marker", Bool set_quote_marker,
     "<bool>:\
    \n    triggers the quotation of markers for htmlc expressions in text\
    \n          (default is false).");

    ("-honor_ssi_directive", Bool set_honor_ssi_directive,
     "<bool>:\
    \n    triggers SSI directives execution\
    \n          (default is true).");

    ("-honor_non_ssi_directive", Bool set_honor_non_ssi_directive,
     "<bool>:\
    \n    triggers non-SSI directives execution\
    \n          (default is false).");

    ("-honor_substitution_in_text", Bool set_honor_substitution_in_text,
     "<bool>:\
    \n    triggers htmlc variables substitution inside text\
    \n          (default is true).");

    ("-honor_non_ssi_include", Bool set_honor_non_ssi_include,
     "<bool>:\
    \n    triggers non-SSI inclusion directives execution\
    \n          (default is false).");

    ("-honor_non_ssi_binding", Bool set_honor_non_ssi_binding,
     "<bool>:\
    \n    triggers non-SSI binding directives execution\
    \n          (default is false).");

    ("-keep_non_ssi_include", Bool set_keep_non_ssi_include,
     "<bool>:\
    \n    triggers the copy in the output of honored non-SSI include directives\
    \n          (default is false).");

    ("-keep_non_ssi_binding", Bool set_keep_non_ssi_binding,
     "<bool>:\
    \n    triggers the copy in the output of honored non-SSI binding directives\
    \n          (default is false).");

    ("-delimit_non_ssi_include", Bool set_delimit_non_ssi_include,
     ":\
    \n    triggers delimitation of included files for \
          non-SSI inclusion directives.");

    ("-delimit_ssi_include", Bool set_delimit_ssi_include,
     ":\
    \n    triggers delimitation of included files for \
          SSI inclusion directives.");

    ("-ssi", Unit set_ssi_mode,
     ":\
    \n    use SSI mode, suitable settings for HTML files generation.\
    \n    SSI mode means:\
    \n     - activate SSI directives (inclusion and binding SSI directives\
    \n       are honored and disappear from the output),\
    \n     - substitution of htmlc variables in text is honored.\
    \n    This is the default mode.");

    ("-non_ssi", Unit set_non_ssi_mode,
     ":\
    \n    use non-SSI mode, suitable settings for text file generation.\
    \n    Non-SSI mode means:\
    \n     - activate non-SSI directives (inclusion and \
             binding non-SSI directives\
    \n       are honored and disappear from the output),\
    \n     - substitution of htmlc variables in text is omitted.");

    ("-expand", Unit set_expand_mode,
     ":\
    \n    use expand mode, suitable settings for make file \
          with includes expansion.\
    \n    Expand mode means:\
    \n     - activate non-SSI directives (inclusion and \
             binding non-SSI directives\
    \n       are honored and kept in the output),\
    \n     - substitution of htmlc variables in text is omitted,\
    \n     - end of lines are output verbatim (no fancy treatment).");

   ]
   add_file_name
   (get_usage ())
;;

let get_command_line () =
  String.concat " "
    (Array.to_list Sys.argv)
;;

let fatal_error loc kind reason =
  prerr_endline
    (Printf.sprintf
       "htmlc: when called with arguments ``%s'',\n\
        %s\
        encountered fatal %s error\n\
        ``%s''"
       (get_command_line ()) loc kind reason);
  exit 2
;;

let main () =
  htmlc_init ();
  try
    (* Add the current directory as a last resort place to find files. *)
    add_to_search_path ".";
    if Array.length Sys.argv > 1 then parse_args ();
    (* Register the entire search path. *)
    Htmlc_options.set_search_path (get_search_path ());
    (* Setting environment to expand variables. *)
    Htmlc.bind_htmlc_definitions ();
    Env.debug_print_sorted_env ();
    treat_non_file_options ();
    (* Now go on compiling! *)
    List.iter
      (function (src, tgt) -> Htmlc.treat_domains src tgt)
      (get_io_domains ());
    exit 0
  with
  | Htmlc.Error (fname, line_number, x) ->
      let x_str = Htmlc.string_of_exc x in
      fatal_error
        (Printf.sprintf "file %s line %i,\n" fname line_number)
        "uncaught exception"
        x_str
  | Sys_error s ->
      fatal_error "" "system" s
  | Failure s ->
      fatal_error "" "execution" s
  | x ->
      let x_str = Htmlc.string_of_exc x in
      prerr_endline
        (Printf.sprintf "htmlc: encountering spurious exception ``%s''" x_str);
      assert false
;;

Printexc.print main ()
;;
