(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lib_date.mli,v 1.16 2012-03-25 00:57:01 weis Exp $ *)

(** {3 A library to deal with dates in various languages} *)

(** {6 Numberings for dates} *)

type year = int
and month = int
;;
(** The year and month numberings.
    Years are numbered with their rank in the Gregorian calendar.
    Months are numbered from [0] for January to [11] for December.
*)

type month_day = int
and week_day = int
;;
(** The month day and week day numberings.
    Month days are numbered from [1] for the first day in the month to [28],
    [29], [30], or [31] for the last day in the month.
    Week days are numbered from [0] for Sunday to [6] for Saturday.
*)

type hour = int
and minute = int
and second = int
;;
(** The hour, minute and second numberings.
    Hours are numbered from [0] to [23].
    Minutes are numbered from [0] to [59].
    Seconds are numbered from [0] to [60];
    [60] is the number of the exceptional leap second,
    cf. http://en.wikipedia.org/wiki/Leap_second.
*)

(** {6 String representation for dates} *)

type year_string = string
and month_string = string
;;
(** The year represented as a string and a month with its name according to
  the language. *)

type month_day_string = string
and week_day_string = string
;;
(** The month day represented as a string and the week day with its name
  according to the language. *)

(** {6 General purpose auxiliaries for dates} *)

val year_is_leap : year -> bool;;
(** Checks if the year argument is a leap year. *)
val last_month_day_of_month : month -> year -> month_day;;
(** [last_month_day_of_month month year] return the number of the last day
    of the month for the given year (one of [28], [29], [30], or [31]). *)
val date_is_valid : month_day -> month -> year -> bool;;
(** [date_is_valid month_day month year] check if the given arguments
    represent a valid date. *)
val compare_date :
    (month_day * month * year) -> (month_day * month * year) -> int
;;
(** Compare two dates represented as triple of integers [(month_day,
    month, year)], with the same convention as the polymorphic
    comparison [compare]. [compare_date (d1, m1, y1) (d2, m2, y2)]
    returns respectively [0], [1], or [-1], if date [(d1, m1, y1)] is
    respectively equal to, greater than, or lower than
    the date [(d2, m2, y2)]. *)
val week_day_of_date : month_day -> month -> year -> week_day;;
(** [week_day_of_date month_day month year] return the number of the
    day in the week that corresponds to the date
    [month_day, month, year].
    (Week days are numbered from [0] for Sunday to [6] for Saturday.) *)

(** {6 Translation from number representation to day and month names} *)

val string_of_month : Lang.t -> month -> month_string;;
val string_of_week_day : Lang.t -> week_day -> week_day_string;;
(** Mapping from days and months, represented as theirs numbers, to
    their usual character string names.
    The numbering starts from 0: thus Sunday and January have number 0.
    These functions raise [Invalid_argument] if their arguments are out of
    the proper range ([0..6] for day numbers and [0..11] for month
    numbers), or if the language is not supported. *)
val string_of_month_day : Lang.t -> month_day -> month_day_string;;
(** [string_of_month_day lang m] return the string representation of the
    month day [m] for the given language [lang].
    For instance, [string_of_month_day Uk 2] is ["2nd"]. *)
val string_of_date : Lang.t ->
    month_day -> month -> year ->
    week_day_string * month_day_string * month_string * year_string
;;
(** [string_of_date lang d m y] calculate the week day associated
    with the date [d, m, y], and return the string representations of
    the tuple [(week_day, d, m, y)].
    For instance [string_of_date Uk (21, 3, 2000)] is
    [("Tuesday", "21", "March", "2000")]. *)

val string_of_full_date : Lang.t ->
  week_day -> month_day -> month -> year ->
  week_day_string * month_day_string * month_string * year_string
;;
(** [string_of_full_date lang w d m y] return a string representation
    of the tuple [(w, d, m, y)], supposed to be a date [d, m, y] with
    week day [w]. *)

val string_of_mois : month -> month_string;;
val string_of_jour : week_day -> week_day_string;;
(** Same as above for french days and months. *)

val string_of_mes : month -> month_string;;
val string_of_dia : week_day -> week_day_string;;
(** Same as above for spanish days and months. *)

val string_of_monat : month -> month_string;;
val string_of_tag : week_day -> week_day_string;;
(** Same as above for german days and months. *)

val week_day_of_string : Lang.t -> week_day_string -> week_day;;
val month_of_string : Lang.t -> month_string -> month;;
(** Maps a day (resp. a month) represented as a string for the language
    [lang] to the corresponding number. Recognition of valid strings is
    case unsensitive and usual abbreviations are accepted.
    Raise [Invalid_argument "unknown language"], if the language is
    not supported. *)

(** {6 Parsing and printing dates according to some given language} *)

val format_date :
    Lang.t -> month_day -> month -> week_day option -> year -> string
;;
(** [format_date lang month_day month week_day year] return a string
    representation of a date according to the language [lang]. The
    date is represented by 4 integers: [week_day] and
    [month] are the numbers encoding the names of the day and
    the name of the month. Integers [week_day] (option type) and [month] should
    be in the suitable range for calling [string_of_month] and
    [string_of_week_day].
    The [month_day] and [year] arguments are the number of the day in
    the month and the number of the year. If the year number is lower
    than 1000, then 1900 is added to it.
    There is no verification that the date is valid.
    For instance,
    [format_date Uk 31 12 (Some 5) 00] is ["Friday, December 31, 1900"],
    [format_date Uk 31 12 None 00] is ["December 31, 1900"],
    [format_date Uk 31 12 (Some 5) 101] is ["Friday, December 31, 2001"],
    [format_date Uk 31 12 (Some 5) 2000] is ["Friday, December 31, 2000"]. *)
val string_of_date_Uk :
  month_day -> month -> week_day option -> year -> string
;;
val string_of_date_Fr :
  month_day -> month -> week_day option -> year -> string
;;
val string_of_date_Es :
  month_day -> month -> week_day option -> year -> string
;;
val string_of_date_De :
  month_day -> month -> week_day option -> year -> string
;;
val string_of_date_It :
  month_day -> month -> week_day option -> year -> string
;;
(** Same as above for french, spanish, german, or english translations. *)

val format_digital_date : Lang.t -> month_day -> month -> year -> string;;
(** [format_digital_date lang month_day month year] return a digital
    representation of a date according to the language [lang]. The date is
    represented by 3 integers: [month] is the number encoding the name of the
    month. Integer [month] should be in the suitable range for calling
    [string_of_month].
    The [month_day] and [year] arguments are the number of the day in
    the month and the number of the year. If the year number is lower
    than 1000, then 1900 is added to it.
    There is no verification that the date is valid.
    For instance,
    [format_digital_date Uk 31 12 00] is ["1900/12/31"],
    [format_digital_date Uk 27 12 101] is ["2001/12/27"],
    [format_digital_date Uk 31 2 2000] is ["2000/02/31"]. *)
val string_of_digital_date_Uk : month_day -> month -> year -> string;;
val string_of_digital_date_Fr : month_day -> month -> year -> string;;
val string_of_digital_date_Es : month_day -> month -> year -> string;;
val string_of_digital_date_De : month_day -> month -> year -> string;;
val string_of_digital_date_It : month_day -> month -> year -> string;;
(** Same as above for english, french, spanish, or german translations. *)

val format_iso_time : hour -> minute -> second -> string;;
(** [format_iso_time hour minute second] return the international
    standard notation of a time. This format is [hh:mm:ss] where
    - [hh] refers to a zero-padded hour between 00 and 24, where 24 is only used
      to notate the midnight at the end of a calendar day.
    - [mm] refers to a minute between 00 and 59.
    - [ss] refers to a second between 00 and 59 (or 60 in the exceptional
      case of a leap second, cf. http://en.wikipedia.org/wiki/Leap_second).
    The time is represented by 3 integers: [hour], [minute] and [second] in the
    proper ranges.
    There is no verification that the time is valid.
    For instance,
    [format_iso_time 1 12 0] is ["01:12:00"],
    [format_iso_time 23 59 60] is ["23:59:60"],
    [format_iso_time 24 12 13] is ["24:12:13"]. *)
val string_of_iso_time : hour -> minute -> second -> string;;
(** An alias for [format_iso_time]. *)

val format_iso_date : month_day -> month -> year -> string;;
(** [format_iso_date month_day month year] return the international
    standard notation of a date. This format is [YYYY-MM-DD].
    The date is represented by 3 integers: [month] is the number encoding
    the name of the month. Integer [month] should be in the suitable range
    for calling [string_of_month].
    The [month_day] and [year] arguments are the number of the day in
    the month and the number of the year. If the year number is lower
    than 1000, then 1900 is added to it.
    There is no verification that the date is valid.
    For instance,
    [format_iso_date 31 12 00] is ["1900-12-31"],
    [format_iso_date 27 12 101] is ["2001-12-27"],
    [format_iso_date 31 2 2000] is ["2000-02-31"]. *)
val string_of_iso_date : month_day -> month -> year -> string;;
(** An alias for [format_iso_date]. *)

val format_full_iso_date :
  month_day -> month -> year -> hour -> minute -> second -> string
;;
(** [format_full_iso_date month_day month year hour minute second] return
    the international standard notation of a date and time.
    This format is [<date>T<time>Z] where:
    - [<date>] is [format_iso_date month_day month year] and
    - [<time>] is [format_iso_time hour minute second].

   Note that [Z] means Zulu Time i.e. UTC or GMT. *)
val string_of_full_iso_date :
   month_day -> month -> year -> hour -> minute -> second -> string
;;
(** An alias for [format_full_iso_date]. *)

val parse_time : string -> hour * minute * second;;
(** [parse_time s] parse the string [s] and return a time, the validity of
 which is not verified.
 Separators between words in [s] can be any of [':'], [' '], ['\t'], ['/'],
 [','], ['-'], or ['.'] and can be repeated.
 - If only one word is found in [s], this word is supposed to be made of
 digits that encode the time. Six digits means [hhmmss] (as [010320] to mean
 one hour three minutes and twenty seconds), four digits means [hhmm] (as
 [0103] to mean one hour three minutes and zero second), two digits means [hh]
 (as [01] to mean one hour zero minute and zero second).
 - If two words are found, they are supposed to be the hour and the minute
 in that order, and the resulting time defaults to the first second of the
 given hour and minute.
 - If three words are found, they are supposed to be the hour, the
 minute and the second in that order. *)

val parse_date : Lang.t -> string -> week_day * month_day * month * year;;
(** [parse_date lang s] parse the string [s] according to the language
 [lang], and return a date, the validity of which is not verified.
 Separators between words in [s] can be any of [' '], ['\t'], ['/'], [','],
 ['-'], or ['.'] and can be repeated.
 - If only one word is found in [s], this word is supposed to be made of
 digits that encode the date. Eight digits means [ddmmyyyy] (as
 [01032000] to mean first of March 2000), six digits means [ddmmyy]
 (as [010300] to mean first of March 1900), less than six digits means
 [yy] or [yyyy] (only the year number is provided, day and month
 default to first of January).
 - If two words are found, they are supposed to be the month and the year
 in that order, and the resulting date defaults to the first day of the
 given month and year.
 - If three words are found, they are supposed to be the month day, the
 month and the year. Order is language dependant, and follows the
 conventions of [format_date].
 - If four words are found, they are supposed to be the week day, the
 month day, the month and the year. Order is language dependant, and
 follows the conventions of [format_date].

 Note: if necessary the week day of the date may be found using the
 function [week_day_of_date]. *)

val parse : string -> week_day * month_day * month * year;;
(** [parse s] is equivalent to [parse_date Uk s]. *)
val format : month_day -> month -> week_day option -> year -> string;;
(** [format month_day month week_day year] is equivalent to
    [format_date Uk month_day month week_day year]. *)

val valid_date : Lang.t -> string -> week_day * month_day * month * year;;
(** [valid_date lang s] parse the string [s] according to the
 language [lang] and return a valid date or raise [invalid_arg]. *)

val normalize_date : Lang.t -> string ->
   week_day_string * month_day_string * month_string * year_string;;
(** [normalize_date lang d] parse date [d] and return the four
 fields of a completely specified date ([week_day], [month_day], [month],
 [year]). *)
