(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lang.mli,v 1.8 2009-09-15 08:38:44 weis Exp $ *)

(** {3 Dealing with languages.} *)

type file_name = string;;

type t =
   | Fr | Uk | Es | De | It
 (** The type defining the languages that are currently supported by Caml. *)
;;

val string_of_lang: t -> string;;
 (** Returns a string representation of the given language argument,
    as a toplevel Internet two characters long country name. *)

val lang_of_string: string -> t;;
 (** Returns the language associated to a toplevel Internet two
    characters long country name.

    Raise [Not_found], if no language is available for that country
    name. *)

(** {6 Encoding file names according to the language used. *)

val prefix_of_file_name : file_name -> string;;
val suffix_of_file_name : file_name -> string;;
val lang_suffix_of_file_name : file_name -> string;;
(** Return the ``internal suffix'' of a file name, that is a sub-string
   delimited by [-] and [.] characters (as [fra] in [foo-fra.html] or
   [eng] in [foo-eng.html]). Raise [Not_found], if there is no such
   suffix. *)

val lang_of_file_name : file_name -> t;;
(** Return the language in which a file is written guessed from its name. We
   suppose that the language is indicated by a proper ``internal
   suffix'', as defined by [lang_suffix].
   Suffixes are respectively ["fra"] or ["fr"] for [Fr],
   ["eng"], ["en"], ["us"], or ["uk"] for [Uk],
   ["esp"] or ["es"] for [Es], ["deu"] or ["de"] for [De],
   and ["ita"] or ["it"] for [It].
   Default to [Uk], if no suffix has been found. *)

val suffix_of_lang : t -> string;;
(** Returns a normalized three characters suffix value for a language. *)

val lang_of_suffix : string -> t;;
 (** Returns the language associated to a given suffix argument which may be
    a toplevel Internet country name, or a three letter prefix of the
    country name. Raise [Not_found], if no language is available for
    that suffix. *)

val set : t -> unit;;
(** Set the language to use. Then [get] will return [lang] and no guess
   is ever tried to find a language specification in file names when
   using [get_with_file_name]. *)

val get : unit -> t;;
(** Returns the currently set language or the default language, as set
   by [set_default], or else returns [Uk]. *)

val get_with_file_name : file_name -> t;;
(** Returns the currently set language or the language specified by the
   file name argument's language extension, or else the value of [get ()]. *)

val set_default : t -> unit;;
(** Set the default value of the language to use when no proper extension
   of the file name at hand can be found. *)

val get_default : unit -> t;;
(** Get the value of the language to use as default when no proper extension
   of the file name at hand can be found. *)

val set_from_string : string -> unit;;
(** Set the language according to the given, two characters long,
   Internet country name. *)

val set_default_from_string : string -> unit;;
(** Set the default language according to the given, two characters long,
   Internet country name. *)

val print : Format.formatter -> t -> unit
(** Print the given [Lang.t] into the [Formatter.formatter]. *)
;;
