(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2009,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: io_domain.ml,v 1.5 2012-03-25 01:05:37 weis Exp $ *)

(** {3 Definition and operations on input/output domains.} *)

type io_domain =
   | Io_string of string
   | Io_file of File.file_name
   | Io_standard
(** The various entities to perform ios onto:
   either an immediate string, a file name, or a standard channel.
   Could add functions as source/target of Ios ??? *)
;;

type t = io_domain
(** A short public alias for I/O domains. *)
;;

let of_filename = function
  | "-" -> Io_standard
  | fname -> Io_file fname
;;

let name_in = function
  | Io_string _ -> ""
  | Io_file ifname -> File.name_in ifname
  | Io_standard -> File.name_in "-"
;;

let name_out = function
  | Io_string _ -> ""
  | Io_file ofname -> File.name_out ofname
  | Io_standard -> File.name_out "-"
;;

let permissions = function
  | Io_string _ -> failwith "Io_domains.permissions: string domain"
  | Io_file fname -> File.permissions fname
  | Io_standard -> File.permissions "-"
;;

let make_read_only = function
  | Io_file fname -> File.make_read_only_file fname
  | Io_standard | Io_string _ -> failwith "Io_domain.make_read_only: not a file"
;;

let permissions_in = function
  | Io_string _ -> 0o444 (* = r--r--r-- for a string *)
  | Io_file ifname -> File.permissions_in ifname
  | Io_standard -> File.permissions_in "-"
;;

let permissions_out = function
  | Io_string _ -> 0o666 (* = rw-rw-rw- for a string *)
  | Io_file ofname -> File.permissions_out ofname
  | Io_standard -> File.permissions_out "-"
;;

let open_in = function
    (* We return a meaningless channel for string io domains.
       We could choose to fail, but this would lead to more complex code
       outside this module. *)
  | Io_string _ -> File.open_in "-"
  | Io_file ifname -> File.open_in ifname
  | Io_standard -> File.open_in "-"
;;

let open_out perms = function
    (* We return a meaningless channel for string io domains.
       We could choose to fail, but this would lead to more complex code
       outside this module. *)
  | Io_string _ -> File.open_out_with_perms perms "-"
  | Io_file ofname -> File.open_out_with_perms perms ofname
  | Io_standard -> File.open_out_with_perms perms "-"
;;

let close_in d ic =
  match d with
  | Io_string _ -> ()
  | Io_file fname -> File.close_in fname ic
  | Io_standard -> ()
;;

let close_out d oc =
  match d with
  | Io_string _ -> ()
  | Io_file fname -> File.close_out fname oc
  | Io_standard -> ()
;;

let remove_temporary_target temp_d tgt_d =
  match temp_d with
  | Io_string _ -> ()
  | Io_file temp_fname ->
    let tgt_name =
      match tgt_d with
      | Io_file tgt_name -> tgt_name
      | Io_standard | Io_string _ -> "-" in
    File.remove_temporary_target temp_fname tgt_name
  | Io_standard -> ()
;;

let rename src_d tgt_d =
  match src_d, tgt_d with
  | Io_file src, Io_file tgt -> File.rename src tgt
  | Io_standard, Io_standard -> ()
  | Io_standard, _ | _, Io_standard ->
    failwith "Io_domain.rename: cannot rename to standard I/O"
  | Io_string _, _ | _, Io_string _ ->
    failwith "Io_domain.rename: cannot rename a string"
;;

let remove = function
  | Io_string _ -> ()
  | Io_file ifname -> File.remove ifname
  | Io_standard -> File.remove "-"
;;

let temporary_target ext src_d tgt_d =
  match src_d with
  | Io_string _ -> tgt_d
  | Io_standard -> tgt_d
  | Io_file src_name ->
    match tgt_d with
    | Io_string _ | Io_standard -> tgt_d
    | Io_file tgt_name ->
      Io_file (File.temporary_target ext src_name tgt_name)
(** Return a temporary domain (with extension [ext] in case of new file
  domain). This domaine is suitable to run a domain processor from [src] to
  [tgt], while preserving the [src] domain integrity. *)
;;
