(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: htmlc_options.mli,v 1.5 2011-11-02 14:33:28 weis Exp $ *)

(** {3 [Html_options] module: setting up Htmlc configuration} *)

(** {6 General compiler settings} *)

val set_honor_substitution_in_text : bool -> unit
(** Do we perform substitution in text or not ? *)
;;

val set_marker_char : string -> unit;;
(** Setting the Htmlc marker character (default is ['$']).
    The escape character ['\\'] cannot be the marker character. *)

val set_quote_marker : bool -> unit
(** Do we quote markers in text or not ? *)
;;

(** {6 Server Side Include directives} *)

(** We define bells and whistles for Server Side Include directives (SSI
    directives for short). *)

val set_honor_ssi_directive : bool -> unit
(** Do we evaluate SSI directives or not ? *)
;;

val set_delimit_ssi_include : bool -> unit
(** Do we delimit included file for SSI inclusion directives or not ? *)
;;

(** {6 Non Server Side Include directives} *)

(** We define bells and whistles for directives out of Server Side Include
    directives (non-SSI directives for short). *)

val set_honor_non_ssi_directive : bool -> unit
(** Do we evaluate non-SSI directives or not ? *)
;;

val set_honor_non_ssi_include : bool -> unit
(** Do we perform non-SSI file inclusion directives or not ? *)
;;

val set_honor_non_ssi_binding : bool -> unit
(** Do we perform non-SSI variable definitions or not ? *)
;;

val set_keep_non_ssi_include : bool -> unit
(** Do we perform non-SSI file inclusion directives copy or not ? *)
;;

val set_keep_non_ssi_binding : bool -> unit
(** Do we perform non-SSI variable definitions copy or not ? *)
;;

val set_delimit_non_ssi_include : bool -> unit
(** Do we delimit included file for non-SSI inclusion directives or not ? *)
;;

(** {6 Useful compiler predefined settings *)

val set_ssi_mode : unit -> unit
(** Do we honor SSI directives or not ? *)
;;

val set_non_ssi_mode : unit -> unit
(** Do we honor non-SSI directives or not ? *)
;;

val set_expand_mode : unit -> unit
(** Do we expand the file argument or not ? *)
;;

(** {6 Setting result file permissions} *)

val set_writable_target : bool -> unit
(** By default the Htmlc compiled files are read only, to prevent the user
    from editing a file that can be overwritten at any time.
    Nonetheless, evaluating [set_writable_target true] allows to obtain a
    writable target file. *)
;;

val set_target_perms : int -> unit
(** Set the given permissions in order to set the permissions of the target
  file afterwards. *)
;;

(** {6 Input file protection *)
val set_force_output_file_name : unit -> unit
(** Tell the compiler to output into the input file, if specified so.
    Default is: attempt to output to the input file aborts compilation. *)
;;

(** {6 Multi-lines handling} *)

(** How do we treat escaped end of lines: the resulting line may or may not
    have escaped end of lines and additional material in it.

   - [set_honor_line_continuation true]: the result still have escaped end of
     lines (including spaces and backslashes).
   - [set_honor_line_continuation false]: the result is free of escaped end of
     lines (including spaces and backslashes), in the spirit of Caml string tokens.
*)
val set_honor_line_continuation : bool -> unit
;;

(** {6 Language settings} *)

val set_default_language : string -> unit;;
val set_language : string -> unit;;

(** {6 Debugging} *)
val set_debug : unit -> unit
(** Turn debugging on. *)
;;

(* Internal use only. *)
val htmlc_setting : Setting.setting;;
val get_htmlc_definitions : unit -> Types.htmlc_definitions list;;
val add_htmlc_definition : Types.htmlc_definitions -> unit;;

val get_marker_char : unit -> char;;
val get_debug : unit -> bool;;

val get_print_env : unit -> bool;;
val set_print_env : unit -> unit;;
val get_print_setting : unit -> bool;;
val set_print_setting : unit -> unit;;

val get_compile_to_read_only : unit -> bool;;

val get_target_perms : unit -> Setting.file_permission option;;
val get_honor_ssi_directive : unit -> bool;;
val get_delimit_ssi_include : unit -> bool;;
val get_honor_line_continuation : unit -> bool;;

val get_quote_marker : unit -> bool;;

val get_honor_substitution_in_text : unit -> bool;;
val get_honor_non_ssi_directive : unit -> bool;;
val get_honor_non_ssi_include : unit -> bool;;

val get_delimit_non_ssi_include : unit -> bool;;

val get_honor_non_ssi_binding : unit -> bool;;
val get_keep_non_ssi_include : unit -> bool;;
val get_keep_non_ssi_binding : unit -> bool;;
val get_ssi_mode : unit -> bool;;
val get_non_ssi_mode : unit -> bool;;
val get_expand_mode : unit -> bool;;

val get_force_output_file_name : unit -> bool;;

val get_supported_languages : unit -> Lang.t list;;
val get_print_supported_languages : unit -> bool;;
val set_print_supported_languages : unit -> unit;;

val set_search_path : File.dir_name list -> unit;;

