(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: htmlc_options.ml,v 1.5 2011-11-02 14:33:25 weis Exp $ *)

open Setting;;

(** {3 [Html_options] module: setting up Htmlc configuration} *)

let htmlc_setting = Setting.default;;

(** The list of htmlc binding definitions to be entered before entering the
   file processing phase. *)
let get_htmlc_definitions, add_htmlc_definition =
  (fun () ->
   let definitions = htmlc_setting.htmlc_definitions in
   htmlc_setting.htmlc_definitions <- [];
   List.rev definitions
  ),
  (fun d ->
   htmlc_setting.htmlc_definitions <- d :: htmlc_setting.htmlc_definitions)
;;

(** {6 Debugging and printing options} *)
let get_debug, set_debug =
  (fun () -> htmlc_setting.Setting.debug),
  (fun () -> Debug.set_debug (); htmlc_setting.debug <- true)
;;

let get_print_env, set_print_env =
  (fun () -> htmlc_setting.print_env),
  (fun () -> htmlc_setting.print_env <- true)
;;

let get_print_setting, set_print_setting =
  (fun () -> htmlc_setting.print_setting),
  (fun () -> htmlc_setting.print_setting <- true)
;;

(** {6 Configuration of file handling} *)

(** Do we produce read only compiled files ? *)
let get_compile_to_read_only, set_writable_target =
  (fun () -> not htmlc_setting.set_writable_target),
  (fun b -> htmlc_setting.set_writable_target <- b)
;;

(** Which file perms do we use for compiled files ? *)
let get_target_perms, set_target_perms =
  (fun () -> htmlc_setting.target_perms),
  (fun m -> htmlc_setting.target_perms <- Some m)
;;

(** {6 Evaluation of SSI directives} *)

(** Do we evaluate SSI directives or not ? *)
let get_honor_ssi_directive, set_honor_ssi_directive =
  (fun () -> htmlc_setting.honor_ssi_directive),
  (fun b -> htmlc_setting.honor_ssi_directive <- b)
;;

(** Do we delimit SSI included files ? *)
let get_delimit_ssi_include, set_delimit_ssi_include =
  (fun () -> htmlc_setting.delimit_ssi_include),
  (fun b -> htmlc_setting.delimit_ssi_include <- b)
;;

(** {6 Multi-lines treatment} *)

(** Do we keep multi-line lines as verbatim ? *)
let get_honor_line_continuation, set_honor_line_continuation =
  (fun () -> htmlc_setting.honor_line_continuation),
  (fun b -> htmlc_setting.honor_line_continuation <- b)
;;

(** {6 Marker quotation} *)

(** Setting the marker char *)
let get_marker_char, set_marker_char =
  (fun () -> htmlc_setting.marker_char),
  (fun s ->
   if String.length s = 1 then
   let c = s.[0] in
   match c with
   | '\\' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which is the escape character"
          c)
   | '\"' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which is the string delimiter character"
          c)
   | '\'' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which is the char delimiter character"
          c)
   | '#' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which starts htmlc uniline comments"
          c)
   | '(' | '=' | ')' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which is a reserved symbol"
          c)
   | '+' | '-' | '0' .. '9' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which may start a number"
          c)
   | 'A'..'Z' | 'a' .. 'z' | '_' ->
     failwith
       (Printf.sprintf
          "htmlc: cannot set the marker character to %C \
           which may start an identifier"
          c)
   | c -> htmlc_setting.marker_char <- c)
;;

(** Do we quote markers in texts ? *)
let get_quote_marker, set_quote_marker,
(** Do we perform substitution in text or not ? *)
    get_honor_substitution_in_text, set_honor_substitution_in_text =

  let get_quote_marker () = htmlc_setting.quote_marker
  and set_quote_marker b =
    if htmlc_setting.honor_substitution_in_text && b then
      failwith "Cannot fire marker quoting \
                if substitution in text is set."
    else htmlc_setting.quote_marker <- b

  and get_honor_substitution_in_text () =
    htmlc_setting.honor_substitution_in_text
  and set_honor_substitution_in_text b =
    if htmlc_setting.quote_marker && b then
      failwith "Cannot fire substitution in text \
                if we have to quote markers."
    else htmlc_setting.honor_substitution_in_text <- b in

  get_quote_marker, set_quote_marker,
  get_honor_substitution_in_text, set_honor_substitution_in_text
;;

(** {6 Non-ssi directives} *)

(** Do we evaluate non-ssi directives
   (i.e. directives out of SSI directives) ? *)
let get_honor_non_ssi_directive, set_honor_non_ssi_directive =
  (fun () -> htmlc_setting.honor_non_ssi_directive),
  (fun b -> htmlc_setting.honor_non_ssi_directive <- b)
;;

(** Do we perform non-ssi inclusion directives ? *)
let get_honor_non_ssi_include, set_honor_non_ssi_include =
  (fun () -> htmlc_setting.honor_non_ssi_include),
  (fun b -> htmlc_setting.honor_non_ssi_include <- b)
;;

(** Do we delimit non-ssi included files ? *)
let get_delimit_non_ssi_include, set_delimit_non_ssi_include =
  (fun () -> htmlc_setting.delimit_non_ssi_include),
  (fun b -> htmlc_setting.delimit_non_ssi_include <- b)
;;

(** Do we perform non-ssi binding directives ? *)
let get_honor_non_ssi_binding, set_honor_non_ssi_binding =
  (fun () -> htmlc_setting.honor_non_ssi_binding),
  (fun b -> htmlc_setting.honor_non_ssi_binding <- b)
;;

(** Do we keep non-ssi included files ? *)
let get_keep_non_ssi_include, set_keep_non_ssi_include =
  (fun () -> htmlc_setting.keep_non_ssi_include),
  (fun b -> htmlc_setting.keep_non_ssi_include <- b)
;;

(** Do we perform non-ssi binding directives ? *)
let get_keep_non_ssi_binding, set_keep_non_ssi_binding =
  (fun () -> htmlc_setting.keep_non_ssi_binding),
  (fun b -> htmlc_setting.keep_non_ssi_binding <- b)
;;

(** {6 Three predefined settings of the compiler} *)

(** SSI mode *)
let get_ssi_mode, set_ssi_mode =
  (fun () -> htmlc_setting.ssi_mode),
  (fun () ->
     set_honor_ssi_directive true;
     set_honor_substitution_in_text true;
     set_honor_non_ssi_directive false;
     set_honor_non_ssi_include false;
     set_honor_non_ssi_binding false;
     set_keep_non_ssi_include false;
     set_keep_non_ssi_binding false;
     set_delimit_non_ssi_include false;
     set_delimit_ssi_include false;
     htmlc_setting.ssi_mode <- true;
  )
;;

(** Non-SSI mode *)
let get_non_ssi_mode, set_non_ssi_mode =
  (fun () -> htmlc_setting.non_ssi_mode),
  (fun () ->
     set_honor_ssi_directive true;
     set_honor_substitution_in_text true;
     set_honor_non_ssi_directive true;
     set_honor_non_ssi_include true;
     set_honor_non_ssi_binding true;
     set_keep_non_ssi_include false;
     set_keep_non_ssi_binding false;
     set_delimit_non_ssi_include false;
     set_delimit_ssi_include false;
     htmlc_setting.non_ssi_mode <- true;
  )
;;

(** Expand mode *)
let get_expand_mode, set_expand_mode =
  (fun () -> htmlc_setting.expand_mode),
  (fun () ->
     set_honor_ssi_directive true;
     set_honor_substitution_in_text false;
     set_honor_non_ssi_directive true;
     set_honor_non_ssi_include true;
     set_honor_non_ssi_binding true;
     set_keep_non_ssi_include false;
     set_keep_non_ssi_binding true;
     set_delimit_non_ssi_include true;
     set_delimit_ssi_include true;
     set_honor_non_ssi_binding true;
     set_honor_line_continuation true;
     htmlc_setting.expand_mode <- true;
  )
;;

(** The -f option. *)
let get_force_output_file_name, set_force_output_file_name =
  (fun () -> htmlc_setting.force_output_file_name),
  (fun () -> htmlc_setting.force_output_file_name <- true)
;;

(* Languages *)
(** Currently supported languages. *)
let get_supported_languages () = htmlc_setting.supported_languages
;;

let get_print_supported_languages, set_print_supported_languages =
  let print_supported_languages = ref false in
  (fun () -> !print_supported_languages),
  (fun () -> print_supported_languages := true)
;;

(* Setting the default and current language used by Htmlc functions. *)
let set_default_language s =
  Lang.set_default_from_string s;
  htmlc_setting.default_language <- Lang.get_default ();
;;

let set_language s =
  Lang.set_from_string s;
  htmlc_setting.language <- Lang.get ();
;;

(** Path *)

(* Addition of the collected list of directories to the [search_path],
   being careful to update the Htmlc settings. *)
let set_search_path sp =
  List.iter Path.push sp;
  htmlc_setting.search_path <- Path.get ();
;;
