(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: htmlc.ml,v 1.120 2012-10-12 08:14:03 weis Exp $ *)

(**
The SSI directives we deal with:

Fully supported by htmlc: "DATE_LOCAL" "DATE_GMT" "LAST_MODIFIED"
   "DOCUMENT_PATH_INFO" "USER_NAME" (default is "<unknown>")
   "DOCUMENT_NAME"
Not yet supported: "DOCUMENT_URI" "QUERY_STRING_UNESCAPED"

When echoing vars, the encoding is taken into account:
encoding can be
- none (no encoding),
- url (URL encoded), or
- entity (HTML entities encoding)

Note: when echoing an Htmlc defined variable (or expression), you need to
explicitely write a $, as in
<!--#echo var="$(the_date)" -->

Configuration directive:

 - Formats attributes:
   * time: "timefmt" is not yet implemented (needs meta formats in Ocaml...)
      example: <!--#config timefmt="%A, %B %e, %Y" -->

   * size: "sizefmt" is implemented; features "bytes" and "abbrev"
 - Error messages:
   Not implemented.
*)

open Types;;

type encoding = | E_none | E_url | E_entity;;
type sizefmt = | S_bytes | S_abbrev;;

type htmlc_fmts = {
   mutable sizefmt : sizefmt;
   mutable timefmt : string;
 }
;;

(* Error messages and alerts, *)
exception Error of string * int * exn;;

let rec string_of_exc = function
  | Parser.Error _ as x -> Parser.string_of_exc x
  | Error (_fname, line_number, x) ->
      Printf.sprintf
        "line %i, encountering exception\n%s.\n"
        line_number (string_of_exc x)
  | x -> Printexc.to_string x
;;

let unknown_qualifier m q =
  Env.unknown
    (Printf.sprintf "qualifier %s" q)
    (Printf.sprintf "for %s" m)
;;

let unknown_command c = Env.unknown "command" c;;

let unknown_modifier m = Env.unknown "modifier" m;;

let warning s = prerr_endline (Printf.sprintf "htmlc: warning %s" s);;

(** {6 HTML encoding stuff} *)

(** Simple encoding function for trivial HTML special characters. *)
let html_encode_escape ob s =
  let l = String.length s in
  for i = 0 to l - 1 do
    match s.[i] with
    | '<' -> Buffer.add_string ob "&lt;"
    | '>' -> Buffer.add_string ob "&gt;"
    | '&' -> Buffer.add_string ob "&amp;"
    (* Should be somewhat more complex and use the stuff from dehtml ... *)
    | c -> Buffer.add_char ob c
  done
;;

let decimal_to_hexa i =
  if i < 10 then char_of_int (i + int_of_char '0')
  (* 55 = int_of_char 'A' - 10 *)
  else char_of_int (i + 55)
;;

let hexa_char c =
  let s = String.make 3 '%'
  and i = int_of_char c in
  s.[1] <- decimal_to_hexa (i / 16);
  s.[2] <- decimal_to_hexa (i mod 16);
  s
;;

let url_encode ob s =
  let l = String.length s in
  for i = 0 to l - 1 do
    match s.[i] with
    | ' ' -> Buffer.add_char ob '+'
    | '0'..'9' | 'a'..'z' | 'A'..'Z' as c -> Buffer.add_char ob c
    | '\n' -> Buffer.add_string ob "%0D%0A"
    | c -> Buffer.add_string ob (hexa_char c)
  done
;;

(** {6 Environments and definitions} *)

open Htmlc_options;;

(** Environment file fname has to be loaded. *)
let add_env_file_name fname =
  Debug.printf "Adding environment %s to be processed" fname;
  add_htmlc_definition (Htmlc_environment fname)
;;

(** List of bindings in file fname has to be defined. *)
let add_bindings_file_name fname =
  Debug.printf "Adding bindings %s to be processed" fname;
  add_htmlc_definition (Htmlc_bindings fname)
;;

(** A binding was found in the command line:
   a new Htmlc_definition is added to the bindings to be defined. *)
let add_definition_ident, add_definition_value =
  let definition_ident = ref None in
  (fun ident -> definition_ident := Some ident),
  (fun value ->
    let ident =
      match !definition_ident with
      | None -> failwith "No identifier defined for command line binding."
      | Some ident -> ident in
    definition_ident := None;
    add_htmlc_definition (Htmlc_definition (ident, value)))
;;

(** {6 Auxiliaries for file inclusion} *)

(** Specialised buffer handling functions. *)

(** Addition of a text string with substitution. *)
let htmlc_add_string ob s =
  if Htmlc_options.get_quote_marker () then
    let marker = Htmlc_options.get_marker_char () in
    Substitute.add_string
      (Eval.htmlc_quote_marker marker) ob s else
  if Htmlc_options.get_honor_substitution_in_text () then
    Substitute.add_string Eval.htmlc_subst ob s else
  Buffer.add_string ob s
;;

let htmlc_substitute_string =
  let htmlc_line = Buffer.create Configuration.line_buffer_length in
  function s ->
   Buffer.clear htmlc_line;
   htmlc_add_string htmlc_line s;
   Buffer.contents htmlc_line
;;

(** File inclusion functions. *)
let cannot_include_file fname x =
  let failure_string =
    match x with
    | Failure s -> s
    | Sys_error s -> s
    | x -> string_of_exc x in
  Debug.failwith
    (Printf.sprintf
       "while loading file %s: %s" fname failure_string)
;;

(** Include in result the contents of the file [fname] using the scanning
   buffer processing function argument [include_scan_buf]. *)
let htmlc_load_included_file hmode include_scan_buf fname =
  let ic = Path.open_in fname in
  let sb =
    Preprocessor.call_preprocessors hmode fname
      (Scanf.Scanning.from_channel ic) in
  try
    include_scan_buf sb;
    close_in ic
  with
  | x -> cannot_include_file fname x
;;

(** Include in result the contents of a file by reading it one line at a time
   and feeding the lines to an [include_line] function argument. *)
let htmlc_include_scan_buf_by_lines include_line sb =
  while not (Scanf.Scanning.end_of_input sb) do
    (* We read a line and process it. *)
    Scanf.bscanf sb "%[^\n]" include_line;
    Scanf.bscanf sb "%[\n]" include_line;
  done
;;

(** Include in result the contents of a file as ``HTML verbatim'': each HTML
   entity is properly encoded to appear in the HTML result as in the original
   file. *)
let add_html_verbatim_file ob fname =
  let include_scan_buf =
    htmlc_include_scan_buf_by_lines (html_encode_escape ob) in
  htmlc_load_included_file Types.Mode_ssi include_scan_buf fname
;;

(** Include in result the content of a file ``verbatim'': no treatment
   whatsoever is applied to the contents of the included file. *)
let add_verbatim_file ob fname =
  let include_scan_buf =
    htmlc_include_scan_buf_by_lines (Buffer.add_string ob) in
  htmlc_load_included_file Types.Mode_verbatim include_scan_buf fname
;;

(** {6 Binding identifiers} *)

(** Given a scanning buffer sb, an ident [name], and a(n unread) character,
   find the specified value for the name, then add the corresponding
   binding to the environment.
   We parse ident *[= *]value[;;]\n
   and value is either a string, a char, a boolean, or a floating point
   number. *)
let rec bind_ident_parsing sb name args =
  let marker = Htmlc_options.get_marker_char () in
  Debug.printf "bind_ident_parsing %s\n" name;
  Scanf.bscanf sb "%0c" (function
  | '=' -> Scanf.bscanf sb "= " (bind_ident_parsing sb name) args
  | _ ->
    let e = Parser.parse_expression marker sb in
    Scanf.bscanf sb " %_[;\n\r]"
      (Eval.htmlc_bind name args)
      (Eval.string_of_lxm e))
;;

let rec bind_ident_end_of_line sb name args =
  Debug.printf "bind_ident_end_of_line %s\n" name;
  Scanf.bscanf sb "%0c" (function
  | '=' -> Scanf.bscanf sb "= " (bind_ident_end_of_line sb name) args
  | _ ->
    let e = Parser.parse_expression_to_end_of_line sb in
    Scanf.bscanf sb "%_[\n\r]"
      (Eval.htmlc_bind name args)
      (Eval.string_of_lxm e))
;;

let bind_ident parsing_mode =
  match parsing_mode with
  | Parser.End_of_line_mode -> bind_ident_end_of_line
  | Parser.Parsing_mode -> bind_ident_parsing
;;

let treat_caml_binding parsing_mode sb =
  (* We handle either:
     - a Caml like "let (rec) ident = expression(;;)" binding definition,
     - or a make file like "ident = expression" binding definition. *)
  let bind_ident = bind_ident parsing_mode in
  Scanf.bscanf sb " %0c" (function
  | 'a' .. 'z' | 'A' .. 'Z' | '_' ->
    begin match Parser.parse_ident_list sb with
      (* A let binding. We must use Parsing_mode to correctly handle it. *)
    | "let", "rec" :: fname :: (_ :: _ as args)
    | "let", fname :: (_ :: _ as args) ->
      (* A function with at least one argument. *)
      bind_ident_parsing sb fname args
    | "let", ident :: ([] as args) -> bind_ident_parsing sb ident args
      (* Not a let binding. We can use specified mode to handle it. *)
    | ident, ([] as args) -> bind_ident sb ident args
    | ident, args ->
      (* A function with arguments. *)
      bind_ident sb ident args end
  | c ->
    raise
      (Scanf.Scan_failure (Printf.sprintf "%C cannot start an ident" c)))
;;

let treat_make_binding parsing_mode sb =
  (* We handle an ``ident = expression'' binding definition. *)
  let bind_ident = bind_ident parsing_mode in
  (function
   | 'a' .. 'z' | 'A' .. 'Z' | '_' ->
     begin match Parser.parse_ident_list sb with
     | ident, ([] as args) -> bind_ident sb ident args
     | ident, args ->
       (* A function with arguments. *)
       bind_ident sb ident args end
   | c ->
     raise
       (Scanf.Scan_failure (Printf.sprintf "%C cannot start an ident" c)))
;;

(** {6 Loading environments} *)

let load_env_sb parsing_mode fname sb =
  Debug.printf "Loading environment %s." fname;
  try
    while true do
      Scanf.bscanf sb " %0c" (function
      (** Skip our own comments *)
      | '#' -> Scanf.bscanf sb "%_s@\n " ()
      (** Skip also a simplified version of Caml comments:
         "(\*.*\n". *)
      | '(' (* ')' helping emacs *) ->
        Scanf.bscanf sb "%2s" (function
        (** This is a Caml comment: skip it. *)
        | "(*" -> Parser.skip_caml_comments sb
        (** This is a syntax error: report it. *)
        | delim ->
          Scanf.bscanf sb "%s@\n" (fun line ->
            raise (Scanf.Scan_failure (Printf.sprintf "%s%s" delim line))))
      (** Otherwise find an ident, and call [bind_ident] to bind it. *)
      | _ -> treat_caml_binding parsing_mode sb)
    done with
  | End_of_file -> Debug.printf "Environment %s loaded.\n" fname
  | Scanf.Scan_failure s ->
    let mess = Scanf.bscanf sb "%l%n" (fun l n ->
      Printf.sprintf
        "file %s, line %d, character %d, wrong environment entry, %s"
        fname l n s) in
    Debug.failwith mess
;;

let load_env_file mode fname =
  let include_sb sb = load_env_sb mode fname sb in
  htmlc_load_included_file Mode_verbatim include_sb fname
;;

let bind_htmlc_definitions () =
  let bind_bindings = function
    | Htmlc_environment fname ->
      load_env_file Parser.Parsing_mode fname
    | Htmlc_bindings fname ->
      load_env_file Parser.End_of_line_mode fname
    | Htmlc_definition (ident, value) ->
      Eval.htmlc_bind_variable ident value in
  List.iter bind_bindings (get_htmlc_definitions ())
;;

let load_env_file_names fnames =
  List.iter add_env_file_name fnames;
  bind_htmlc_definitions ()
;;

(* Sizes of files. *)
let sizefmt_of_string = function
  | "bytes" -> S_bytes
  | "abbrev" -> S_abbrev
  | s -> unknown_qualifier "sizefmt" s
;;

let htmlc_fmts = { sizefmt = S_bytes; timefmt = "%A, %B %e, %Y"; };;

let htmlc_set_sizefmt unit_name =
  htmlc_fmts.sizefmt <- sizefmt_of_string (String.lowercase unit_name)
;;

let size_of_file_in_byte fname = (Unix.stat fname).Unix.st_size;;

let size_of_file encoding fname =
  let sz = size_of_file_in_byte fname in
  match encoding with
  | S_bytes -> string_of_int sz
  | S_abbrev ->
    if sz < 1000000
      then Printf.sprintf "%dKB" ((sz + 500) / 1000) else
    if sz < 1000000000
      then Printf.sprintf "%dMB" ((sz + 500000) / 1000000) else
    Printf.sprintf "%d.%dGB"
      (sz / 1000000000)
      ((sz mod 1000000000) / 100000000)
;;

let treat_fsize ob mode fname modifiers =
  match mode with
  | "file" as m ->
    let sizefmt =
      let is = Scanf.Scanning.from_string modifiers in
      Scanf.kscanf is
        (fun _sb _exc -> htmlc_fmts.sizefmt)
        " %[^= ] = %S"
        (fun mode encoding ->
         match String.lowercase mode with
         | "sizefmt" -> sizefmt_of_string (String.lowercase encoding)
         | _ -> unknown_qualifier m mode) in
    htmlc_add_string ob (size_of_file sizefmt fname)
  | _ -> unknown_modifier mode
;;

(** Echo variables in files. *)
let echo_var ob name modifiers =
  let encoding =
    let is = Scanf.Scanning.from_string modifiers in
    Scanf.kscanf is
      (fun _sb _exc -> E_none)
      " %[^= ] = %S"
      (fun mode encoding ->
       match String.lowercase mode with
       | "encoding" ->
         (match String.lowercase encoding with
          | "none" -> E_none
          | "url" -> E_url
          | "entity" -> E_entity
          | _ -> unknown_qualifier mode encoding)
       | _ -> unknown_modifier mode) in
  match encoding with
  | E_none -> htmlc_add_string ob name
  | E_url -> url_encode ob (htmlc_substitute_string name)
  | E_entity -> html_encode_escape ob (htmlc_substitute_string name)
;;

(* Time formats. *)
let timefmt_of_string s =
  warning "timefmt not yet supported";
  s
;;

let htmlc_set_timefmt name =
  htmlc_fmts.timefmt <- timefmt_of_string (String.lowercase name)
;;

(** The language may be set explicitly through:

   (0) setting the $lang variable into the source file,
   (1) setting the -lang option of htmlc,
   (2) setting the $LANG system variable,

   or

   (3) implicitly via the file name extension.

   The function [htmlc_get_lang] implements this logic (case
   (3) being obtained via module [Lang]), and sets the respective
   priorities of those different methods. *)

let htmlc_get_lang fname =
  try begin
     match Env.htmlc_get "lang" with
     | Env.Constant lang -> Lang.lang_of_string lang
     | Env.Function (_, _)
     | Env.Primitive (_, _) ->
       raise Not_found end with
  | Not_found -> Lang.get_with_file_name fname
;;

(** {6 Treatment of directives} *)

(** Still fake treatment of IF SSI directive. *)
let within_ssi_if, incr_within_ssi_if, decr_within_ssi_if =
  let within_ssi_if = ref 0 in
  (fun () -> !within_ssi_if > 0),
  (fun () -> incr within_ssi_if),
  (fun () -> decr within_ssi_if)
;;

let htmlc_default_line ob line _sb _exc =
  Debug.printf "htmlc_default_line: ``%s''\n" line;
  htmlc_add_string ob line
;;

(** {7 Non SSI directives} *)

let open_delimit_non_ssi_include ob fname =
  if Htmlc_options.get_delimit_non_ssi_include () then
    htmlc_add_string ob (Printf.sprintf "#(include ``%s''\n" fname)
;;

let close_delimit_non_ssi_include ob fname =
  if get_delimit_non_ssi_include () then
    htmlc_add_string ob (Printf.sprintf "#include ``%s'')\n" fname)
;;

let treat_non_ssi_directive treat_buffer ob line _sb _exc =
  Debug.printf "treat_non_ssi_directive: ``%s''\n" line;

  let treat_non_ssi_binding mode is c =
    Debug.printf "treat_non_ssi_binding: ``%c''\n" c;
    try
      treat_make_binding mode is c;
      if get_keep_non_ssi_binding () then htmlc_add_string ob line
    with
    | exc -> htmlc_default_line ob line is exc in

  let k_non_binding _sb exc =
    Debug.printf "k_non_binding: ``%s''\n" line;

    let is = Scanf.Scanning.from_string line in
    htmlc_default_line ob line is exc in

  let is = Scanf.Scanning.from_string line in

  let k_non_non_ssi_include _sb _exc =
    Debug.printf "k_non_non_ssi_include: ``%s''\n" line;

    let is = Scanf.Scanning.from_string line in

    if get_honor_non_ssi_binding () then
      Scanf.kscanf is
        k_non_binding
        "%0c"
        (fun c -> treat_non_ssi_binding Parser.End_of_line_mode is c) in

  let treat_non_ssi_include file_spec =
    let is = Scanf.Scanning.from_string file_spec in
    let fname_opt =
      try
        let e = Parser.parse_expression_to_end_of_line is in
        Some (Eval.htmlc_eval_lxm e)
      with
      | _ -> None in
    begin match fname_opt with
    | None ->
      (* We cannot understand the expression: emit a warning or fail ? *)
      ()
    | Some fname ->
      (** We include the file between
         ``#(include fname''
         and
         ``#include fname)''
         line markers. (In case we have been asked to do so.) *)
      open_delimit_non_ssi_include ob fname;
      htmlc_load_included_file Types.Mode_non_ssi (treat_buffer ob) fname;
      close_delimit_non_ssi_include ob fname
    end;
    if get_keep_non_ssi_include () then htmlc_add_string ob line in

  if get_honor_non_ssi_include () then
    Scanf.kscanf is
      k_non_non_ssi_include
      "include %s@!"
      treat_non_ssi_include else
  if get_honor_non_ssi_binding () then
    Scanf.kscanf is
      k_non_binding
      "%0c"
      (fun c -> treat_non_ssi_binding Parser.End_of_line_mode is c) else
  htmlc_add_string ob line
;;

let execute_command = Execute.htmlc_command htmlc_add_string;;
let execute_cgi = Execute.htmlc_cgi htmlc_add_string;;

(** {7 SSI directives} *)

(** In case we use expand mode SSI inclusion delimitors must be valid text
    comments, hence start with a sharp sign. *)
let add_text_comment fmt =
  if get_expand_mode () then "#" ^^  fmt else fmt
;;

let open_delimit_ssi_include ob fname =
  if get_delimit_ssi_include () then
    htmlc_add_string ob
      (Printf.sprintf (add_text_comment "<!-- (include ``%s'' -->\n") fname)
;;

let close_delimit_ssi_include ob fname =
  if get_delimit_ssi_include () then
    htmlc_add_string ob
      (Printf.sprintf (add_text_comment "<!-- include ``%s'') -->\n") fname)
;;

let rec scan_line_to_string src_name name =
  let ob = Buffer.create Configuration.line_buffer_length in
  scan_line src_name ob name;
  Buffer.contents ob

and scan_line src_name ob line =

  Debug.printf "scan_line: ``%s''\n" line;

  let is = Scanf.Scanning.from_string line in

  let k_non_ssi sb exc =
    if get_honor_non_ssi_directive () then
      treat_non_ssi_directive (treat_buffer src_name) ob line sb exc
    else htmlc_default_line ob line sb exc in

  Scanf.kscanf is

    k_non_ssi

    " <!-- # %s %[^= ] = %S%s@-->%[\000-\255]%!"

    (fun command mode name modifiers rest_of_line ->

     Debug.printf "scan_line: rest_of_line is ``%s''\n" rest_of_line;

     let treat_ssi_directive = function
       | "include" ->
         let include_virtual_file fname =
           htmlc_load_included_file Mode_ssi (treat_buffer src_name ob) fname in

         let treat_ssi_include_mode mode fname =
           match mode with
           | "verbatim" ->
             add_html_verbatim_file ob fname
           | "cat" ->
             add_verbatim_file ob fname
           | "environment" ->
             load_env_file Parser.Parsing_mode fname
           | "bindings" ->
             load_env_file Parser.End_of_line_mode fname
           | "virtual" ->
             include_virtual_file fname
           | _ -> include_virtual_file fname in

         let treat_ssi_include mode fname =
           open_delimit_ssi_include ob fname;
           treat_ssi_include_mode mode fname;
           close_delimit_ssi_include ob fname in

         treat_ssi_include mode name

       | "echo" as c ->
         let treat_ssi_echo mode name =
           match mode with
           | "var" ->
             begin match String.lowercase name with
             | "last_modified" ->
               let lang = htmlc_get_lang src_name in
               htmlc_add_string ob
                 (Date.last_modification_date_of_file lang src_name)
             | "date_local" ->
               let lang = htmlc_get_lang src_name in
                htmlc_add_string ob
                  (Date.local_date_of_day lang)
             | "date_gmt" ->
               let lang = htmlc_get_lang src_name in
               htmlc_add_string ob
                 (Date.date_of_day lang)
             | "user_name" ->
               let user =
                 try Sys.getenv "USER" with
                 | _ -> "<unknown>" in
               htmlc_add_string ob user
             | "document_path_info" ->
               htmlc_add_string ob (Filename.dirname src_name)
             | "document_name" ->
               htmlc_add_string ob (Filename.basename src_name)
             | "query_string_unescaped"
             | "document_uri"
             | _ -> echo_var ob name modifiers end
           | q -> unknown_qualifier c q in
         treat_ssi_echo mode name
       | "set"
       | "define" as c ->
         let value =
           if c = "define" then name else scan_line_to_string src_name name in
         let name = Eval.find_dollar_ident mode in
         Eval.htmlc_bind_variable name value
       | "exec" as c ->
         let treat_ssi_exec mode name =
           match mode with
           (** Should set the Unix environment of commands and CGIs before
              launching them:
              - for commands DOCUMENT_NAME, DOCUMENT_URI,
                DATE_LOCAL, DATE_GMT, LAST_MODIFIED should be available.
              - for CGI, in addition to those variables we should
                have the usual CGI's environment. *)
           | "cmd" -> execute_command src_name ob name
           | "cgi" -> execute_cgi src_name ob name
           | q -> unknown_qualifier c q in
         treat_ssi_exec mode name
       | "fsize" -> treat_fsize ob mode (htmlc_substitute_string name) modifiers
       | "if" | "elif" | "else" as c ->
         let treat_ssi_if mode _name =
           match mode with
           | "expr" ->
               incr_within_ssi_if ();
               Debug.failwith "Conditionals not yet implemented"
           | q -> unknown_qualifier c q in
         treat_ssi_if mode name
       | "endif" ->
         if not (within_ssi_if ())
         then Debug.failwith "endif not inside a conditional"
         else decr_within_ssi_if ()
       | "config" as c ->
         (** errmsg or timefmt or sizefmt (bytes or abbrev) *)
         begin match mode with
         | "timefmt" -> htmlc_set_timefmt name
         | "sizefmt" -> htmlc_set_sizefmt name
         | "errmsg" as q
         | q -> unknown_qualifier c q
         end
       | "flastmod" as c ->
         (** File last modification time *)
         let fname =
           match mode with
           | "file" -> htmlc_substitute_string src_name
           | q -> unknown_qualifier c q in
         let lang = htmlc_get_lang fname in
         htmlc_add_string ob (Date.last_modification_date_of_file lang fname)
       | "printenv" as c ->
         (** Prints some parts of (or the entire) environment
            (of the HTTP server process ?) *)
         begin match mode with
         | "htmlc" -> Env.print_env ob name
         | q -> unknown_qualifier c q
         end
       | c -> unknown_command c in

     let treat_ignored_ssi_directive s =
       Scanf.sscanf s
         "%s@<!--%s@#%s@=%[^\"]%S%s@-->%_[\000-\255]%!"
         (fun beginning_of_line before_command command mode name modifiers ->
          let verbatim_ssi =
            Printf.sprintf
              "%s<!--%s#%s=%s%S%s-->"
              beginning_of_line before_command command mode name modifiers in
          htmlc_add_string ob verbatim_ssi) in

     let evaluate_ssi_directive command =
       if get_honor_ssi_directive ()
       then treat_ssi_directive command
       else treat_ignored_ssi_directive line in

     (** If we know the command, we treat it
        and go on with the rest of the line. *)
     evaluate_ssi_directive command;
     if rest_of_line <> "" then scan_line src_name ob rest_of_line)

(** The main routine to expand files. *)
and treat_buffer src_name ob (sb : Scanf.Scanning.scanbuf) =

  Debug.printf "Treat_buffer.\n";

  (*let sb = Scanf.bscanf sb "%[\000-\255]" (fun s ->
     Debug.printf "Next thing to scan for htmlc is \"%s\"\n" s;
     Scanf.Scanning.from_string s) in*)

  try
    while not (Scanf.Scanning.end_of_input sb) do
      (** Being careful for the case of the last line of the file,
         when this line does not end with a new line character.
         Also carefully treat escaped end of lines in the input. *)
      let line = Parser.get_line sb in
      scan_line src_name ob line
    done
  with
  | x ->
    raise (
      Error (
        src_name,
        Scanf.bscanf sb "%l" (fun line_number -> line_number),
        x)
    )
;;

(** {6 Compiling a file} *)

let treat_bufferized_file src_name ob sb =
  let sb = Preprocessor.call_source_preprocessors src_name sb in

  (*let sb =
    Scanf.bscanf sb "%[\000-\255]" (fun s ->
      Debug.printf "Next thing to scan for htmlc is \"%s\"\n" s;
      Scanf.Scanning.from_string s) in*)

  treat_buffer src_name ob sb;

  let os = Buffer.contents ob in
  Debug.printf "Treat_file_channel: \"%s\" returned\n\"%s\"\n" src_name os;
  os
;;

let treat_file_channel src_name ob sb oc =
  let os = treat_bufferized_file src_name ob sb in
  output_string oc os;
  flush oc
;;

let htmlc_set_output_permissions d =
  match d with
  | Io_domain.Io_string _ | Io_domain.Io_standard -> ()
  | Io_domain.Io_file fname ->
    if fname <> "-" then begin
      match get_target_perms () with
      | None ->
        if get_compile_to_read_only () then File.make_read_only_file fname
      | Some perms -> File.set_file_permissions fname perms
    end
(** Set the permissions of an output domain, if possible. *)
;;

let treat_domains =

  let ob = Buffer.create Configuration.file_buffer_length in

  (fun src tgt ->
   Buffer.clear ob;
   let src_name = Io_domain.name_in src in
   let ic = Io_domain.open_in src in
   let sb =
     match src with
     | Io_domain.Io_file _ -> Scanf.Scanning.from_channel ic
     | Io_domain.Io_standard -> Scanf.Scanning.stdib
     | Io_domain.Io_string str -> Scanf.Scanning.from_string str in

   try
     let perms = Io_domain.permissions_in src in
     let temp_d = Io_domain.temporary_target ".htmlc" src tgt in
     let oc = Io_domain.open_out perms temp_d in
     at_exit
       (fun () -> Io_domain.remove_temporary_target temp_d tgt);

     try

       treat_file_channel src_name ob sb oc;

       Io_domain.close_in src ic;
       Io_domain.close_out temp_d oc;

       htmlc_set_output_permissions temp_d;

       Io_domain.rename temp_d tgt with
        (* In any case, we remove the partial file obtained by aborted
           execution. *)
     | x ->
       Debug.printf "Treat_domains: spurious exception %s\n"
         (Printexc.to_string x);
       Io_domain.remove tgt;
       raise x with
   (* In any case, close the channel reading from src_name. *)
   | x -> Io_domain.close_in src ic; raise x
  )
;;
