(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2008,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: file.mli,v 1.7 2011-05-05 09:05:04 weis Exp $ *)

(** {3 Operations on files} *)

(** {6 Data types for file names} *)

type file_name = string
and dir_name = string
and explicit_file_name = string;;
(** The name of a file, the name of a directory, and the explicit name of a
    file. *)

type output_file_name = string
(** An alias to distinguish output file names. *)
;;

type input_file_name = string
(** An alias to distinguish input file names. *)
;;

type file_extension = string
(** An alias to distinguish file name extensions from strings. *)
;;

type file_suffix = string
(** An alias to distinguish file name suffixes from strings. *)
;;

type suffix_marker = char
(** An alias for the character that indicates the beginning of a file suffix.
  The suffix of a file name starts with the last [suffix_marker] character in
  file name and spreads to the end of the file name. *)
;;

(** {6 Designating file names} *)

val name_out : output_file_name -> file_name
(** Returns the name of output file name. *)
;;

val name_in : input_file_name -> file_name
(** Returns the name of an input file name. *)
;;

(** {6 Getting and setting file permissions} *)

val permissions : file_name -> Unix.file_perm
(** Returns the file permission of a file name. *)
;;

val permissions_out : output_file_name -> Unix.file_perm
(** Returns the file permission of an output file name. *)
;;

val permissions_in : input_file_name -> Unix.file_perm
(** Returns the file permission of an input file name. *)
;;

val make_writable_file : file_name -> unit
(** Turn a file writable at least for the owner.

   Land the current permission with 0o7577 to remove the writable flag to
   the owner, then add 0o200 to get the file writable by the owner.
   Note: if the file was already writable for others or the group, we keep
   this permission. *)
;;

val make_read_only_file : file_name -> unit
(** Turn a file unwritable for anybody.

   Land the current permission with 0o7555 to remove the writable flag to
   anybody.
   Note: if the file was already writable for others or the group, we
   suppress this permission. *)
;;

val set_file_permissions : file_name -> Unix.file_perm -> unit
(** [set_file_permissions perm fname] sets the file permissions of file
    [fname] to [perm]. *)
;;

val is_readable_directory : file_name -> bool
;;

val is_readable_file : file_name -> bool
;;

val is_executable_file : file_name -> bool
;;

(** {6 Primitives on channels cautiously dealing with [sdin] and [stdout]} *)

(** stdout and stdin are (conventionally) associated to the file name
   "-". We redefine open, close, and remove to implement this convention. *)

val remove : file_name -> unit
(** Removing a file: we ensure the file is writable before removing it. *)
;;

val rename : file_name -> file_name -> unit
(** Rename a file into another one.
  Nothing happens when arguments are equal.
  Raise [Failure] when one (and only one) of the arguments is a standard
  channel. *)
;;

val open_out_with_perms : Unix.file_perm -> file_name -> out_channel
(** Open out for writing: we ensure the given permissions and then
  ensure the file is writable. *)
;;
val open_out : file_name -> out_channel
(** Open out for writing: we ensure the file is writable. *)
;;

val close_out : file_name -> out_channel -> unit
(** Close an [out_channel] without closing [stdout]. *)
;;

val open_in : file_name -> in_channel
(** Open in for reading: we ensure the file is readable and return a
  [Pervasives.in_channel] without trying to reopen [stdin]. *)
;;

val close_in : file_name -> in_channel -> unit
(** Close an [in_channel] without closing [stdin]. *)
;;

(** {6 Dealing with file name extensions and suffixes.} *)

val get_extension : file_name -> file_extension
(** Returns the extension of a file name if any. *)
;;

val chop_extension : file_name -> file_name
(** Remove the extension of a file name if any. *)
;;

val add_extension : file_extension -> file_name -> file_name
(** Add the given extension to a file name if not a standard channel. *)
;;

val get_suffix : suffix_marker -> file_name -> file_suffix
(** Returns the suffix of a file name if any.
  The suffix of a file name starts with the last [suffix_marker] character
  in file name and spreads to the end of the file name. *)
;;

val chop_suffix : suffix_marker -> file_name -> file_name
(** Remove the suffix of a file name if any. *)
;;
val add_suffix : file_suffix -> file_name -> file_name
(** Add the given suffix to a file name if not a standard channel. *)
;;
val add_suffix_extension : file_suffix -> file_extension -> file_name -> file_name
(** Remove the extension of the file name argument, then add the given suffix
   and the given extension. *)
;;

(** {6 Allocating temporary files} *)

val temporary_target :
  file_extension -> file_name -> file_name -> file_name
(** [File.temporary_target ext src_name tgt_name] return a temporary file name
  suitable to run a file processor from [src_name] to [tgt_name], preserving
  [src_name] file integrity.
  The result can be [tgt_name], or even [src_name], provided that [src_name]
  file integrity is preserved; otherwise, it is a new [file_name] with extension
  [ext]. *)
;;

val remove_temporary_target : file_name -> file_name -> unit
(** Remove the temporary file name [temp_file_name] when it is not the
  [tgt_name] file. *)
;;
