(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: file.ml,v 1.16 2011-05-05 09:05:04 weis Exp $ *)

(** {3 Operations on files} *)

(** {6 Data types for file names} *)

type file_name = string
and dir_name = string
and explicit_file_name = string;;
(** The name of a file, the name of a directory, and the explicit name of a
    file. *)

type output_file_name = string
(** An alias to distinguish output file names. *)
;;

type input_file_name = string
(** An alias to distinguish input file names. *)
;;

type file_extension = string
(** An alias to distinguish file name extensions from strings. *)
;;

type file_suffix = string
(** An alias to distinguish file name suffixes from strings. *)
;;

type suffix_marker = char
(** An alias for the character that indicates the beginning of a file suffix.
  The suffix of a file name starts with the last [suffix_marker] character in
  file name and spreads to the end of the file name. *)
;;

(** {6 Designating file names} *)

let name_out = function
  | "-" -> "stdout"
  | ofname -> ofname
;;

let name_in = function
  | "-" -> "stdin"
  | ifname -> ifname
;;

(** {6 Getting and setting file permissions} *)

let permissions fname =
  if fname = "-" then failwith "File.permissions: ambiguous file ``-''" else
  let stats = Unix.stat fname in
  stats.Unix.st_perm
(** Returns the file permission of a file name. *)
;;

let permissions_out ofname =
  if ofname = "-" then 0o666 (* = rw-rw-rw- for stdout *)
  else permissions ofname
(** Returns the file permission of an output file name. *)
;;

let permissions_in ifname =
  if ifname = "-" then 0o444 (* = r--r--r-- for stdin *)
  else permissions ifname
(** Returns the file permission of an input file name. *)
;;

let cautious_chmod fname perm =
  try Unix.chmod fname perm with
  | Unix.Unix_error (err, _, _) ->
    Debug.warning (Printf.sprintf "cannot chmod %s: %s" fname (Unix.error_message err))
;;

let gen_set_file_permissions cautious fname perm =
  if fname <> "-" then
  let fperm = permissions fname in
  if perm <> fperm then
    let chmod = if cautious then cautious_chmod else Unix.chmod in
    chmod fname perm
(** We set file permissions of file [fname] to [perm].
    Note: we explicitely do nothing if the file permission is already the
    permission we want to set: this avoid some corner cases such as /dev/null
    (see [make_writable_file] below) *)
;;

let cautious_set_file_permissions = gen_set_file_permissions true
(** Cautious version of [gen_set_file_permissions]: use the [cautious_chmod]
    above, hence just emits a warning in case of failure of the [Unix.chmod]
    primitive. *)
;;

let set_file_permissions = gen_set_file_permissions false
(** Uncautious version of [gen_set_file_permissions]: directly use the
    [Unix.chmod] primitive. Hence could fail in case of invalid chmod
    attempt. *)
;;

let make_writable_file fname =
  let fperm = permissions fname in
  let nperm = (0o7577 land fperm) lor 0o0200 in
  set_file_permissions fname nperm;
(** We turn a file writable at least for the owner.

   We land the current permission with 0o7577 to remove the writable flag to
   the owner, then we add 0o200 to get the file writable by the owner.

   Note: if the file was already writable for others or the group, we keep
   this permission.

   Note: using [set_file_permissions], we explicitely do nothing in case the
   new permission is the same as the file permission: certain OSes do not
   allow setting permission of some files, even if you ask for setting the
   already existing permission of the file (e.g. /dev/null under BSD).
*)
;;

let make_read_only_file fname =
  let fperm = permissions fname in
  let nperm = 0o7555 land fperm in
  cautious_set_file_permissions fname nperm
(** We turn a file unwritable for anybody.

   We land the current permission with 0o7555 to remove the writable flag to
   anybody.

   Note: if the file was already writable for others or the group, we
   suppress this permission. *)
;;


let is_executable_perm perm =
  (perm land 0o0200) <> 0
;;

let is_readable_perm perm =
  (perm land 0o0600) <> 0
;;

let is_executable_file fname =
  Sys.file_exists fname &&
  let fperm = permissions fname in
  is_executable_perm fperm
;;

let is_readable_file fname =
  Sys.file_exists fname &&
  let fperm = permissions fname in
  is_readable_perm fperm
;;

let is_readable_directory dname =
  Sys.file_exists dname &&
  Sys.is_directory dname &&
  let fperm = permissions dname in
  is_executable_perm fperm &&
  is_readable_perm fperm
;;

(** {6 Primitives on channels cautiously dealing with [sdin] and [stdout]} *)

(** stdout and stdin are (conventionally) associated to the file name
   "-". We redefine open, close, and remove to implement this convention. *)

let remove fname =
  if fname <> "-" then begin
    if Sys.file_exists fname then begin
      make_writable_file fname;
      Sys.remove fname
    end
  end
(** Removing a file: we ensure the file is writable before removing it. *)
;;

let rename src tgt =
  if src <> tgt then
    if src = "-" || tgt = "-" then
      failwith "File.rename: only one argument is a standard channel"
    else begin
      remove tgt;
      Sys.rename src tgt;
    end
(** Rename a file into another one.
  Nothing happens when arguments are equal.
  Raise [Failure] when one (and only one) of the arguments is a standard
  channel. *)
;;

let open_out fname =
  if fname = "-" then stdout else begin
    (try make_writable_file fname with _ -> ());
    Pervasives.open_out fname
  end
(** Open out for writing: we ensure the file is writable. *)
;;

let open_out_with_perms perms fname =
  if fname = "-" then stdout else begin
    let oc = open_out fname in
    cautious_set_file_permissions fname perms;
    make_writable_file fname;
    oc
  end
(** Open out for writing: we ensure the file is writable. *)
;;

let close_out fname oc =
  if fname <> "-" then Pervasives.close_out oc
(** Close an [out_channel] without closing [stdout]. *)
;;

let open_in fname =
  if fname = "-" then stdin else
  if is_readable_file fname then Pervasives.open_in fname else
  failwith
    (Printf.sprintf "File.open_in: file %s is not readable" fname)
(** Opening an [in_channel] without trying to reopen [stdin]. *)
;;

let close_in fname ic =
  if fname <> "-" then Pervasives.close_in ic
(** Close an [in_channel] without closing [stdin]. *)
;;

(** {6 Dealing with file extensions} *)

let get_suffix suffix_marker fname =
  try
    let pos = String.rindex fname suffix_marker in
    String.sub fname pos (String.length fname - pos)
  with Not_found -> ""
(** Find the extension of a file name if any. *)
;;

let chop_suffix suffix_marker fname =
  try String.sub fname 0 (String.rindex fname suffix_marker)
  with Not_found -> fname
;;

let add_suffix suffix fname =
  if fname = "-" then fname else fname ^ suffix
(** Add the given suffix to a file name if not a standard channel. *)
;;

let get_extension = get_suffix '.'
(** Find the extension of a file name if any. *)
;;

let chop_extension = chop_suffix '.'
(** Remove the extension of a file name if any. *)
;;

let add_extension = add_suffix
(** Add the given extension to a file name if not a standard channel. *)
;;

let add_suffix_extension suffix ext fname =
  if fname = "-" then fname else
  let name = chop_extension fname in
  name ^ suffix ^ ext
(** Insert the given suffix between the file name and its extension
   (if any), then add the new given extension. *)
;;

let temporary_target ext src_name tgt_name =
  if tgt_name <> src_name || tgt_name = "-" then tgt_name else
  add_extension ext src_name
(** Return a temporary file name with extension [ext], suitable to run a file
  processor from [src_name] to [tgt_name], preserving [src_name] file
  integrity. *)
;;

let remove_temporary_target temp_file_name tgt_name =
  if temp_file_name <> tgt_name then remove temp_file_name
(** Remove the temporary file name [temp_file_name] when it is not the
  [tgt_name] file. *)
;;
