(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: eval.ml,v 1.10 2011-03-23 18:39:05 weis Exp $ *)

(** {3 Htmlc evaluation machinery.} *)

let string_of_number f =
 let i = int_of_float f in
 let ifl = float_of_int i in
 if ifl = f then string_of_int i else string_of_float f
(** Given a floating point number, try to figure out if it is indeed an
   integer to return the corresponding string. Otherwise returns the
   (Caml compatible) string representation of the number. *)
;;

let string_of_lxm_marker marker = function
  | Parser.Number f -> string_of_number f
  | Parser.String s -> s
  | Parser.Char s -> s
  | Parser.Bool s -> s
  | Parser.Ident s -> Printf.sprintf "%c(%s)" marker s
  | Parser.Quoted s -> Printf.sprintf "%c('%s')" marker s
  | Parser.Forced s -> Printf.sprintf "%s" (Parser.unquote s)
  | Parser.Symbol _
  | Parser.Let | Parser.Rec | Parser.Quote
  | Parser.Eq | Parser.SemiSemi
(*   | Parser.If | Parser.Then | Parser.Else*)
  | Parser.Eof -> failwith "string_of_lxm: cannot evaluate a symbol."
;;

let string_of_lxm lxm =
  let marker = Htmlc_options.get_marker_char () in
  string_of_lxm_marker marker lxm
;;

let rec htmlc_eval_string s =
  let b = Buffer.create (String.length s) in
  Substitute.add_string htmlc_subst b s;
  Buffer.contents b
(** Specialized buffer handling functions. *)

and htmlc_subst ident =
  let marker = Htmlc_options.get_marker_char () in

  Debug.printf "htmlc_subst: ``%s'' is " ident;
  Debug.printf "%S\n"
    (try
       match Env.htmlc_get ident with
       | Env.Constant s -> s
       | Env.Function (args, body) ->
         Printf.sprintf "<fun %s -> %s>"
           (String.concat " " args) body
       | Env.Primitive (args, _) ->
         Printf.sprintf "<prim %s -> <fun>>"
           (String.concat " " args) with
     | Not_found -> Printf.sprintf "%s" "unbound in environment");

  if Substitute.is_quoted_string ident then ident else
  if Substitute.is_forced_string ident
    then htmlc_eval_string (Parser.unforce_string ident) else

  let ib = Scanf.Scanning.from_string ident in
  let lxm, lxms = Parser.parse_expression_list marker ib in

  Debug.printf "htmlc_subst: ``%s'' is read as %s.\n"
    ident
    (String.concat "||"
      (List.map (Parser.string_of_lexeme marker) (lxm :: lxms)));

  let ident =
    match lxm with
    | Parser.Ident ident -> ident
    | Parser.Number _
    | Parser.String _
    | Parser.Char _
    | Parser.Bool _
    | Parser.Symbol _
    | Parser.Quoted _
    | Parser.Forced _
    | Parser.Let | Parser.Rec | Parser.Quote
    | Parser.Eq | Parser.SemiSemi
    (*   | Parser.If | Parser.Then | Parser.Else*)
    | Parser.Eof ->
      failwith (Printf.sprintf "%s is not a valid identifier." ident) in

  Debug.printf "htmlc_subst: ``%s'' is a regular ident." ident;

  match Env.htmlc_get ident with
  | Env.Constant s ->
    Debug.printf "bound to %S.\n" s;
    s

  | Env.Primitive (_ids, body) ->
    Debug.printf "bound to a primitive.\n";
    body (htmlc_eval_lxms lxms)

  | Env.Function (ids, body) ->
    Debug.printf "bound to a function.\n";
    Debug.printf "htmlc_subst: applying a function with %i arguments.\n"
      (List.length ids);
    Debug.printf "htmlc_subst: and body %S.\n" body;
    Debug.printf "htmlc_subst: applying to %i values.\n"
      (List.length lxms);

    let bind () =
      let vals = htmlc_eval_lxms lxms in
      List.iter2
        (fun id s -> Env.htmlc_set id (Env.Constant s)) ids vals in
    let unbind () =
      List.iter Env.htmlc_remove ids in
    try
      let s =
        bind ();
        htmlc_eval_string body in
      unbind ();
      s with
    | x ->
      unbind ();
      raise x
(** Substitute a $ident or $(ident) or $(f arg1 arg2 ... argn)
 to some value (i.e. a string). *)

and htmlc_eval_lxm lxm = htmlc_eval_string (string_of_lxm lxm)

and htmlc_eval_lxms lxms =
  List.map htmlc_eval_lxm lxms
;;

let htmlc_bind ident args value =
  Debug.printf "htmlc_bind: %s" ident;
  Debug.printf " to %S" value;
  Debug.printf " = %S\n"
    (match args with
    | [] -> htmlc_eval_string value
    | _ -> "<fun>");

  match args with
  | [] -> Env.htmlc_set ident (Env.Constant (htmlc_eval_string value))
  | args -> Env.htmlc_set ident (Env.Function (args, value))
(** Binding a variable to a value that can be a function. *)
;;

let htmlc_bind_variable ident value = htmlc_bind ident [] value;;
(** Binding a variable to a simple value which is not a function. *)

let add_env_name, add_env_value, bind_names_to_values =
  let names = ref [] in
  let values = ref [] in
  (fun s -> names := s :: !names),
  (fun s -> values := s :: !values),
  (fun () ->
    List.iter2
      (fun name v -> htmlc_bind_variable name v)
      (List.rev !names) (List.rev !values);
    names := [];
    values := [])
(** Bind pairs (name, value) found in a file environment. *)
;;

let find_dollar_ident =
 let htmlc_var = Buffer.create 100 in
 function name ->
   Substitute.add_string (fun s -> s) htmlc_var name;
   let s = Buffer.contents htmlc_var in
   Buffer.clear htmlc_var;
   s
(** Find a dollar ident into a name found in the input. *)
;;

let htmlc_quote_marker marker ident =
  let ib = Scanf.Scanning.from_string ident in
  let _lxm, lxms = Parser.parse_expression_list marker ib in
  match lxms with
  (* A simple identifier. *)
  | [] -> Printf.sprintf "\\\\%c%s" marker ident
  (* A function application,
     or something more complex than an identifier: use parens. *)
  | _ -> Printf.sprintf "\\\\%c(%s)" marker ident
;;
