(***********************************************************************)
(*                                                                     *)
(*                           Htmlc                                     *)
(*                                                                     *)
(*                    Pierre Weis, INRIA Rocquencourt                  *)
(*                                                                     *)
(*  Copyright 1997-2007,                                               *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* $Id: env.ml,v 1.17 2011-11-30 18:26:17 weis Exp $ *)

(** {3 Definition and operations on the Htmlc environment.} *)

type ident = Types.ident;;
(** Variables environment to expand into files. *)

type htmlc_value =
   | Constant of string
   | Function of ident list * string
   | Primitive of ident list * (string list -> string)
(** Htmlc values computed to bind to variables. *)
;;

let string_of_htmlc_value = function
  | Constant s -> s
  | Function _ -> "<fun>"
  | Primitive _ -> "<prim>"
;;

let htmlc_env = Hashtbl.create 17;;
(** The htmlc environment. *)

let htmlc_get = Hashtbl.find htmlc_env;;

let htmlc_set name value =
Debug.printf "Binding %s to %s\n" name (string_of_htmlc_value value);
  Hashtbl.add htmlc_env name value
;;

let htmlc_remove name =
Debug.printf "Unbinding %s (was %s).\n"
  name (string_of_htmlc_value (htmlc_get name));
  Hashtbl.remove htmlc_env name
;;
(** Necessary to evaluate function bodies. *)

let hashtbl_get_all_bindings t =
  let bds = ref [] in
  Hashtbl.iter (fun k v -> bds := (k, v) :: !bds) t;
  !bds
;;

let hashtbl_get_all_sorted_bindings t =
  List.sort (fun (k, _) (k', _) -> compare k k') (hashtbl_get_all_bindings t)
;;

let htmlc_get_all_bindings () =
  hashtbl_get_all_bindings htmlc_env

and htmlc_get_all_sorted_bindings () =
  hashtbl_get_all_sorted_bindings htmlc_env
;;

open Date;;

let year_of_day () =
  year_of_system_time (Date.system_time_of_day ())
and month_of_day () =
  month_of_system_time (system_time_of_day ())
and month_day_of_day () =
  month_day_of_system_time (system_time_of_day ())
;;

let time_of_day () =
  time_of_system_time (system_time_of_day ())
and local_time_of_day () =
  time_of_system_time (local_system_time_of_day ())
;;

let long_date_of_day_Fr () =
  long_date_of_system_time Lang.Fr (system_time_of_day ())
and long_local_date_of_day_Fr () =
  long_date_of_system_time Lang.Fr (local_system_time_of_day ())
;;

let long_date_of_day_Uk () =
  long_date_of_system_time Lang.Uk (system_time_of_day ())
and long_local_date_of_day_Uk () =
  long_date_of_system_time Lang.Uk (local_system_time_of_day ())
;;

let short_date_of_day_Fr () =
  short_date_of_system_time Lang.Fr (system_time_of_day ())
and short_local_date_of_day_Fr () =
  short_date_of_system_time Lang.Fr (local_system_time_of_day ())
;;

let short_date_of_day_Uk () =
  short_date_of_system_time Lang.Uk (system_time_of_day ())
and short_local_date_of_day_Uk () =
  short_date_of_system_time Lang.Uk (local_system_time_of_day ())
;;

let iso_date () =
  iso_date_of_system_time (system_time_of_day ())
and iso_local_date () =
  iso_date_of_system_time (local_system_time_of_day ())
;;

let iso_time () =
  iso_time_of_system_time (system_time_of_day ())
and iso_local_time () =
  iso_time_of_system_time (local_system_time_of_day ())
;;

let full_iso_date () =
  full_iso_date_of_system_time (system_time_of_day ())
;;

let host () = Execute.command "hostname";;

let htmlc_init_env () =
  let htmlc_bind_variable_to_string ident v = htmlc_set ident (Constant v) in
  htmlc_bind_variable_to_string "htmlc_version" Version.htmlc_version;
  htmlc_bind_variable_to_string "htmlc_long_version" Version.htmlc_long_version;
  let this_year = year_of_day () in
  let this_month = month_of_day () + 1 in
  let this_month_day = month_day_of_day () in
  htmlc_bind_variable_to_string "this_year" (string_of_int this_year);
  htmlc_bind_variable_to_string "next_year" (string_of_int (this_year + 1));
  htmlc_bind_variable_to_string "last_year" (string_of_int (this_year - 1));
  htmlc_bind_variable_to_string "this_month" (string_of_int this_month);
  htmlc_bind_variable_to_string "this_month_day" (string_of_int this_month_day);
  htmlc_bind_variable_to_string "la_date" (date_of_day_Fr ());
  htmlc_bind_variable_to_string "la_date_locale" (local_date_of_day_Fr ());
  htmlc_bind_variable_to_string "l_heure" (time_of_day ());
  htmlc_bind_variable_to_string "l_heure_locale" (local_time_of_day ());
  htmlc_bind_variable_to_string "la_date_longue" (long_date_of_day_Fr ());
  htmlc_bind_variable_to_string "la_date_locale_longue" (long_local_date_of_day_Fr ());
  htmlc_bind_variable_to_string "la_date_courte" (short_date_of_day_Fr ());
  htmlc_bind_variable_to_string "la_date_locale_courte" (short_local_date_of_day_Fr ());
  htmlc_bind_variable_to_string "the_date" (date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_local_date" (local_date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_time" (time_of_day ());
  htmlc_bind_variable_to_string "the_local_time" (local_time_of_day ());
  htmlc_bind_variable_to_string "the_long_date" (long_date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_long_local_date" (long_local_date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_short_date" (short_date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_short_local_date" (short_local_date_of_day_Uk ());
  htmlc_bind_variable_to_string "the_iso_date" (iso_date ());
  htmlc_bind_variable_to_string "the_iso_local_date" (iso_local_date ());
  htmlc_bind_variable_to_string "the_iso_time" (iso_time ());
  htmlc_bind_variable_to_string "the_iso_local_time" (iso_local_time ());
  htmlc_bind_variable_to_string "the_full_iso_date" (full_iso_date ());
  htmlc_bind_variable_to_string "host" (host ());
(** Predefined bindings: the Htmlc's version of time, year, and date values. *)
;;

let unknown s1 s2 = Debug.failwith (Printf.sprintf "unknown %s %s" s1 s2);;

let print_env_gen get_all_bindings b modifiers =
  let print_args b = function
  | [] -> ()
  | arg :: args ->
    Printf.bprintf b "%s" arg;
    List.iter (fun arg -> Printf.bprintf b " %s" arg) args in
  let print_binding (k, v) =
    match v with
    | Constant v ->
      Printf.bprintf b "     let %s = %S;;\n" k v
    | Function (args, v) ->
      Printf.bprintf b "     let %s %a = %s;;\n" k print_args args v
    | Primitive (args, _f) ->
      Printf.bprintf b "external %s %a = htmlc_%s_prim %a;;\n"
        k print_args args k print_args args in
  let bds =
    match modifiers with
    (* Since ``_'' cannot be the name of a bound identifier, an ``_'' bound
       name means the entire list of the htmlc bindings. *)
    | "_" -> get_all_bindings ()
    (* We suppose only one name in the list (FIXME). *)
    | s ->
      try let v = htmlc_get s in [s, v]
      with
      | Not_found -> unknown "variable" (Printf.sprintf "%S" s) in
  List.iter print_binding bds
;;

let print_env = print_env_gen htmlc_get_all_bindings
(* We print known bindings without sorting. *)

and print_sorted_env = print_env_gen htmlc_get_all_sorted_bindings
(* We print known bindings after sorting. *)
;;

let debug_print_env_gen get_all_bindings () =
  if Htmlc_options.get_print_env () then begin
    Debug.printf "\nPrinting the current HTMLC environment:\n";
    let b = Buffer.create 2048 in
    print_env_gen get_all_bindings b "_";
    Printf.fprintf stderr "%s\n" (Buffer.contents b);
    flush stderr;
  end
;;

let debug_print_sorted_env =
  debug_print_env_gen htmlc_get_all_sorted_bindings
;;
