(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: date.mli,v 1.14 2012-05-18 14:21:31 weis Exp $ *)

type file_name = string
;;

(** {3 Getting the system time of day} *)

(** A module to deal with Greenwich meridian time (also known as GMT or UTC)
  or local dates in various natural languages. *)

(** {6 Type definitions for time and dates. *)

type system_time = Unix.tm
(** The time as reported by the {!Unix} module. *)
;;

type date = string
and digital_date = string
and iso_date = string
and full_iso_date = string

and time = string
and iso_time = string
(** A nice aliasing for the string representations of dates and times. *)
;;

type year = int
and month = int
and month_day = int
and week_day = int
(** A nice aliasing for the numerical components of a date. *)
;;

type hour = int
and minute = int
and second = int
(** A nice aliasing for the numerical components of a time. *)
;;

(** {6 Greenwich meridian time dates.} *)

val date_of_day : Lang.t -> date;;
(** Return the current date as a string according to the [lang] argument.
    Assumes Greenwich meridian time zone. *)

val date_of_day_Uk : unit -> date;;
val date_of_day_Fr : unit -> date;;
(** The string representation of the current system time of day, respectively
    in english and french. Assumes Greenwich meridian time zone. *)

val date : unit -> date;;
 (** Synonym for [date_of_day_Uk]. *)

val last_modification_date_of_file : Lang.t -> file_name -> date;;
(** Last modification time of a file in the given language.
    Returns [system_time_of_day ()], if file name is ["-"].
    Assumes Greenwich meridian time zone. *)

(** {6 Local system time dates.} *)

val local_date_of_day : Lang.t -> date;;
(** Return the current date as a string according to the [lang] argument.
    Assumes the local time zone. *)

val local_date_of_day_Uk : unit -> date;;
val local_date_of_day_Fr : unit -> date;;
 (** The string representation of the current system time of day, respectively
    in english and french. Assumes the local time zone. *)

val local_date : unit -> date;;
(** Synonym for [local_date_uk]. *)

(** {6 Basic primitives on dates.} *)

val year_of_system_time : system_time -> year;;
val month_of_system_time : system_time -> month;;
val month_day_of_system_time : system_time -> month_day;;
val week_day_of_system_time : system_time -> week_day;;
val hour_of_system_time : system_time -> hour;;
val minute_of_system_time : system_time -> minute;;
val second_of_system_time : system_time -> second;;

val system_time_of_day : unit -> system_time;;
 (** Return the system time of day as a Unix system time.
    Assumes Greenwich meridian time zone. *)

val local_system_time_of_day : unit -> system_time;;
 (** Return the system time of day as a Unix system time.
    Assumes local time zone. *)

val last_modification_system_time_of_file : file_name -> system_time;;
 (** Last modification system time of a file.
    Returns [system_time_of_day ()], if file name is ["-"]. *)

val date_of_system_time : Lang.t -> system_time -> date;;
 (** Translate a Unix system time to a string according to the [lang]
    argument. *)

val time_of_system_time : system_time -> time;;
 (** Returns a string representation of the system time of day in the following
   format: ["hh:mm:ss"] where ["hh"] is the hour, ["mm"] the minutes and ["ss"]
   the seconds. *)

val long_date_of_system_time : Lang.t -> system_time -> date;;
 (** Returns a string representation of the system time of day according to the
   [Lang.t] argument.
   [long_date_of_system_time lang tm] is equivalent to
   [date_of_system_time lang tm ^ time_of_system_time tm]. *)

val short_date_of_system_time : Lang.t -> system_time -> date;;
 (** Returns a string representation of the system time of day according to the
   [Lang.t] argument.
   [short_date_of_system_time lang tm] is equivalent to
   [date_of_system_time lang tm]. *)

val digital_date_of_system_time : Lang.t -> system_time -> digital_date;;
 (** Returns a digital representation of the system time of day according to the
   [Lang.t] argument. *)

val iso_date_of_system_time : system_time -> iso_date;;
 (** Returns a string representation compliant to the ISO 8601 standard of
 the date of the system time. *)

val iso_time_of_system_time : system_time -> iso_time;;
 (** Returns a string representation compliant to the ISO 8601 standard of
 the time of the system time. *)

val full_iso_date_of_system_time : system_time -> full_iso_date;;
 (** Returns a string representation compliant to the ISO 8601 standard of
 the date and time of the system time. *)

val format_date_Uk : int -> int -> int option -> int -> string;;
 (** [format_date_Uk] is [Lib_date.format_date Uk]. *)
val format_date_Fr : int -> int -> int option -> int -> string;;
 (** [format_date_Fr] is [Lib_date.format_date Fr]. *)
