(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: date.ml,v 1.12 2010-05-12 08:18:15 fclement Exp $ *)

open Unix;;
open Lib_date;;
open Lang;;

type file_name = string;;

type system_time = Unix.tm;;

type date = string
and digital_date = string
and iso_date = string
and full_iso_date = string

and time = string
and iso_time = string
;;

type year = int
and month = int
and month_day = int
and week_day = int
;;

type hour = int
and minute = int
and second = int
(** A nice aliasing for the numerical components of a time. *)
;;

let system_time_of_day () = gmtime (time ());;
let local_system_time_of_day () = localtime (time ());;

let year_of_system_time tm = 1900 + tm.Unix.tm_year
and month_of_system_time tm = tm.Unix.tm_mon
and month_day_of_system_time tm = tm.Unix.tm_mday
and week_day_of_system_time tm = tm.Unix.tm_wday
and hour_of_system_time tm = tm.Unix.tm_hour
and minute_of_system_time tm = tm.Unix.tm_min
and second_of_system_time tm = tm.Unix.tm_sec
;;

let time_of_system_time tm =
  Printf.sprintf "%02d:%02d:%02d"
    (hour_of_system_time tm)
    (minute_of_system_time tm)
    (second_of_system_time tm)
;;

let date_of_system_time lang tm =
  let month_day = month_day_of_system_time tm in
  let month = month_of_system_time tm in
  let week_day = Some (week_day_of_system_time tm) in
  let year = year_of_system_time tm in
  format_date lang month_day month week_day year
;;

let long_date_of_system_time lang tm =
  Printf.sprintf "%s %s"
   (date_of_system_time lang tm)
   (time_of_system_time tm)
;;

let short_date_of_system_time lang tm =
  let month_day = month_day_of_system_time tm in
  let month = month_of_system_time tm in
  let week_day = None in
  let year = year_of_system_time tm in
  let short_date = format_date lang month_day month week_day year in
  Printf.sprintf "%s" short_date
;;

let digital_date_of_system_time lang tm =
  let month_day = month_day_of_system_time tm in
  let month = month_of_system_time tm in
  let year = year_of_system_time tm in
  format_digital_date lang month_day month year
;;

let iso_date_of_system_time tm =
  let month_day = month_day_of_system_time tm in
  let month = month_of_system_time tm in
  let year = year_of_system_time tm in
  format_iso_date month_day month year
;;

let iso_time_of_system_time tm =
  let hour = hour_of_system_time tm
  and minute = minute_of_system_time tm
  and second = second_of_system_time tm in
  format_iso_time hour minute second
;;

let full_iso_date_of_system_time tm =
  let month_day = month_day_of_system_time tm in
  let month = month_of_system_time tm in
  let year = year_of_system_time tm in
  let hour = hour_of_system_time tm
  and minute = minute_of_system_time tm
  and second = second_of_system_time tm in
  format_full_iso_date month_day month year hour minute second
;;

let date_of_day lang =
  date_of_system_time lang (system_time_of_day ())
;;
let local_date_of_day lang =
  date_of_system_time lang (local_system_time_of_day ())
;;

let date_of_day_Uk () = date_of_day Uk;;
let date_of_day_Fr () = date_of_day Fr;;

let date = date_of_day_Uk;;

let local_date_of_day_Uk () = local_date_of_day Uk;;
let local_date_of_day_Fr () = local_date_of_day Fr;;

let local_date = local_date_of_day_Uk;;

let last_modification_system_time_of_file s =
  if s = "-" then system_time_of_day () else
  let st = Unix.stat s in
  let last_mod_system_time = st.Unix.st_mtime in
  let tm = Unix.gmtime last_mod_system_time in
  tm
;;

let last_modification_date_of_file lang s =
  date_of_system_time lang (last_modification_system_time_of_file s)
;;

let format_date_Uk = format_date Uk;;
let format_date_Fr = format_date Fr;;
