(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lib_strings.ml,v 1.3 2003/03/03 10:43:21 weis Exp $ *)

(* Returns the sub-string of [s] starting at index [i_start]
   (inclusively) to index [i_stop] (excluded). *)
let extract_string s i_start i_stop =
 let len = i_stop - i_start in
 String.sub s i_start len;;

(* Test if, starting from index [index], the string [s] matches the
   pattern [pat] *)
let starts_by s index pat =
 let ls = String.length s
 and lpat = String.length pat in
 let lim = index + lpat in
 if index < 0 || ls > 0 && index >= ls then invalid_arg "starts_by" else
 let rec starts i j =
  i >= lim || s.[i] = pat.[j] && starts (i + 1) (j + 1) in
 lim <= ls && starts index 0;;

(* Find the index of character [c] in string [s] starting from index [index],
   which should be a valid index into [s], and looking forward to the
   end of string [s]. *)
let pos_string s index c =
 let lim = String.length s - 1 in
 if index < 0 || index > lim then invalid_arg "pos_string" else
 let rec search i =
  if i > lim then raise Not_found else
  if s.[i] = c then i else search (i + 1) in
 search index;;

(* Find the index of character [c] in string [s] starting from index [index],
   which should be a valid index into [s], and looking backward to the
   beginning of string [s]. *)
let pos_string_rev s index c =
 let lim = String.length s - 1 in
 if index < 0 || index > lim then invalid_arg "pos_string_rev" else
 let rec search i =
  if i < 0 then raise Not_found else
  if s.[i] = c then i else search (i - 1) in
 search index;;

(* Find the sub string of [s] that starts by character [starter] and
   ends by [stopper], starting from valid index [index]. *)
let sub_string_from_to s index starter stopper =
  let i_start =
    try pos_string s index starter with
    | Not_found -> failwith (Printf.sprintf "Cannot find %c in %s" starter s) in
  let i_stop =
    try pos_string s (i_start + 1) stopper with
    | Not_found -> failwith (Printf.sprintf "Cannot find %c in %s" stopper s) in
  extract_string s (i_start + 1) i_stop;;

(* Returns the sub string starting from the position following the
   last occurrence of character [c] to the end of [s].
   If [c] is not found, [s] is returned *)
let from_char_to s c idx =
 let len = String.length s in
 if idx >= len then invalid_arg "from_char_to" else
 let rec search i =
  if i < 0 then s else
  if s.[i] = c then extract_string s (i + 1) len
  else search (i - 1) in
 search (idx - 1);;

let from_char_to_end s c = from_char_to s c (String.length s - 1);;

let from_to_char s idx c =
 let len = String.length s in
 if idx >= len || idx < 0 then invalid_arg "from_char_to" else
 let rec search i =
  if i >= len then String.sub s idx (len - idx) else
  if s.[i] = c then extract_string s idx i
  else search (i + 1) in
 search idx;;
