(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lib_date.mli,v 1.3 2003/11/10 08:33:03 weis Exp $ *)

val year_is_leap : int -> bool;;
 (* Checks if the int argument represents a leap year. *)
val last_day_of_month : int -> int -> int;;
 (* [last_day_of_month month year] returns the number of the last day
    of the month for the given year. *)
val date_is_valid : int -> int -> int -> bool;;
 (* [date_is_valid month_day month year] checks if the given arguments
    represent a valid date. *)
val compare_date : (int * int * int) -> (int * int * int) -> int;;
 (* Compares two dates represented as triple of integers [(month_day,
    month, year)], with the same convention as the polymorphic
    comparison [compare]. [compare_date (d1, m1, y1) (d2, m2, y2)]
    returns respectively [0], [1], or [-1], if date [(d1, m1, y1)] is
    respectively equal to, greater than, or lower than
    the date [(d2, m2, y2)]. *)
val week_day_of_date : int -> int -> int -> int;;
 (* [week_day_of_date month_day month year] return the number of the
    day in the week that corresponds to the date
    [month_day, month, year].
    (Week days are numbered from [0] for sunday to [6] for saturday.) *)

(* Translation from integer representation to day and month names. *)

val string_of_month : Lang.t -> int -> string;;
val string_of_week_day : Lang.t -> int -> string;;
 (* Mapping from days and months, represented as theirs numbers, to
    their usual character string names.
    The numbering starts from 0: thus Sunday and January have number 0.
    These functions raise [Invalid_argument] if their arguments are out of
    the proper range ([0..6] for day numbers and [0..11] for month
    numbers), or if the language is not supported. *)
val string_of_date : Lang.t ->
    int -> int -> int ->
    string * string * string * string;;
 (* [string_of_date lang d m y] calculates the week day associated
    with the date [d, m, y], and returns the string representations of
    the tuple [(week_day, d, m, y)].
    For instance [string_of_date Uk (21, 3, 2000)] is
    [("Tuesday", "21", "March", "2000")]. *)

val string_of_full_date : Lang.t ->
    int -> int -> int -> int ->
    string * string * string * string;;
 (* [string_of_full_date lang w d m y] returns a string representation
    of the tuple [(w, d, m, y)], supposed to be a date [d, m, y] with
    week day [w]. *)

val string_of_mois : int -> string;;
val string_of_jour : int -> string;;
 (* Same as above for french days and months. *)

val week_day_of_string : Lang.t -> string -> int;;
val month_of_string : Lang.t -> string -> int;;
 (* Map a day  (resp. a month) represented as a string for the language
    [lang] to the corresponding number. Recognition of valid strings is
    case unsensitive and usual abbreviations are accepted.
    Raise [Invalid_argument "unknown language"], if the language is
    not supported. *)

val format_date : Lang.t -> int -> int -> int -> int -> string;;
 (* [format_date lang month_day month week_day year] returns a string
    representation of a date according to the language [lang]. The
    date is represented by 4 integers: [week_day] and
    [month] are the numbers encoding the names of the day and
    the name of the month. Their should be in a range suitable for calling
    [string_of_month] and [string_of_week_day].
    The [month_day] and [year] arguments are the number of the day in
    the month and the number of the year. If the year number is lesser
    than 1000, then 1900 is added to it.
    There is no verification that the date is valid.
    For instance,
    [format_date Uk 31 12 5 00] is ["Friday, December 31, 1900"],
    [format_date Uk 31 12 5 101] is ["Friday, December 31, 2001"],
    [format_date Uk 31 12 5 2000] is ["Friday, December 31, 2000"]. *)
val string_of_Uk_date : int -> int -> int -> int -> string;;
val string_of_Fr_date : int -> int -> int -> int -> string;;
 (* Same as above for french or english translations. *)

val parse_date : Lang.t -> string -> int * int * int * int;;
 (* [parse_date lang s] parse the string [s] according to the language
 [lang], and return a date, the validity of which is not verified.
 Separators between words in [s] can be any of [' '], ['\t'], ['/'], [','],
 ['-'], or ['.'] and can be repeated.
 - If only one word is found in [s], this word is supposed to be made of
 digits that encode the date. Eight digits means [ddmmyyyy] (as
 [01032000] to mean first of March 2000), six digits means [ddmmyy]
 (as [010300] to mean first of March 1900), less than six digits means
 [yy] or [yyyy] (only the year number is provided, day and month
 default to first of January).
 - If two words are found, they are supposed to be the month and the year
 in that order, and the resulting date defaults to the first day of the
 given month and year.
 - If three words are found, they are supposed to be the month day, the
 month and the year. Order is language dependant, and follows the
 conventions of [format_date].
 - If four words are found, they are supposed to be the week day, the
 month day, the month and the year. Order is language dependant, and
 follows the conventions of [format_date].

 Note: if necessary the week day of the date may be found using the
 function [week_day_of_date]. *)

val parse : string -> int * int * int * int;;
 (* [parse s] is equivalent to [parse_date Uk s]. *)
val format : int -> int -> int -> int -> string;;
 (* [format month_day month week_day year] is equivalent to
    [format_date Uk month_day month week_day year]. *)

val valid_date : Lang.t -> string -> int * int * int * int;;
 (* [valid_date lang s] parse the string [s] according to the
 language [lang] and return a valid date or raises [invalid_arg]. *)

val normalize_date : Lang.t -> string -> string * string * string * string;;
 (* [normalize_date lang d] parses date [d] and returns the four
 fields of a completely specified date ([month_day], [month],
 [week_day], [year]). *)
