(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: substitute.ml,v 1.2 2003/03/03 11:25:33 weis Exp $ *)

(* Variable substitution in strings:
   Substitute $ident (or $(ident)) in a string are substituted
   according to an environment that maps the variables to strings. *)

let closing = function
  | '(' -> ')'
  | '[' -> ']'
  | '{' -> '}'
  | '<' -> '>'
  | _ -> failwith "opening";;

(* opening and closing: open and close characters, typically ( and )
   k balance of opening and closing chars
   s the string where wre searching
   p the index where we start the search *)
let advance_to_closing opening closing k s start =
 let rec advance k i lim =
   if i >= lim then raise Not_found else
   if s.[i] = opening then advance (k + 1) (i + 1) lim else
   if s.[i] = closing then
    if k = 0 then i else advance (k - 1) (i + 1) lim
   else advance k (i + 1) lim in
 advance k start (String.length s);;

let ident_start_char = function
  | '_'
  | 'a' .. 'z' | 'A' .. 'Z' |
    ''|''|''|''|''|''|''|''|''|''|''|''|''|
    ''|''|''|''|''|''|''|''|''|''|''|''|'' -> true
  | _ -> false;;

let ident_in_char = function
 | '0' .. '9' | '\'' -> true
 | c -> ident_start_char c;;

let advance_to_non_alpha s start =
 let rec advance i lim =
  if i >= lim then lim else
  match s.[i] with
  | 'a' .. 'z' | 'A' .. 'Z' | '0' .. '9' | '\'' | '_' |
    ''|''|''|''|''|''|''|''|''|''|''|''|''|
    ''|''|''|''|''|''|''|''|''|''|''|''|'' -> advance (i + 1) lim
  | _ -> i in
 advance start (String.length s);;

(* We are just at the beginning of an ident in s, starting at p *)
let find_ident s start =
 match s.[start] with
    (* Parenthesized ident ? *)
  | '(' | '[' | '{' | '<' as c ->
     let new_start = start + 1 in
     let stop = advance_to_closing c (closing c) 0 s new_start in
     String.sub s new_start (stop - start - 1), stop + 1
    (* Regular ident *)
  | c when ident_start_char c ->
     let stop = advance_to_non_alpha s (start + 1) in
     String.sub s start (stop - start), stop
  | c -> raise Not_found;;

(* Find the value associated to a variable in the user's (shell) environment. *)
let get_var ident =
  try Sys.getenv ident with
  | Not_found -> 
      failwith
        (Printf.sprintf "No (system) environment variable ``%s''" ident);;

(* Add a string [s] to a buffer, while substituting variables in it.
   Variables of the form $ident (or $(ident)) are replaced, according
   to the function f_subst.
   Variables of the form [<$ident>] are replaced by their value in the
   user's environment. *)
let rec substitute_buffer f_subst b s =
 let lim = String.length s in
 let rec subst previous i =
  if i < lim then
   begin
    match s.[i] with
    | '$' as current when previous = '\\' ->
       Buffer.add_char b current;
       subst current (i + 1)
    | '$' as current when previous = '<' ->
       let new_start = i + 1 in
       let stop =
         advance_to_closing previous (closing previous) 0 s new_start in
       let ident, next_i =
         String.sub s new_start (stop - i - 1), stop + 1 in
       substitute_buffer f_subst b (get_var ident);
       subst ' ' next_i
    | '$' as current ->
       begin try
        let ident, next_i = find_ident s (i + 1) in
        let v =
         try f_subst ident with
         | Not_found ->
            failwith
             (Printf.sprintf
               "No (Htmlc) environment variable ``%s''" ident) in
        substitute_buffer f_subst b v;
        subst ' ' next_i
       with
       | Not_found -> Buffer.add_char b current; subst current (i + 1)
       end
    | '<' as current ->
       subst current (i + 1)
    | current when previous == '<' ->
       go_on previous current i
    | current when previous == '\\' ->
       go_on previous current i
    | '\\' as current ->
       subst current (i + 1)
    | current ->
       Buffer.add_char b current;
       subst current (i + 1)
   end
 and go_on previous current i =
    Buffer.add_char b previous;
    Buffer.add_char b current;
    subst current (i + 1) in
 subst ' ' 0;;

let add_string env = substitute_buffer (Hashtbl.find env);;
