(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lib_strings.mli,v 1.2 2003/03/01 16:01:25 weis Exp $ *)

(* lib_strings : a collection of functions for string manipulation. *)

val extract_string : string -> int -> int -> string;;
  (* [extract_string s start stop] returns the sub string of [s]
  starting at index [i_start] (inclusively) to index [i_stop] (excluded). *)

val starts_by : string -> int -> string -> bool;;
  (* [starts_by s index pat] tests if, starting from index [index],
     the string [s] matches the pattern [pat].
     [index] should be a valid index into [s], otherwise exception
     [Invalid_argument] is raised.
     The empty string never matches a non empty pattern.
     Every string matches the empty pattern. *)

val pos_string : string -> int -> char -> int;;
  (* [pos_string s index char] finds the position of character [char]
     in string [s] starting from index [index], which should be a
     valid index into [s], and looking forward to the end of string
     [s].
     Raises exception [Not_found] if there is no such position.
     [index] should be a valid index into [s], otherwise exception
     [Invalid_argument] is raised. *)
val pos_string_rev : string -> int -> char -> int;;
  (* [pos_string_rev s index char] finds the position of the occurrence of
  the character [char] that precedes the index [index], which should be a
  valid index into [s].
  Raises exception [Not_found] if there is no such position.
  [index] should be a valid index into [s], otherwise exception
  [Invalid_argument] is raised. *)

val sub_string_from_to : string -> int -> char -> char -> string;;
  (* [sub_string_from_to s index starter stopper] finds the sub string of
     [s] that is (strictly) comprised between the delimiters [starter]
     and [stopper]: that string starts after the first occurrence of
     character [starter] and ends before the first following
     occurrence of character [stopper]. Search for [starter] begins at
     index [index].
     Raises exception [Not_found] if there is no such position.
     [index] should be a valid index into [s], otherwise exception
     [Invalid_argument] is raised. *)

val from_char_to_end : string -> char -> string;;
  (* [from_char_to_end s index char] returns the sub string of string [s] that
     starts from the position following the last occurrence of
     character [char], and ends to the end of the string [s].
     If [char] is not found, the string [s] is returned. *)

val from_char_to : string -> char -> int -> string;;
  (* [from_char_to s char index] returns the sub string of string [s] that
     starts from the position following the last occurrence of
     character [char] preceding [index], and ends to [index - 1].
     If [char] is not found, the sub-string of [s] starting from [0]
     to [index - 1] is returned. *)

val from_to_char : string -> int -> char -> string;;
  (* [from_to_char s index char] returns the sub string of string [s] that
     starts from the valid position [index] to the position preceding the
     first occurrence of character [char] following [index].
     If [char] is not found, the sub string from [index] to the end of
     [s] is returned.
      Raises [Invalid_arg] if [index] is not a valid index into [s]. *)

