(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lib_date.ml,v 1.2 2003/03/01 16:01:25 weis Exp $ *)

open Printf;;
open Lang;;
open Str;;

let month_names lang =
 match lang with
 | Fr -> [| "janvier"; "fvrier"; "mars"; "avril"; "mai"; "juin"; "juillet";
            "aot"; "septembre"; "octobre"; "novembre"; "dcembre"|]
 | Uk -> [| "January"; "February"; "March"; "April"; "May"; "June"; "July";
            "August"; "September"; "October"; "November"; "December" |]
 | _ -> invalid_arg "unknown language";;

let recognizer_months lang =
 match lang with
 | Fr ->
    [| "Jan\\($\\|v$\\|vier$\\)"; "F[e||e']v\\($\\|rier$\\)";
       "Mar\\($\\|s$\\)"; "Avr\\($\\|il$\\)";
       "Mai"; "Jui\\($\\|n\\)";
       "Juil\\($\\|let$\\)"; "Ao[u||u^]\\($\\|t$\\)";
       "Sep\\($\\|t$\\|tembre$\\)"; "Oct\\($\\|obre$\\)";
       "Nov\\($\\|embre$\\)"; "D[e||e']c\\($\\|embre$\\)" |]
 | Uk ->
    [| "Jan\\($\\|uary$\\)"; "Feb\\($\\|ruary$\\)";
       "Mar\\($\\|ch$\\)"; "Apr\\($\\|il$\\)";
       "May"; "Jun\\($\\|e$\\)";
       "Jul\\($\\|y$\\)"; "Aug\\($\\|ust$\\)";
       "Sep\\($\\|tember$\\)"; "Oct\\($\\|ober$\\)";
       "Nov\\($\\|ember$\\)"; "Dec\\($\\|ember$\\)" |]
 | _ -> invalid_arg "unknown language";;

let week_day_names lang =
 match lang with
 | Fr -> [| "dimanche"; "lundi"; "mardi"; "mercredi";
            "jeudi"; "vendredi"; "samedi" |]
 | Uk -> [| "Sunday"; "Monday"; "Tuesday"; "Wednesday";
            "Thursday"; "Friday"; "Saturday" |]
 | _ -> invalid_arg "unknown language";;

let recognizer_week_days lang =
 match lang with
 | Fr -> [| "di\\($\\|m\\($\\|anche$\\)\\)";
            "lu\\($\\|n\\($\\|di$\\)\\)";
            "ma\\($\\|r\\($\\|di$\\)\\)";
            "me\\($\\|r\\($\\|credi$\\)\\)";
            "je\\($\\|u\\($\\|di$\\)\\)";
            "ve\\($\\|n\\($\\|dredi$\\)\\)";
            "sa\\($\\|m\\($\\|edi$\\)\\)" |]
 | Uk -> [| "Sun\\($\\|day$\\)"; "Mon\\($\\|day$\\)";
            "Tue\\($\\|s\\($\\|day$\\)\\)";
            "Wed\\($\\|n\\($\\|esday$\\)\\)";
            "Thu\\($\\|r\\($\\|sday$\\)\\)";
            "Fri\\($\\|day$\\)";
            "Sat\\($\\|u\\($\\|rday$\\)\\)" |]
 | _ -> invalid_arg "unknown language";;

let reg_exps_week_days =
 let fr_regs = Array.map regexp_case_fold (recognizer_week_days Fr)
 and uk_regs = Array.map regexp_case_fold (recognizer_week_days Uk) in
 function
 | Fr -> fr_regs
 | Uk -> uk_regs
 | _ -> invalid_arg "unknown language";;

let reg_exps_months =
 let fr_regs = Array.map regexp_case_fold (recognizer_months Fr)
 and uk_regs = Array.map regexp_case_fold (recognizer_months Uk) in
 function
 | Fr -> fr_regs
 | Uk -> uk_regs
 | _ -> invalid_arg "unknown language";;

(* Computing days of week and time intervals *)
let year_is_leap a =
  (a mod 4 = 0 && not (a mod 100 = 0)) || a mod 400 = 0;;

let last_day_of_month m a =
  match m with
  | 1 | 3 | 5 | 7 | 8 | 10 | 12 -> 31
  | 2 -> if year_is_leap a then 29 else 28
  | 4 | 6 | 9 | 11 -> 30
  | _ -> invalid_arg "last_day_of_month";;

let date_is_valid j m a =
 j >= 1 && m >= 1 && m <= 12 && j <= last_day_of_month m a;;

let compare_date (j1, m1, a1) (j2, m2, a2) =
 match compare a1 a2 with
 | 0 ->
    begin match compare m1 m2 with
    | 0 -> compare j1 j2
    | c -> c end
 | c -> c;;

let rec days_of_month m a =
 if m <= 0 then invalid_arg "days_of_month" else
 match m with
 | 1 -> 0
 | n -> days_of_month (n - 1) a + last_day_of_month (m - 1) a;;

let days_of_year a = a * 365 + (a / 4) - (a / 100) + a / 400;;

let days_of_date (j1, m1, a1 as d1) =
 j1 + days_of_month m1 a1 + days_of_year (a1 - 1);;

let sub_date (j1, m1, a1 as d1) (j2, m2, a2 as d2) =
 let j_d1 = days_of_date d1 in
 let j_d2 = days_of_date d2 in
 j_d1 - j_d2;;

(* 1 Janvier 1900 is a monday *)
let week_day_of_date month_day month year =
 let days = sub_date (month_day, month, year) (1, 1, 1900) in
 (1 + (days mod 7)) mod 7;;

(* Formating dates to strings *)
let string_of_week_day lang week_day =
 if week_day < 0 || week_day > 6 then invalid_arg "string_of_week_day" else
 (week_day_names lang).(week_day);;

let string_of_month lang month =
 if month < 0 || month > 11 then invalid_arg "string_of_month" else
 (month_names lang).(month);;

let string_of_full_date lang week_day month_day month year =
  (string_of_week_day lang week_day,
   string_of_int month_day,
   string_of_month lang month,
   string_of_int year);;

let string_of_date lang month_day month year =
  let week_day = week_day_of_date month_day month year in
  string_of_full_date lang week_day month_day month year;;

let string_of_mois = string_of_month Fr;;
let string_of_jour = string_of_week_day Fr;;

let find p v =
 let limit = Array.length v in
 let rec ind i =
  if i >= limit then raise Not_found else
  if p v.(i) then i else ind (succ i) in
 ind 0;;

let index item v = find (function x -> x = item) v;;

let week_day_of_string lang day_name =
 find (function regexp -> string_match regexp day_name 0)
      (reg_exps_week_days lang);;

let month_of_string lang month_name =
 succ (find (function regexp -> string_match regexp month_name 0)
            (reg_exps_months lang));;

let year_of_int i = if i <= 1000 then i + 1900 else i;;

let format_date lang month_day month week_day year =
 match lang with
 | Fr -> sprintf "%s %d %s %d"
          (string_of_week_day Fr week_day) month_day
          (string_of_month Fr month) (year_of_int year)
 | Uk -> sprintf "%s, %s %d, %d"
          (string_of_week_day Uk week_day)
          (string_of_month Uk month) month_day (year_of_int year)
 | _ -> invalid_arg "format_date";;

let string_of_Uk_date = format_date Uk;;
let string_of_Fr_date = format_date Fr;;

(* Parsing dates *)

let rec get_int s limit i =
 let rec get j =
  if j >= limit then invalid_arg s else
  match s.[j] with
  | '0' .. '9' -> get (succ j)
  | c ->
     if j = i then 0, succ j else
       begin try int_of_string (String.sub s i (j - i)), j
       with Failure _ -> invalid_arg s end in

 if i >= limit then invalid_arg s else
 match s.[i] with
 | ' ' -> get_int s limit (i + 1)
 | '-' -> let i, j = get (i + 1) in -i, j
 | c -> get i;;

let is_separator = function
  | ' ' | '\t' | '/' | ',' | '-' | '.' -> true
  | _ -> false;;

let get_separator s limit i =
 let rec get j =
  if j >= limit then invalid_arg s else
  if is_separator s.[j] then get (succ j)
  else j in
 get i;;

let get_word s limit i =
 if i >= limit then invalid_arg s else
 let rec get j =
  if j >= limit then j else
  if is_separator s.[j] then j else get (succ j) in
 get i;;

let get_date s =
 let limit = String.length s in
 let rec get accu i =
  let j = get_word s limit i in
  let w = String.sub s i (j - i) in
  let new_accu = w :: accu in
  if j >= limit then new_accu else
   let new_i = get_separator s limit j in
   get new_accu new_i in
 List.rev (get [] 0);;

let month_of_string_or_int lang s =
 try int_of_string s
 with Failure _ -> month_of_string lang s;;

let year_of_string s =
 if String.length s <= 2 then 2000 + int_of_string s
 else int_of_string s;;

let rec date_of_tokens lang toks =
 match toks with
 | [year] ->
     date_of_int_string lang year
 | [month; year] ->
     let month_day = 1 in
     let month = month_of_string_or_int lang month in
     let year = year_of_string year in
     let week_day = week_day_of_date month_day month year in
     (week_day, month_day, month, year)
 | words ->
    match lang with
    | Fr ->
       begin match words with
       | [month_day; month; year] ->
          let month_day = int_of_string month_day in
          let month = month_of_string_or_int lang month in
          let year = year_of_string year in
          let week_day = week_day_of_date month_day month year in
          (week_day, month_day, month, year)
       | [week_day; month_day; month; year] ->
          let week_day = week_day_of_string lang week_day in
          let month_day = int_of_string month_day in
          let month = month_of_string_or_int lang month in
          let year = year_of_string year in
          (week_day, month_day, month, year)
       | _ -> invalid_arg "parse_date" end
    | Uk ->
       begin match words with
       | [month; month_day; year] ->
          let month = month_of_string_or_int lang month in
          let month_day = int_of_string month_day in
          let year = year_of_string year in
          let week_day = week_day_of_date month_day month year in
          (week_day, month_day, month, year)
       | [week_day; month; month_day; year] ->
          let week_day = week_day_of_string lang week_day in
          let month = month_of_string_or_int lang month in
          let month_day = int_of_string month_day in
          let year = year_of_string year in
          (week_day, month_day, month, year)
       | _ -> invalid_arg "parse_date" end
    | _ -> invalid_arg "parse_date"

(* Date can also be written using digits only: yy, yyyy, ddmmyy, or ddmmyyyy *)
and date_of_int_string lang s =
 match String.length s with
 | 8 ->
    date_of_tokens lang [String.sub s 0 2; String.sub s 2 2; String.sub s 4 4]
 | 6 ->
    date_of_tokens lang [String.sub s 0 2; String.sub s 2 2; String.sub s 4 2]
 | _ ->
   let year = s in
   let month_day = 1 in
   let month = 0 in
   let year = year_of_string year in
   let week_day = week_day_of_date month_day month year in
   (week_day, month_day, month, year);;

let parse_date lang s =
 let toks = get_date s in
 date_of_tokens lang toks;;

let valid_date lang s =
 let (week_day, month_day, month, year as date) = parse_date lang s in
 if week_day = week_day_of_date month_day month year then date
 else invalid_arg "valid_date";;

let parse s = parse_date Uk s;;

let format month_day month week_day year =
  format_date Uk month_day month week_day year;;

let normalize_date lang s =
 let (week_day, month_day, month, year) = parse_date lang s in
 string_of_full_date lang week_day month_day month year;;

