(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: lang.ml,v 1.5 2003/03/01 16:24:26 weis Exp $ *)

open Lib_strings;;

type file_name = string;;

type t =
   |  Fr | Uk | Es | De | It;;

(* The language chosen when no one more appropriate can be selected. *)
let default_language = ref Uk;;

let prefix_of_file_name s = from_to_char s 0 '.';;

let suffix_of_file_name s = from_char_to_end s '.';;

(* Try to detect the language in which a file is written. We suppose that
   the language is indicated by a proper ``internal extension'', that
   is a character string delimited by [-] and [.] characters (as [fra]
   in [foo-fra.html]). If [-] is not found, then the sub-string from
   the beginning of [s] to the position of [.] is returned (for instance [fra]
   in [fra.htm]). If neither [-] nor [.] is found then the entire name
   is returned. *)
(*
Could simply be
let lang_suffix s =
 let len = String.length s in
 let minus =
  try pos_string_rev s (len - 1) '-' with Not_found -> -1 in
 let dot =
  try pos_string_rev s (len - 1) '.' with Not_found -> len in
 String.sub s (minus + 1) (dot - minus - 1);;
*)

let lang_suffix_of_file_name s =
 let len = String.length s in
 let minus =
  try pos_string_rev s (len - 1) '-' with Not_found -> -1 in
 let dot =
  try pos_string_rev s (len - 1) '.' with Not_found -> len in
 match minus, dot with
 | -1, -1 -> s
 | -1, 0 -> ""
 | -1, lastdot ->
    begin try
     let antedot = pos_string_rev s (lastdot - 1) '.' in
     String.sub s (antedot + 1) (lastdot - antedot - 1)
    with Not_found -> String.sub s (minus + 1) (dot - minus - 1) end
 | minus, -1 -> String.sub s (minus + 1) (dot - minus - 1)
 |  _, _ -> String.sub s (minus + 1) (dot - minus - 1);;

let lang_of_string = function
  | "fr" -> Fr
  | "en" | "uk" | "us" -> Uk
  | "es" -> Es
  | "de" -> De
  | "it" -> It
  | s -> failwith ("lang_of_string: unknown language " ^ s);;

let lang_of_suffix = function
  | "fra" | "fr" -> Fr
  | "eng" | "en" | "uk" -> Uk
  | "esp" | "es" -> Es
  | "deu" | "de" -> De
  | "ita" | "it" -> It
  | s -> failwith ("lang_of_suffix unknown suffix " ^ s);;

let string_of_lang = function
  | Fr -> "fr"
  | Uk -> "en"
  | Es -> "es"
  | De -> "de"
  | It -> "it";;

let suffix_of_lang = function
  | Fr -> "fra"
  | Uk -> "eng"
  | Es -> "esp"
  | De -> "deu"
  | It -> "ita";;

let lang_of_file_name s = lang_of_suffix (lang_suffix_of_file_name s);;

let language = ref None;;

let get_set_language () =
 match !language with
 | Some lang -> lang
 | None -> raise Not_found;;

let get () =
 match !language with
 | Some lang -> lang
 | None -> !default_language;;

let get_with_file_name s =
  try get_set_language () with
  | Not_found ->
      begin try lang_of_file_name s with
      | Failure _ -> !default_language end;;

(* Logic: htmlc -language lang => no guess is ever tried: language is lang.
          htmlc -default-language lang => when guessing which language
          is used by a file, the default will be this one. *)

let set lang = language := Some lang;;

let set_default lang = default_language := lang;;

let set_from_string s = set (lang_of_string s);;

let set_default_from_string s =
  set_default (lang_of_string s);;
