(***********************************************************************)
(*                                                                     *)
(*                           CIME Caml                                 *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* $Id: htmlc.ml,v 1.14 2003/08/09 13:30:37 weis Exp $ *)

open Arg;;
open Lib_strings;;
open Lang;;
open Date;;

(* Variables environment to expand into files. *)
let htmlc_env = Hashtbl.create 17;;
let getenv = Hashtbl.find htmlc_env;;
let env_files = ref [];;
let add_e_filename s =
  (* Need a quoted name to extract it as if it has been specified in a file. *)
  let s = Printf.sprintf "\"%s\"" s in
  env_files := s :: !env_files;;

let htmlc_add_string b s = Substitute.add_string htmlc_env b s;;

let htmlc_output_line b line =
  htmlc_add_string b line; Buffer.add_char b '\n';;

let htmlc_substitute_string =
 let htmlc_line = Buffer.create 100 in
 function s ->
  htmlc_add_string htmlc_line s;
  let s = Buffer.contents htmlc_line in
  Buffer.clear htmlc_line;
  s;;

let name_of_included_file s =
  htmlc_substitute_string (sub_string_from_to s 0 '"' '"');;

let rest_of_included_line s =
  let start =
    try
      1 + String.index_from s
       (1 + String.index_from s
         (1 + String.index_from s 0 '"')
        '"')
       '>'
    with
    | Not_found -> max_int in
  let l = String.length s - start in
  if l <= 0 then "" else
  String.sub s start l;;

(* File inclusion functions. *)
let cannot_include_file s x =
(*  prerr_endline
   (Printf.sprintf "Htmlc: error while reading %s." s); *)
  failwith
    (Printf.sprintf "Htmlc: aborting while loading file %s: %s" s x);;

let load_included_file include_ic add_rest line b =
  let included_file = name_of_included_file line in
  let rest_of_line = rest_of_included_line line in
  try
   let ic = Path.open_in included_file in
   include_ic ic b;
   add_rest rest_of_line
  with
  | x -> cannot_include_file included_file (Printexc.to_string x);;

let html_encode_escape b s =
 let l = String.length s in
 for i = 0 to l - 1 do
  match s.[i] with
  | '<' -> Buffer.add_string b "&lt;"
  | '>' -> Buffer.add_string b "&gt;"
  | '&' -> Buffer.add_string b "&amp;"
  | c -> Buffer.add_char b c
 done;;

let add_verbatim_ic ic b =
 try
  while true do
   let line = input_line ic in
   html_encode_escape b line;
   Buffer.add_char b '\n'
  done
 with End_of_file -> close_in ic;; 

let add_verbatim_file line b =
 load_included_file add_verbatim_ic (htmlc_add_string b) line b;;

(* Given a scanning buffer sb, an ident [name], and a(n unread) character
   find the specified value for the name, then add the corresponding
   binding to the environment.
   We parse ident *[= *]value\n
   and value is either a string, a char a boolean or a floating point
   number. *)
let rec bind_name sb name = function
  | '=' ->
     Scanf.bscanf sb "= %0c" (bind_name sb name)
  | '"' ->
     Scanf.bscanf sb "%S%_[ \n]"
       (fun v -> Hashtbl.add htmlc_env name v)
  | '0' .. '9' | '-' | '+' ->
     Scanf.bscanf sb "%f%_[ \n]"
       (fun v -> Hashtbl.add htmlc_env name (string_of_float v))
  | '\'' ->
     Scanf.bscanf sb "%C%_[ \n]"
       (fun v -> Hashtbl.add htmlc_env name (Char.escaped v))
  | 't' | 'f' ->
     Scanf.bscanf sb "%B%_[ \n]"
       (fun v -> Hashtbl.add htmlc_env name (string_of_bool v))
  | _ ->
     Scanf.bscanf sb "%[^ \n]%_[ \n]"
       (fun v -> Hashtbl.add htmlc_env name v)
;;

let load_env_sb ic sb =
  try
   while true do
     Scanf.bscanf sb " %0c" (function
     (* Skip comments *)
     | '#' -> Scanf.bscanf sb "%_s@\n" ()
     (* Otherwise find an ident, and call [bind_name] to bind it. *)
     | _ -> Scanf.bscanf sb " %[^ =\"\t\n\r] %0c" (bind_name sb))
   done
  with
  | End_of_file -> close_in ic
  | Scanf.Scan_failure s ->
      prerr_endline (Printf.sprintf "Htmlc: wrong environment entry, %s" s);
      failwith "load environment";;

let load_env_file_in_line line b =
 let load_env_ic ic b = load_env_sb ic (b ic) in
 let add_rest r = htmlc_add_string b r in
 load_included_file load_env_ic add_rest line (Scanf.Scanning.from_channel);;

let load_env_file fname =
 let load_env_ic ic b = load_env_sb ic (b ic) in
 load_included_file load_env_ic ignore fname (Scanf.Scanning.from_channel);;

let load_env_files () =
 try List.iter load_env_file !env_files with
 | Failure s -> prerr_endline s; exit 4;;

let rec treat_channel src ic b =
 let lang = Lang.get_with_file_name src in
 try
  while true do
   let line = input_line ic in
   if starts_by line 0 "<!--#include verbatim=\"" then
    add_verbatim_file line b else
   if starts_by line 0 "<!--#include environment=\"" then
    load_env_file_in_line line b else
   if starts_by line 0 "<!--#include virtual=\"" ||
       starts_by line 0 "<!--#include=\"" then
    load_included_file (treat_channel src) (htmlc_add_string b) line b else
   if starts_by line 0 "<!--#echo var=\"LAST_MODIFIED\"" then
    let len =
     if starts_by line 29 " -->" then 33 else
     if starts_by line 29 "-->" then 32 else 0 in
    begin
     htmlc_add_string b (last_modification_date_of_file lang src);
     htmlc_add_string b (String.sub line len (String.length line - len))
    end
   else htmlc_output_line b line;
  done
 with End_of_file -> close_in ic;;

let treat_file =
 let b = Buffer.create (20 * 1024) in
 fun src tgt ->
 (* printf "Source %s, Target %s\n" src
     (if tgt = "" then "std_out" else tgt); *)
   Buffer.reset b;
   try
    let ic = open_in src in
    try
     let oc = if tgt = "" then stdout else open_out tgt in
     try
       treat_channel src ic b;
       close_in ic;
       output_string oc (Buffer.contents b);
       if tgt <> "" then close_out oc
     with x -> if tgt <> "" then Sys.remove tgt; raise x
    with
    | x -> close_in ic; raise x
   with
   | Sys_error s | Failure s -> prerr_endline s; exit 2;;

let args = ref [];;

let source = ref "";;
let target = ref "";;

let get_usage () =
  "Usage: htmlc <options> <files>\n\
   htmlc -s <source.shtml>\n\
   or htmlc -c <source.html>\n\
   or htmlc <source>\n\
   or htmlc <source> <destination>\n\
   or htmlc [-i|-f|-from] <source> [-o|-t|-to] <destination>\n\
   \n\
   Options are:\n  \
   -s <source.shtml>: expand file <source.shtml> into file <source.html>\n  \
   -c <source.html>: expand file <source.html> into file <source.htm>\n  \
   -I <include-dir>: add <include-dir> to the list of include directories\n  \
   -v: print the htmlc version number";;

let print_usage s = prerr_endline s;;

let usage () = print_usage (get_usage ()); exit 1;;

let get_filenames s =
 if !source = "" then source := s else target := s;
 if !target <> "" then begin
  args := (!source, !target) :: !args; source := ""; target := "" end;;

let add_o_filename s =
 match !args with
 | [] -> args := ("", s) :: !args
 | (s1, "") :: rest -> args := (s1, s) :: rest
 | (s1, s2) :: rest -> args := ("", s) :: !args;;

let add_i_filename s =
 match !args with
 | [] -> args := (s, "") :: !args
 | ("", s2) :: rest -> args := (s, s2) :: rest
 | (s1, s2) :: rest -> args := (s, "") :: !args;;

let add_file suffix s =
 add_i_filename s;
 let name = Filename.chop_extension (Filename.basename s) in
 add_o_filename (name ^ suffix);;

let add_html_file = add_file ".htm";;
let add_shtml_file = add_file ".html";;

let add_filename add suffix s =
 if Filename.check_suffix s suffix then add s else
  begin
    prerr_endline
      (Printf.sprintf "File name %s has not the required %s suffix."
         s suffix);
    exit 3
  end;;

let add_html_filename = add_filename add_html_file ".html";;
let add_shtml_filename = add_filename add_shtml_file ".shtml";;

let set_lang s = Lang.set (Lang.lang_of_string s);;

let print_version () = prerr_endline ("Htmlc Version " ^ Version.current);;

let main () =
 parse [
  ("-I", String Path.push,
   "<dirname>: add directory <dirname> in front of the list\n      \
     of directories searched for include files");
  ("-lang", String Lang.set_from_string,
   "<language>: language is set to <language> \
    (no more guess from file-name)");
  ("-default-lang", String Lang.set_default_from_string,
   "<language>: default language is set to <language>");
  ("-s", String add_shtml_filename,
   "<filename.shtml>: input file is set to <filename.shtml>,\n     \
    output file is set to filename.html");
  ("-c", String add_html_filename,
   "<filename.html>: input file is set to <filename.html>,\n     \
    output file is set to filename.htm");
  ("-o", String add_o_filename,
   "<filename>: output file is set to <filename>");
  ("-t", String add_o_filename,
   "<filename>: output file is set to <filename>");
  ("-to", String add_o_filename,
   "<filename>: output file is set to <filename>");
  ("-i", String add_i_filename,
   "<filename>: input file is set to <filename>");
  ("-f", String add_i_filename,
   "<filename>: input file is set to <filename>");
  ("-from", String add_i_filename,
   "<filename>: input file is set to <filename>");
  ("-v", Unit print_version,
   ": print the htmlc version number");
  ("-env", String add_e_filename,
   "<filename>: \
    read <filename> to define variable bindings to substitute");
  ]
  get_filenames
  (get_usage ());
 (* Un seul argument fourni, on envoit le rsultat sur std_out *)
 if !args = [] then
  if !source = "" then
   if !Arg.current = 1 then usage () else ()
  else args := [!source, !target];
 (* Setting environment to expand variables. *)
 load_env_files ();
 List.iter (function (src, tgt) -> treat_file src tgt) (List.rev !args);
 exit 0;;

Printexc.print main ();;
