%{
/*
 *  idlscan.l
 *
 *  Copyright (C) 1997 Martin von Lwis
 */
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <iluptype.h>
#include "iluidl.h"
#include "idlparser.h"

#define MAX_INCLUDE_DEPTH 20
YY_BUFFER_STATE include_stack[MAX_INCLUDE_DEPTH];
static int include_stack_ptr=0;
void OpenInclude(char *statement,int brackets);
void PopInclude();

char *current_cpp;
void CppAppend(char*);
void CppProcess();
void CppDefine(string s);
string CppDefinitionOf(string s);
boolean IfDefed(string s,int len);
boolean CppReplace(string s);

static int ifnesting=0,iffailed=0;
static int cpp_continue_state;

typedef struct cpp_define_s{
	string name;
	string value;
}*cpp_define;

list CppDefines;

#define YY_USER_INIT init_scanner();
void init_scanner();

/* token value allocation */
refany alloc_int(long);
refany alloc_real(double);
refany alloc_char(char);

%}
/* CPP is exclusive: in CPP mode, normal rules are not considered */
%x CPP
%x CPPGET
%x COMMENT
%x FAILED
%option noyywrap
/* nested comments and such using yy_push_state */
%option stack
ID	[[:alpha:]_][[:alnum:]_]*
DEC	[+-]?[[:digit:]]+
HEX	0x[[:xdigit:]]+
REAL	[-+]?[[:digit]]+\.[[:digit:]]+
COM	\/\*
%%
{COM}		yy_push_state(COMMENT);
<COMMENT>{
	\*\/	yy_pop_state();
	{COM}	yy_push_state(COMMENT);
	.	/*skip*/
	<<EOF>>	fprintf(stderr,"unterminated comment\n");
}

^#[ \t]*	BEGIN(CPPGET);
<CPPGET>{
	[^\\\n]*	CppAppend(yytext);
	\\\n	/*ignore backslash-newline*/
	\\	CppAppend(yytext);
	\n	CppProcess();
}
<CPP>{
	{COM}	yy_push_state(COMMENT);
	^include[ \t]+\<[^>]+>[ \t]*	{
		OpenInclude(yytext,1);
	}
	^include[ \t]+\"[^\"]+\"[ \t]*	{
		OpenInclude(yytext,0);
	}
	^ifdef[ \t]+{ID}	{
					if(!IfDefed(yytext,6)){
						cpp_continue_state=FAILED;
						iffailed++;
					}
					ifnesting++;
				}
	^ifndef[ \t]+{ID}	{	if(IfDefed(yytext,7)){
						cpp_continue_state=FAILED;
						iffailed++;
					}
					ifnesting++;
				}
	^else[ \t]*	cpp_continue_state=FAILED;iffailed++;
	^endif[ \t]*	ifnesting--;iffailed--;
	^define[ \t]+{ID}.*	{
					CppDefine(yytext);
				}
	.+	fprintf(stderr,"Unknown preprocessor directive %s\n",yytext);
	<<EOF>>	PopInclude();BEGIN(cpp_continue_state);yy_set_bol(1);
}

<FAILED>{
	^#[ \t]*ifdef	ifnesting++;iffailed++;
	^#[ \t]*else	if(--iffailed)iffailed++;else{BEGIN(0);}
	^#[ \t]*endif	ifnesting--;iffailed--;if(!iffailed){BEGIN(0);}
	.		/*skip*/
	\n		/*skip*/
}

module		return MODULE;
interface	return INTERFACE;
const		return CONST;
typedef		return TYPEDEF;
readonly	return READONLY;
attribute	return ATTRIBUTE;
exception	return EXCEPTION;
oneway		return ONEWAY;
inout		return INOUT;
in		return IN;
out		return OUT;
raises		return RAISES;
context		return CONTEXT;

float		return FLOAT_T;
double		return DOUBLE_T;
long		return LONG_T;
short		return SHORT_T;
unsigned	return UNSIGNED_T;
char		return CHAR_T;
boolean		return BOOLEAN_T;
void		return VOID_T;
octet		return OCTET_T;
any		return ANY_T;
string		return STRING_T;
struct		return STRUCT;
union		return UNION;
switch		return SWITCH;
case		return CASE;
default		return DEFAULT;
enum		return ENUM;
sequence	return SEQUENCE;

TRUE		return BOOL_TRUE;
FALSE		return BOOL_FALSE;

{DEC}		{	yylval.value=(refany)strtol(yytext,0,10);
			yylval.tag=Taglong;
			return INTEGER_L;
		}
{HEX}		{	yylval.value=(refany)strtol(yytext,0,16);
			yylval.tag=Taglong;
			return INTEGER_L;
		}
{REAL}		{	yylval.value=alloc_real(strtod(yytext,0));
			yylval.tag=TagReal;
			return FLOAT_L;
		}
\'[[:print:]]\'			{	yylval.value=(refany)(int)yytext[1];
					yylval.tag=Tagchar;
					return CHAR_L;
				}
\'\\[0-7]{1,3}\'		{yylval.value=(refany)(int)strtol(yytext+1,NULL,8);
				 yylval.tag=Tagchar;
				 return CHAR_L;
				}
\'\\x[[:xdigit:]]{1,2}\'	{yylval.value=(refany)(int)strtol(yytext+2,NULL,16);
				 yylval.tag=Tagchar;
				return CHAR_L;
				}
\'\\.\'				{	char c;
					switch(yytext[2]){
					case 'n': c='\n';break;
					case 't': c='\t';break;
					case 'v': c='\v';break;
					case 'r': c='\r';break;	
					case 'f': c='\f';break;
					case '\\': c='\\';break;
					case '\'': c='\'';break;
					default: c=yytext[2];break;
					}
					yylval.value=(refany)(int)c;
					yylval.tag=Tagchar;
					return CHAR_L;
				}

{ID}		{ if(!CppReplace(yytext)){
			yylval.value=ilu_strdup(yytext);
			yylval.tag=Tagstring;
			return IDENT;
		  }
		}

::		return SCOPE;
>>		return RSHIFT;
\<\<		return LSHIFT;
[-|^&+*/%]	return yytext[0]; /* arithmetic operators */;
[(){},:;=<>]	return yytext[0];
\[		return yytext[0];
\]		return yytext[0];

[[:space:]]	;
.		return yytext[0];

\/\/.*$		; /* C++ comment */

<INITIAL><<EOF>>	{
		if(--include_stack_ptr<0)
    				yyterminate();
                 else{
  			yy_switch_to_buffer(include_stack[include_stack_ptr]);
			BEGIN(0);
		 }
		}
%%

void init_scanner()
{
	CppDefines=iluparser_new_list();
}

/* yy_scan_bytes for defines */

void OpenInclude(char *statement,int brackets)
{
	char *file;
	char *start,*stop;
	if(brackets){
		start=strchr(statement,'<');
		assert(start);
		stop=strchr(start+1,'>');
		assert(stop);
	}else{
		start=strchr(statement,'"');
		assert(start);
		stop=strchr(start+1,'"');
		assert(stop);
	}
	*stop='\0';
	file=ilu_strdup(start+1);
	assert(include_stack_ptr<MAX_INCLUDE_DEPTH);
	include_stack[include_stack_ptr++]=YY_CURRENT_BUFFER;
	yyin=fopen(file,"r");
	if(!yyin){perror("fopen");exit(0);}
	yy_switch_to_buffer(yy_create_buffer(yyin,YY_BUF_SIZE));
	/* new file is scanned like top-level */
	BEGIN(0);
}

void PopInclude()
{
	--include_stack_ptr;
	assert(include_stack_ptr>=0);
	yy_switch_to_buffer(include_stack[include_stack_ptr]);
}

void CppAppend(char* s)
{
	char *n;
	if(current_cpp){
		n=iluparser_Malloc(strlen(current_cpp)+strlen(s)+1);
		strcpy(n,current_cpp);
		strcat(n,s);
		free(current_cpp);
		current_cpp=n;
	}else{
		current_cpp=ilu_strdup(s);
	}
}

void CppProcess()
{
	/* FIXME: could avoid copy, should delete buffer, string vs file */
	include_stack[include_stack_ptr++]=YY_CURRENT_BUFFER;
	/* this already does the switch_to_buffer */
	yy_scan_string(current_cpp);
	free(current_cpp);
	current_cpp=0;
	/* where to go on after we're done */
	cpp_continue_state=INITIAL;
	BEGIN(CPP);
}

void CppDefine(string s)
{
	string name=ilu_strdup(s+7); /* skip "define " */
	string end;
	cpp_define def;
	while(isspace(*name))name++;
	name=ilu_strdup(name);
	for(end=name+1;*end && isalnum(*end);end++);
	*end='\0';
	end++;
	def=iluparser_Malloc(sizeof(struct cpp_define_s));
	def->name=name;
	def->value=end;
	list_insert(CppDefines,def);
}

static boolean define_find(cpp_define item,string s)
{
	return strcmp(item->name,s)==0;
}

string CppDefinitionOf(string s)
{
	cpp_define def=list_find(CppDefines,(iluparser_FindProc)define_find,s);
	return def ? def->value : NULL;
}

boolean IfDefed(string s,int len)
{
	string name=s+len; /* skip "ifdef " */
	string end;
	boolean result;
	while(!isalnum(*name))name++;
	/* Fixme: do ifdef lookup with name and length */
	name=ilu_strdup(name);
	for(end=name+1;*end && isalnum(*end);end++);
	*end='\0';
	result = CppDefinitionOf(name)!=NULL;
	free(name);
	return result;
}

boolean CppReplace(string s)
{
	string r=CppDefinitionOf(s);
	if(!r)return 0;
	if(include_stack_ptr==MAX_INCLUDE_DEPTH){
		fprintf(stderr,"Recursive macro expansion for %s\n",s);
		exit(1);
	}
	include_stack[include_stack_ptr++]=YY_CURRENT_BUFFER;
	yy_scan_string(r);
	return 1;
}

refany alloc_int(long i)
{
	assert(0);
}

refany alloc_real(double f)
{
	assert(0);
}

refany alloc_char(char c)
{
	assert(0);
}
