/*
 BeginILUCopyright
 
 Copyright (c) 1991-1998 Xerox Corporation.  All Rights Reserved.
 
 Unlimited use, reproduction, modification, and distribution of this
 software and modified versions thereof is permitted.  Permission is
 granted to make derivative works from this software or a modified
 version thereof.  Any copy of this software, a modified version
 thereof, or a derivative work must include both the above copyright
 notice of Xerox Corporation and this paragraph.  Any distribution of
 this software, a modified version thereof, or a derivative work must
 comply with all applicable United States export control laws.  This
 software is made available AS IS, and XEROX CORPORATION DISCLAIMS ALL
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE, AND NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY
 LIABILITY FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS
 EXPRESSLY DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING
 NEGLIGENCE) OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED
 OF THE POSSIBILITY OF SUCH DAMAGES.
 
 EndILUCopyright
*/
/* IluPassport.java */
/* Chris Jacobi, February 19, 1998 1:06 pm PST */

/*
 */
 
/* 
 * $Id: IluPassport.java,v 1.18 1998/10/09 22:50:46 larner Exp $ 
 */


/* 
 * Representation for ilu_Passport. <p>
 * Native code in IluJava_IluPort.c
 */


package xerox.ilu;

/**
 * Like a "wallet" filled with IluIdentity's.<br>
 *
 * Passports can be generated by applications and passed to ILU (called
 * initiating passports), or,
 * constructed by ILU (using data from the other side of the 
 * wire) and passed to the client (called accepting passports).<p>
 *
 * Client generated passports are mutable and stay alive until
 * garbage collected.<br>
 * Ilu generated passports can be used only while the call for which
 * they were constructed is ongoing.  If used afterwards, an exception  
 * will be raised.<p>
 *
 * A passport (currently) can carry at most one IluIdentity of 
 * each IluIdentityType.
 *
 * @see IluIdentity
 * @see IluIdentityType
 * @see IluPort
 * @see IluServiceThread
 */
public final class IluPassport extends IluWPBase {

    private long yIluPassport = 0;
    private IluCall call = null;
    /* friendly */ int valid = 0; 
        //  0 -> mutable passport
	// +1 -> valid immutable passport
	// -1 -> disabled immutable passport
    private java.util.Hashtable cachedIdentities = null;
        // NOT for efficiency reasons.  I believe that
        // identities will be rarely ever queried twice.
        // However this PREVENTS GARBAGE COLLECTION of identities
        // while referenced by passports. 
        
    
    /**
     * Creation of a mutable, client generated passport
     */
    public IluPassport() {
        super();
        nInitClientPassport();
    } //constructor


    /**
     * Ilu internal: Creation of an Ilu generated 
     * passport (which will be immutable for clients)
     */
     private IluPassport(IluCall call) {
        super();
        this.call = call;
        this.valid = 1;
    } //constructor
    
    
    private native void nInitClientPassport();
    private native void nFinalizePassport();
    private native int nAddIdentity(IluIdentity ii);
    private native int nFindIdentity(
        IluCall call, IluIdentityType it, IluIdentity iiContainer
        );
    
    
    /**
     * Not available to the general public.
     * (protected final to make sure)
     */
    protected final void finalize () throws java.lang.Throwable {
        this.cachedIdentities = null;
        if (valid==0) {nFinalizePassport();}	//frees native memory
        super.finalize();                	//IluWPBase requirement 
    } //finalize
    
    
    /** 
     * "private" to guarantee no calls from outside synchronized block. 
     * Returns cached IluIdentity or null if none is cached.
     * Side effect:ensures cachedIdentities is allocated.
     */ 
    private final IluIdentity getCachedIdentity(IluIdentityType it) {
        if (this.cachedIdentities==null) {
            this.cachedIdentities = new java.util.Hashtable();
        }
        java.lang.Object got = this.cachedIdentities.get(it);
        if (got != null) {
            return (IluIdentity) got;
        } else {
            return null;
        }
    } //getCachedIdentity
    
    
    /**
     * Query this passport whether it contains an identity of
     * the specified type and returns it.  Returns null
     * if passport does not contain such identity. 
     */
    public IluIdentity findIdentity(IluIdentityType it) {
        if (it == null) {
            throw new IluSomeSystemException("null IluIdentityType");
        }
        synchronized (this) {
           IluIdentity iiContainer = null;
           if (this.valid < 0) {
               throw new IluSomeSystemException(
                   "passport not valid outside of call"
                   );
           }
           iiContainer = getCachedIdentity(it);
           if (iiContainer != null) {
               return iiContainer;
           }
           iiContainer = new IluIdentity(it);
               //must be native set before handled to application...
           if (iiContainer == null) {
               throw new IluSomeSystemException(
                   "problem with IluIdentityType"
                   );
           }
           int key = nFindIdentity(this.call, it, iiContainer);
               //now iiContainer is native set; must test if successfull
           if (key==2) {
               iiContainer.owner = this; 
               //Ilu kernel said that passport owns memory. 
               //Must prevent garbage collection of passport while 
               //idenity is in use. (This is unusual because we first
               //try to make a copy if an identity is not unencumbered.)
           }
           if (iiContainer.yIluIdentity != 0) {
               //indicates success
               this.cachedIdentities.put(it, iiContainer);
               return iiContainer;
           }
        }
        return null;
    } //findIdentity
    

    /**
     * Adds an identity to this passport. <p>
     *
     * A passport (currently) can hold only one identity
     * of each type. <br>
     * Passport must be mutable or an exception is thrown. <br>
     *
     * In certain (rare) cases an identity may be included at most 
     * to one passport; this is tested and an exception thrown
     * if violated. 
     */
    public void addIdentity(IluIdentity ii) {
        if (ii == null) {
            throw new IluSomeSystemException("null identity");
        }
        synchronized (this) {
            IluIdentityType it = ii.it;
            if (this.valid != 0) {
                throw new IluSomeSystemException(
                    "passport not mutable"
                    );
            }
            IluIdentity previous = this.getCachedIdentity(it);
            if (previous != null) {
               throw new IluSomeSystemException(
                   "passport already contains identity of this type"
                   );
            }
            int returnKey = nAddIdentity(ii);
            if (returnKey < 0) {
                //There are several reasons this might fail...
                //(Among others the rare memory ownership scenario)
                throw new IluSomeSystemException(
                    "couldn't add identity " + returnKey
                    );
            } else {
                //indicates success
                this.cachedIdentities.put(it, ii);
            }
        }
    } //addIdentity
    

    /**
     * Since native memory disappears when a call is finished
     * ilu need to be able to disable the passport before that happens. 
     */
    /*friendly*/ void disable() {
        this.call = null;
        if (this.valid > 0) {
            synchronized (this) {
                this.valid = -1;
            }
        }
    } //disable
    
    
    /**
     * Returns the passport used for the current thread accepting 
     * and serving the current method.  This passport is immutable 
     * for clients. <p>
     * Can be used from a true method within an accepting call 
     * only or an exception is thrown.
     */
    public static IluPassport getAcceptingPassport() {
        java.lang.Thread thread = java.lang.Thread.currentThread();
        if (thread instanceof IluServiceThread) {
            IluCall call = ((IluServiceThread)thread).skeletonCall;
            if (call.skeletonAcceptingPassport == null) {
                IluPassport p = new IluPassport(call);
                call.skeletonAcceptingPassport = p;
                //No locking involved: we are using the call of 
                //the currentThread only.
                //
                //We need either a native call to set up the passport, or,
                //use the native passport only while it can be reconstructed
            }
            return call.skeletonAcceptingPassport;
        } else {
            throw new IluSomeSystemException("Not an ILU thread");
        }
    } //getAcceptingPassport


    /** 
     * Sets the IluPassport used for initiating ILU calls
     * from this thread. <p>
     *
     * Use null IluPassport to clear. <br>
     * Throws an exception if the current thread is not an IluServiceThread.<br>
     *
     * @see IluServiceThread
     * @see IluServiceThread#cleanAllStubFeatures
     */
    public static void setInitiatingPassport(IluPassport passport) {
        IluServiceThread.setInitiatingPassport(passport);
    } //setInitiatingPassport
    
    
} //IluPassport
