/* Avira AntiVir Guard (Monitor). Monitor AvGuard activity in Gnome.
   Written by Avira GmbH <support@avira.com>

   Copyright (c) 2007 Avira GmbH
   All rights reserved.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/file.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <panel-applet.h>
#include <gtk/gtkimage.h>
#include <gtk/gtk.h>

#define LOCKFILE "/var/tmp/avira/guard.lockfile"
#define MESSAGEDIR "/var/tmp/avira/guard.msg"

#define OPEN_SCHIRM "/usr/share/pixmaps/avira-guard-open.png"
#define CLOSED_SCHIRM "/usr/share/pixmaps/avira-guard-closed.png"

#define MAXMESSAGELENGTH 1024

struct avguard_schirm_data
{
	/*
	 * original image
	 */
	GtkWidget *orig_image;

	/*
	 * resized image and size
	 */
	GtkWidget *cached_image;
	GtkRequisition cached_size;
};

struct avguard_check_data
{
	/*
	 * open schirm, closed schirm
	 */
	struct avguard_schirm_data open;
	struct avguard_schirm_data closed;

	/*
	 * handle to this applet
	 */
	PanelApplet *applet;

	/*
	 * handle to the schirm currently displayed
	 */
	struct avguard_schirm_data *contained;

	/*
	 * size of this applet
	 */
	GtkRequisition applet_size;

	/*
	 * flag representing Guard status
	 *     TRUE = active
	 *     FALSE = not active
	 */
	gboolean is_guard_active;
};

/*
 * single, global data structure
 */
static struct avguard_check_data avguard_data;

/*
 * flag to protect against the "main" function
 * being called multiple times
 */
static gboolean avguard_started = FALSE;

/*
 * This function is called repeatedly in a separate thread.
 * The purpose of this function is to make sure the
 * "is_guard_active" flag is correctly set.
 */
static void avguard_run_check(void)
{
	int fd;
	gboolean found_guard;

	/*
	 * originally assume the guard is not active
	 */
	found_guard = FALSE;

	fd = open(LOCKFILE, O_RDWR);
	if (fd >= 0)
	{
		/*
		 * The lockfile exists and we can access it.
		 * We now check if we can grab and exclusive lock.
		 * If yes, the guard is not active (and our
		 * original assumption is correct).
		 */
		if (flock(fd, LOCK_EX | LOCK_NB) == 0)
		{
			/*
			 * we got the lock, guard not active
			 */
			flock(fd, LOCK_UN);
		}
		else
		{
			if (errno == EWOULDBLOCK)
			{
				/*
				 * Someone else has the lock so
				 * we assume it is an active guard.
				 */
				found_guard = TRUE;
			}
		}
		close(fd);
	}

	if (found_guard)
	{
		/*
		 * the guard is active, update the flag if needed
		 */
		if (avguard_data.is_guard_active != TRUE)
		{
			avguard_data.is_guard_active = TRUE;
		}
	}
	else
	{
		/*
		 * the guard is not active, update the flag if needed
		 */
		if (avguard_data.is_guard_active != FALSE)
		{
			avguard_data.is_guard_active = FALSE;
		}
	}
}

/*
 * This function is started as a separate thread.
 * It will simply check the guard at a particular
 * interval.
 * XXX: The thread does not ever end. It is assumed
 *      the system will clean up upon exit().
 */
static gpointer avguard_run_watcher(gpointer data)
{
	do
	{
		avguard_run_check();
		g_usleep(500000);
	}
	while (1);
}

/*
 * This function will generate a non-modal GTK dialog
 * with the provided alert message.
 */
static void avguard_show_message(const char *message)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE, _("Avira AntiVir Alert"));
	gtk_window_set_title(GTK_WINDOW(dialog), _("Avira AntiVir UNIX Workstation"));
	gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog), "%s", message);
	g_signal_connect_swapped(dialog, "response", G_CALLBACK(gtk_widget_destroy), dialog);
	gtk_widget_show(dialog);
}

/*
 * Given a particular filename, this function will read
 * its contents, unlink the file, and generate a dialog
 * box.
 */
static void avguard_display_unlink_file(const char *filename)
{
	FILE *file;

	file = fopen(filename, "r");
	if (file != NULL)
	{
		/*
		 * The message is only displayed if
		 * we were able to unlink the file.
		 */

		if (unlink(filename) == 0)
		{
			char *buffer;

			buffer = malloc(MAXMESSAGELENGTH);

			if (buffer != NULL)
			{
				memset(buffer, 0, MAXMESSAGELENGTH);

				if (fread(buffer, 1, MAXMESSAGELENGTH-1, file) > 0)
				{
					/*
					 * We read something, so let's show it!
					 */
					avguard_show_message(buffer);
				}

				free(buffer);
			}
		}

		fclose(file);
	}
}

/*
 * This function will look for files within the message
 * directory. Any file found will be processed. The
 * function is called at regular intervals using a
 * time in the main thread.
 */
static void avguard_check_messagedir(void)
{
	DIR *dirp;
	struct dirent *direntp;

	dirp = opendir(MESSAGEDIR);
	if (dirp != NULL)
	{
		do
		{
			/*
			 * loop for each entry in the directory
			 */
			direntp = readdir(dirp);
			if (direntp != NULL)
			{
				if (direntp->d_name != NULL)
				{
					char *filename;
					size_t size;

					/*
					 * we build the full path to the file
					 */

					size = strlen(MESSAGEDIR) + 1 + strlen(direntp->d_name) + 1;
					filename = malloc(size);
					if (filename != NULL)
					{
						struct stat statbuf;

						snprintf(filename, size, "%s/%s", MESSAGEDIR, direntp->d_name);

						/*
						 * make sure it is a regular file
						 */

						if (stat(filename, &statbuf) == 0)
						{
							if (S_ISREG(statbuf.st_mode))
							{
								/*
								 * Process file and set exit
								 * conditions for the loop. We
								 * stop now to prevent possible
								 * "popup flooding".
								 */

								avguard_display_unlink_file(filename);
								direntp = NULL;
							}
						}

						free(filename);
					}
				}
			}
		}
		while (direntp != NULL);

		closedir(dirp);
	}
}

/*
 * This function assigns the given schirm to be used
 * as the displayed schirm. If a schirm was previously
 * displayed, it is removed from the panel. However,
 * this function does not actually cause the re-display
 * to occur.
 */
static void avguard_set_schirm(struct avguard_schirm_data *data)
{
	if (avguard_data.contained != NULL)
	{
		/*
		 * remove the previously displayed
		 * schirm from the applet
		 */
		gtk_container_remove(GTK_CONTAINER(avguard_data.applet), avguard_data.contained->cached_image);
	}

	/*
	 * set the given schirm and add it
	 * to the applet
	 */

	avguard_data.contained = data;
	gtk_container_add(GTK_CONTAINER(avguard_data.applet), avguard_data.contained->cached_image);
}

/*
 * This function will take the given schirm and
 * resize it to the given size. If the given schirm
 * is also the schirm currently on display, then we
 * clear the variable so "no schirm" is on display.
 * Either way the old schirm is freed (finalized).
 * However, this function does not actually cause
 * any re-display to occur.
 */
static void avguard_resize_schirm(struct avguard_schirm_data *data, gint width, gint height)
{
	GdkPixbuf *buf1;
	GdkPixbuf *buf2;

	/*
	 * grab the original image
	 */
	buf1 = gtk_image_get_pixbuf(GTK_IMAGE(data->orig_image));

	/*
	 * resize the original image
	 */
	buf2 = gdk_pixbuf_scale_simple(buf1, width, height, GDK_INTERP_HYPER);

	if (data->cached_image != NULL)
	{
		if (data == avguard_data.contained)
		{
			/*
			 * this is the schirm on display,
			 * remove it from display variable
			 */
			avguard_data.contained = NULL;
		}

		/*
		 * Unref and destroy old resized image (which may
		 * or may not have been on display). If the image
		 * was being displayed, this will also remove it
		 * from the applet. The image will then be
		 * finalized.
		 */

		g_object_unref(data->cached_image);
		gtk_widget_destroy(data->cached_image);
	}

	/*
	 * set up resized image variables and
	 * increment refcount
	 */

	data->cached_image = GTK_WIDGET(g_object_ref(gtk_image_new_from_pixbuf(buf2)));
	data->cached_size.width = width;
	data->cached_size.height = height;
}

/*
 * This function will check the guard active flag and
 * the applet size to see if an image refresh needs
 * to occur. This function will actually cause the
 * re-display if needed. The function is called at
 * regular intervals using a time in the main thread.
 */
static void avguard_check_icon(void)
{
	gboolean changes;

	/*
	 * assume the image is already correct
	 */
	changes = FALSE;

	/*
	 * first we check if the guard activity has changed
	 */

	if (avguard_data.is_guard_active)
	{
		if (avguard_data.contained != &(avguard_data.open))
		{
			/*
			 * wrong schirm is displayed, set the
			 * new schirm and flag changes
			 */

			avguard_set_schirm(&(avguard_data.open));
			changes = TRUE;
		}
	}
	else
	{
		if (avguard_data.contained != &(avguard_data.closed))
		{
			/*
			 * wrong schirm is displayed, set the
			 * new schirm and flag changes
			 */

			avguard_set_schirm(&(avguard_data.closed));
			changes = TRUE;
		}
	}

	/*
	 * next we check if a resize has occurred
	 */

	if (avguard_data.contained->cached_size.width != avguard_data.applet_size.width
		|| avguard_data.contained->cached_size.height != avguard_data.applet_size.height)
	{
		struct avguard_schirm_data *cur;

		/*
		 * applet is resized, update image
		 * and flag changes
		 */

		cur = avguard_data.contained;
		avguard_resize_schirm(cur, avguard_data.applet_size.width, avguard_data.applet_size.height);
		avguard_set_schirm(cur);
		changes = TRUE;
	}

	if (changes)
	{
		/*
		 * something changed, re-display
		 */
		gtk_widget_show_all(GTK_WIDGET(avguard_data.applet));
	}
}

/*
 * This function is called by the timer at
 * regular intervals. It triggers the image
 * and message checks.
 */
static gboolean avguard_action_check(gpointer data)
{
	avguard_check_icon();
	avguard_check_messagedir();

	return TRUE;
}

/*
 * This function is called when a resize
 * event occurs. It simply updates the
 * applet size variables. (The actual resize
 * occurs on the timer event.)
 */
static gboolean avguard_resize_event(GtkWidget *applet, gint size)
{
	avguard_data.applet_size.width = size;
	avguard_data.applet_size.height = size;

	return TRUE;
}

/*
 * The "main" function of the applet. It
 * tries to create the lock file if it does
 * not exist and initializes the thread and
 * callback functions.
 */
static gboolean avguard_applet_fill(PanelApplet *applet, const gchar *iid, gpointer data)
{
	/*
	 * make sure this is our applet
	 */
	if (strcmp(iid, "OAFIID:AvGuardApplet") != 0)
		return FALSE;

	/*
	 * this function should never run twice
	 */

	if (avguard_started == FALSE)
	{
		avguard_started = TRUE;
	}
	else
	{
		return TRUE;
	}

	/*
	 * try to create the lock file if
	 * it does not exist
	 */
	{
		FILE *file;
		file = fopen(LOCKFILE, "a");
		if (file != NULL)
			fclose(file);
	}

	/*
	 * initialize the single, global structure
	 */

	memset(&avguard_data, 0, sizeof(avguard_data));

	avguard_data.applet = applet;
	avguard_data.open.orig_image = GTK_WIDGET(g_object_ref(gtk_image_new_from_file(OPEN_SCHIRM)));
	avguard_data.closed.orig_image = GTK_WIDGET(g_object_ref(gtk_image_new_from_file(CLOSED_SCHIRM)));

	avguard_data.is_guard_active = FALSE;

	/*
	 * set the applet to auto-expand
	 */
	panel_applet_set_flags(applet, PANEL_APPLET_EXPAND_MINOR);

	/*
	 * grab the size of the applet
	 */
	{
		guint size;

		size = panel_applet_get_size(applet);

		avguard_data.applet_size.width = size;
		avguard_data.applet_size.height = size;
	}

	/*
	 * register the resize callback
	 */
	g_signal_connect(G_OBJECT(applet), "change_size", GTK_SIGNAL_FUNC(avguard_resize_event), NULL);

	/*
	 * check the guard active status (so that
	 * we start off with the correct image)
	 */
	avguard_run_check();

	/*
	 * create the image
	 */
	avguard_action_check(NULL);

	/*
	 * start the thread to monitor the guard
	 * active status
	 */

	if (!g_thread_supported())
		g_thread_init(NULL);
	g_thread_create(avguard_run_watcher, NULL, FALSE, NULL);

	/*
	 * add a timer to monitor changes needed in
	 * the image and message files
	 */
	g_timeout_add(500, avguard_action_check, NULL);

	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY("OAFIID:AvGuardApplet_Factory",
	PANEL_TYPE_APPLET,
	"Avira AntiVir Guard (Monitor)",
	"0",
	avguard_applet_fill,
	NULL);

