-- ---- example.lua ---------------------------------------------
-- sample Lua script, demonstrates how to interface with Dazuko
-- (OO and procedure style, see bottom)
--
-- synopsis: [lua] example.lua <include> ...
--
-- Copyright (c) 2005 Gerhard Sittig
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
--
-- 1. Redistributions of source code must retain the above copyright notice,
-- this list of conditions and the following disclaimer.
--
-- 2. Redistributions in binary form must reproduce the above copyright notice,
-- this list of conditions and the following disclaimer in the documentation
-- and/or other materials provided with the distribution.
--
-- 3. Neither the name of Dazuko nor the names of its contributors may be used
-- to endorse or promote products derived from this software without specific
-- prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.

require 'dazuko'

-- handle a request
function dump(acc)
	-- dump the details
	io.write("\n")
	io.write("got an access:\n")
	table.foreach(acc, function(k, v)
		io.write("\t", tostring(k)
			,"\t", tostring(v)
			,"\n")
	end)
	-- return 0 -> grant access
	return(0)
end

-- preferences
local group, mode, mask, incl, excl

-- the main routine using OO style
function main_obj()
	local d, rc, ver

	-- preparation
	d = DAZUKO()
	ver = d:ioversion()
	io.write("IO version: ", ver['text'], "\n")
	rc = d:register(group, mode)
	ver = d:version()
	io.write("module version: ", ver['text'], "\n")
	rc = d:setmask(mask)
	rc = d:addincl(incl)
	rc = d:addexcl(excl)

	-- loop getting accesses
	-- XXX how to register for SIGINT?
	while d:getaccess(dump) do
		-- EMPTY
	end

	-- cleanup
	rc = d:removeall()
	rc = d:unregister()
	d = nil
end

-- the main routine accessing libdazuko directly (procedural style)
function main_lib()
	local id, rc, ver

	-- preparation
	ver = libdazuko.ioversion()
	io.write("IO version: ", ver['text'], "\n")
	id = libdazuko.register(group, mode)
	if (id == nil) then
		return
	end
	ver = libdazuko.version()
	io.write("version: ", ver['text'], "\n")
	rc = libdazuko.setmask(id, mask)
	rc = libdazuko.addincl(id, unpack(incl))
	rc = libdazuko.addexcl(id, unpack(excl))

	-- loop getting accesses
	-- XXX how to register for SIGINT?
	while libdazuko.getaccess(id, dump) do
		-- EMPTY
	end

	-- cleanup
	rc = libdazuko.removeall(id)
	rc = libdazuko.unregister(id)
end

-- check parameters
function prep()
	local d
	local i

	-- load the library
	-- actually: have it loaded as a "side effect"
	d = DAZUKO()
	if (d == nil) then
		io.write("cannot load libdazuko / create instance\n")
		os.exit(1)
	end
	d = nil

	-- catch signals (they still interrupt getaccess()
	-- but they don't throw exceptions any longer)
	libdazuko.swallowsig()

	-- set up default values
	group = 'example:lua'
	mode = 'rw'
	mask = libdazuko.DAZUKO_ON_OPEN + libdazuko.DAZUKO_ON_CLOSE
	incl = {}
	excl = { '/proc/', '/dev/' }

	-- scan cmdline parameters
	for i = 1, table.getn(arg) do
		table.insert(incl, arg[i])
	end

	-- sanity check
	if (incl[1] == nil) then
		io.write("need at least one include path\n")
		os.exit(1)
	end

	-- dump parameters
	io.write("will register with group \"", group, "\" and mode \"", mode, "\"\n")
	io.write("access mask will be ", mask, "\n")
	io.write("include directories are:")
	table.foreach(incl, function(k, v) io.write("\n\t", v) end)
	io.write("\n")
	io.write("exclude directories are:")
	table.foreach(excl, function(k, v) io.write("\n\t", v) end)
	io.write("\n")
end

-- program entry point
prep()
if (true) then
	main_obj()
else
	main_lib()
end

-- ---- E O F ---------------------------------------------------
