#include <stdio.h>
#include <stdinc.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <X11/Xlib.h>
#include <X11/extensions/xf86dga.h>
#include <gdk/gdktypes.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <gtk/gtkeventbox.h>

#include <linux/types.h>
#include <linux/videodev.h>

GtkWidget *window;
GtkWidget *tv;
GtkWidget *painter;
GdkGC *painter_gc;
GdkColor chroma_colour;
GdkImage *capture;		/* Capture Image */
GdkImageType capture_type;
gint capture_h,capture_w;
gint window_h, window_w;

/*
 *	TV card driving.
 */

static struct video_window vwin;
static struct video_picture vpic;
static struct video_buffer vbuf;
static struct video_capability vcap;
static struct video_audio vaudio;
static int grabber_format;
static int capture_running;

static int tv_fd;

/*
 *	Methods to work with
 */
 
static int kill_on_overlap = 0;	/* Set for overlay cards that dont clip */
static int clean_display = 0;		/* Set on overlay cards that do DMA */
static int chroma_key = 0;		/* Set for chromakey cards */
static int fixed_size = 0;		/* Size is fixed by the card (we dont scale yet) */
static int use_shm = 0;		/* Capture only card */
static int xsize = 160, ysize = 120;	/* Initial size */
static int xmin, ymin;			/* Limits */
static int xmax, ymax;	

static GdkColor colourmap[256];		/* Colour map */
static int colourmap_size;
static int colourmap_base;
static GdkColormap *private_colourmap;

static int sanity1=0;
static int sanity2=0;


static void make_palette_grey(int num)
{
	char buf[16];
	int i,x=0;
	int step=256/num;
	
	for(i=0;i<256; i+=step)
	{
		sprintf(buf,"#%02X%02X%02X",
			i,i,i);
		gdk_color_parse(buf, &colourmap[x]);
		gdk_color_alloc(gdk_colormap_get_system(), &colourmap[x]);
		x++;
	}
	colourmap_size=num;
	colourmap_base=0;
}

static void make_colour_cube(void)
{
	char buf[16];
	int i;
	
	private_colourmap = gdk_colormap_new(gdk_visual_get_best(),256);
	for(i=0;i<16;i++)
	{
		gdk_color_black(private_colourmap, &colourmap[i]);
	}
	for(i=0;i<225;i++)
	{
		int r, rr;
		r=i%25;
		rr=r%5;
		sprintf(buf,"#%02X%02X%02X",
			(65535*(r/5)/4),
			(65535*(i/25)/8),
			(65535*rr/4));
		gdk_color_parse(buf, &colourmap[i+16]);
		gdk_color_alloc(private_colourmap, &colourmap[i+16]);
	}
	colourmap_size=225;
	colourmap_base=16;
	gdk_window_set_colormap(tv->window, private_colourmap);
	gdk_window_set_colormap(window->window, private_colourmap);
}

static int open_tv_card(int n)
{
	Display *disp;
	int fd;
	unsigned int rwidth;
	int bank, ram, major, minor;
	
	char buf[64];
	
	sprintf(buf,"/dev/video%d",n);
	
	fd=open(buf,O_RDWR);
	
	
	if(fd==-1)
	{
		perror("open");
		return -1;
	}
	
	if(ioctl(fd, VIDIOCGAUDIO, &vaudio)==0)
	{
		vaudio.flags|=VIDEO_AUDIO_MUTE;
		ioctl(fd, VIDIOCSAUDIO, &vaudio);
	}

	if(ioctl(fd, VIDIOCGCAP, &vcap))
	{
		perror("get capabilities");
		return -1;
	}
	
	/*
	 *	Use black as the background unless we later find a 
	 *	chromakey device
	 */
	 
	gdk_color_black(gdk_colormap_get_system(), &chroma_colour);
	 
	/* 
	 *	Now see what we support
	 */
	 
	if(!(vcap.type&VID_TYPE_OVERLAY))
	{
		if(vcap.type&VID_TYPE_CAPTURE)
			use_shm = 1;
		else
		{
			fprintf(stderr,"I don't know how to handle this device.\n");
			return -1;
		}
	}
	else
	{
		/* Ok now see what method to use */
		if(vcap.type&VID_TYPE_CLIPPING)
		{
		}
		else if(vcap.type&VID_TYPE_CHROMAKEY)
		{
			chroma_key = 1;
			/* We want the most vile unused colour we can get
			   for our chroma key - this is pretty foul */
			gdk_color_parse("#FA77A1", &chroma_colour);
			gdk_color_alloc(gdk_colormap_get_system(), &chroma_colour);
		}
		else
			kill_on_overlap = 1;
		if(vcap.type&VID_TYPE_FRAMERAM)
			clean_display = 1;
		
	}
	
	/*
	 *	Set the window size constraints
	 */
	 
	if(!(vcap.type&VID_TYPE_SCALES))
	{
		xmin = xmax = xsize = vcap.maxwidth;
		ymin = ymax = ysize = vcap.maxheight;
		fixed_size = 1;
	}
	else
	{
		xmin = vcap.minwidth;
		ymin = vcap.minheight;
		xmax = vcap.maxwidth;
		ymax = vcap.maxheight;
		xsize = min(max(xsize, xmin), xmax);
		ysize = min(max(ysize, ymin), ymax);
	}

	/*
	 *	Now get the video parameters (one day X will set these)
	 */
	 
	disp=GDK_DISPLAY();
	if (XF86DGAQueryVersion(disp, &major, &minor)) 
		XF86DGAGetVideoLL(disp, DefaultScreen(disp), &(vbuf.base),
		&rwidth, &bank, &ram);
	else 
	{
		fprintf(stderr,"XF86DGA: DGA not available.\n");
		return -1;
	}

	/*
	 *	Hunt the visual
	 */
	 
	vbuf.depth = gdk_visual_get_best_depth();
	vbuf.bytesperline = rwidth*vbuf.depth/8;
	
	/*
	 *	Set the overlay buffer up
	 */
	 
	if(ioctl(fd, VIDIOCSFBUF, &vbuf)==-1)
	{
		perror("Set frame buffer");
		/*
		 *	If we can't use MITSHM then we can't overlay if
		 *	the board can't handle our framebuffer...
		 *
		 *	For SHM we can do post processing.
		 */
		 
		if(!use_shm)
			return -1;
	}

	/*
	 *	Set the format
	 */
 
	ioctl(fd, VIDIOCGPICT , &vpic);

	if(use_shm)
	{	
	
		/*
		 *	Try 24bit RGB, then 565 then 555. Arguably we ought to
		 *	try one matching our visual first, then try them in 
		 *	quality order
		 *
		 */
		
		
		if(vcap.type&VID_TYPE_MONOCHROME)
		{
			vpic.depth=8;
			vpic.palette=VIDEO_PALETTE_GREY;	/* 8bit grey */
			if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
			{
				vpic.depth=6;
				if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
				{
					vpic.depth=4;
					if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
					{
						fprintf(stderr, "Unable to find a supported capture format.\n");
						return -1;
					}
				}
			}
			/*
			 *	Make a grey scale cube - only go to 6bit
			 */
			 
			if(vpic.depth==4)
				make_palette_grey(16);
			else
				make_palette_grey(64);
			
		}
		else
		{
			vpic.depth=24;
			vpic.palette=VIDEO_PALETTE_RGB24;
		
			if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
			{
				vpic.palette=VIDEO_PALETTE_RGB565;
				vpic.depth=16;
				
				if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
				{
					vpic.palette=VIDEO_PALETTE_RGB555;
					vpic.depth=15;
				
					if(ioctl(tv_fd, VIDIOCSPICT, &vpic)==-1)
					{
						fprintf(stderr, "Unable to find a supported capture format.\n");
						return -1;
					}
				}
			}
		}
	}
	/*
	 *	Soft overlays by chroma etc we dont have to care about
	 *	the format for.. but DMA to frame buffer we do
	 */
	else if(vcap.type&VID_TYPE_FRAMERAM)
	{
		vpic.depth = vbuf.depth;
		switch(vpic.depth)
		{
			case 8:
				vpic.palette=VIDEO_PALETTE_HI240;	/* colour cube */
				make_colour_cube();
				break;
			case 15:
				vpic.palette=VIDEO_PALETTE_RGB555;
				break;
			case 16:
				vpic.palette=VIDEO_PALETTE_RGB565;
				break;
			case 24:
				vpic.palette=VIDEO_PALETTE_RGB24;
				break;
			case 32:		
				vpic.palette=VIDEO_PALETTE_RGB32;
				break;
			default:
				fprintf(stderr,"Unsupported X11 depth.\n");
		}
		if(ioctl(fd, VIDIOCSPICT, &vpic)==-1)
		{
			/* Depth mismatch is fatal on overlay */
			perror("set depth");
			return -1;
		}
	}
		
	if(ioctl(fd, VIDIOCGPICT, &vpic)==-1)
		perror("get picture");
			
	grabber_format = vpic.palette;
	
	sanity1=1;
	return fd;
}	


static void close_tv_card(int handle)
{
	close(handle);
}

/*
 *	Capture an image from capture cards
 */

#define READ_VIDEO_PIXEL(buf, format, depth, r, g, b)			\
{									\
	switch (format)							\
	{								\
		case VIDEO_PALETTE_GREY:				\
			switch (depth)					\
			{						\
				case 4:					\
				case 6:					\
				case 8:					\
					(r) = (g) = (b) = (*buf++ << 8);\
					break;				\
									\
				case 16:				\
					(r) = (g) = (b) = 		\
						*((guint16 *) buf);	\
					buf += 2;			\
					break;				\
			}						\
			break;						\
									\
									\
		case VIDEO_PALETTE_RGB565:				\
		{							\
			unsigned short tmp = *(unsigned short *)buf;	\
			(r) = tmp&0xF800;				\
			(g) = (tmp<<5)&0xFC00;				\
			(b) = (tmp<<11)&0xF800;				\
			buf += 2;					\
		}							\
		break;							\
									\
		case VIDEO_PALETTE_RGB555:				\
			(r) = (buf[0]&0xF8)<<8;				\
			(g) = ((buf[0] << 5 | buf[1] >> 3)&0xF8)<<8;	\
			(b) = ((buf[1] << 2 ) & 0xF8)<<8;		\
			buf += 2;					\
			break;						\
									\
		case VIDEO_PALETTE_RGB24:				\
			(r) = buf[0] << 8; (g) = buf[1] << 8; 		\
			(b) = buf[2] << 8;				\
			buf += 3;					\
			break;						\
									\
		default:						\
			fprintf(stderr, 				\
				"Format %d not yet supported\n",	\
				format);				\
	}								\
}						


#define PUT_X11_PIXEL(buf, endian, depth, bpp, r, g, b, gl_map)		\
{									\
	switch (depth)							\
	{								\
		case  1: /* duh?  A Sun3 or what?? */			\
			lum = 3*(r) + 5*(g) + 2*(b);			\
			if (lum >= 5*0x8000)				\
				*buf |= dst_mask;			\
			dst_mask <<= 1;					\
			if (dst_mask > 0xff)				\
			{						\
				buf += (bpp);				\
				dst_mask = 0x01;			\
			}						\
			break;						\
									\
		case  8:	/* FIXME: Mono only right now */	\
			lum = ((3*(r) + 5*(g) + 2*(b)) / (10 * 256/colourmap_size)) >> 8;	\
			if (lum >= colourmap_size)			\
				lum = colourmap_size;			\
			buf[0] = colourmap[lum+colourmap_base].pixel;	\
			buf += (bpp);					\
			break;						\
		case 15:						\
			rgb = (  (((r) >> 11) << 10)	/* 5 bits of red */		\
			| (((g) >> 11) <<  5)	/* 5 bits of green */	\
			| (((b) >> 11) <<  0));	/* 5 bits of blue */	\
			((guint16 *)buf)[0] = rgb;			\
			buf += (bpp);					\
			break;						\
									\
		case 16:						\
			rgb = (  (((r) >> 11) << 11)	/* 5 bits of red */	\
			| (((g) >> 10) <<  5)	/* 6 bits of green */	\
			| (((b) >> 11) <<  0));	/* 5 bits of blue */	\
			((guint16 *)buf)[0] = rgb;			\
			buf += (bpp);					\
			break;						\
									\
		case 24:						\
			/* Is this correctly handling all byte order cases? */		\
			if ((endian) == GDK_LSB_FIRST)			\
			{						\
				buf[0] = (b) >> 8; buf[1] = (g) >> 8; buf[2] = (r) >> 8;\
			}						\
			else						\
			{						\
				buf[0] = (r) >> 8; 			\
				buf[1] = (g) >> 8; 			\
				buf[2] = (b) >> 8;			\
			}						\
			buf += (bpp);					\
			break;						\
									\
		case 32:						\
      /* Is this correctly handling all byte order cases?  It assumes	\
	 the byte order of the host is the same as that of the		\
	 pixmap. */							\
			rgb = (((r) >> 8) << 16) | (((g) >> 8) << 8) | ((b) >> 8);\
			((guint32 *)buf)[0] = rgb;			\
			buf += (bpp);					\
			break;						\
		default:						\
			fprintf(stderr, "Unsupported X11 depth.\n");	\
	}								\
}
 
static void grab_image(unsigned char *dst, int xpos, int ypos)
{
	int x;
	int lines=0;
	int pixels_per_line, bytes_per_line, byte_order, bytes_per_pixel;
	int dst_depth, src_depth;
	guint32 r = 0, g = 0, b = 0, lum, rgb, src_mask, dst_mask;
	unsigned char *mem, *src;	

	src_depth = vpic.depth;
	dst_depth = capture->depth;
	
	pixels_per_line = capture_w;
	bytes_per_line = capture->bpl;
	bytes_per_pixel = capture->bpp;
	byte_order = capture->byte_order;

	mem=malloc(bytes_per_line * capture_h);
	if(mem==NULL)
		return;

	src=mem;
	
	read(tv_fd, mem, bytes_per_line * capture_h);
			
	x = 0;

	src_mask = 0x80;
	dst_mask = 0x01;
	
	while (lines<capture_h)
	{
		READ_VIDEO_PIXEL(src, grabber_format, src_depth, r, g, b);
		PUT_X11_PIXEL(dst, byte_order, dst_depth, bytes_per_pixel, r, g, b,
			win.canvas.graylevel);
		if (++x >= pixels_per_line)
		{
			x = 0;
			dst += bytes_per_line - pixels_per_line * bytes_per_pixel;
			lines++;
		}
	}
	free(mem);
	printf("Used %d bytes\n", src-mem);
}

/*
 *	Paint the drawing area. We use this to keep it clean when not
 *	capturing, for chromakey areas and for mitshm.
 */
 
static void do_painting(void)
{
	gint x,y,w,h,d;
	
	/*
	 *	Called during setup - ignore	 
	 */
	 
	if(painter_gc==NULL)
		return;
		
		
        gdk_window_get_geometry(painter->window, &x, &y, &w, &h, &d);

	if(use_shm && capture && capture_running)
	{
		gdk_draw_image(painter->window, painter_gc, capture,
			0,0,0,0, 
			capture_w, capture_h);
	}
	else
		gdk_draw_rectangle(painter->window, painter_gc, 1, 0, 0, w, h);
	gdk_flush();
}
		

static void tv_grabber(void)
{
	grab_image(capture->mem, 0, 0);
	do_painting();
}

static void tv_capture(int handle, int onoff)
{
	if(use_shm==0)
	{
		if(sanity1 && sanity2!=onoff)
		{
			sanity2=onoff;
			if(ioctl(handle, VIDIOCCAPTURE, &onoff)==-1)
				perror("Capture");
		}
	}
	else
	{
		static gint capture_tag;
		if(capture_running && capture && capture->mem)
			grab_image(capture->mem, 0,0);
		if(onoff)
			capture_tag = gtk_idle_add(tv_grabber, 0);
		else
			gtk_idle_remove(capture_tag);
	}
	capture_running = onoff;
}


static void tv_set_window(int handle)
{
	gint x,y,w,h,d;
	GdkWindow *t=tv->window;
	struct video_window vw;
	
	/*
	 *	GDK holds the window positioning
	 */
	
        gdk_window_get_geometry( t, &x, &y, &w, &h, &d);
        
        /*
         *	We are working on screen positions with the overlay
         *	devices, so see where we are relative to the screen.
         */
        
        gdk_window_get_origin(t, &x, &y);
	
	if(w < xmin || h < ymin)
	{
		/* Too small, turn it off */
		tv_capture(handle,0);
	}
	
	/* Constrain the window */
	
	h=min(h,ymax);
	w=min(w,xmax);
	
	window_h=h;
	window_w=w;
	
	vw.x = x;
	vw.y = y;
	vw.width = w;
	vw.height = h;
	vw.clips = NULL;
	vw.flags = 0;
	vw.clipcount = 0;
	if(h<32 || w<32)
		return;
	if(sanity1!=0)
	{
		if(ioctl(tv_fd, VIDIOCSWIN, &vw)==-1)
			perror("set video window");
		if(ioctl(tv_fd, VIDIOCGWIN, &vw)==-1)
			perror("get video window");
		/* x,y,h,w are now the values that were chosen by the
		   device to be as close as possible - eg the quickcam only
		   has 3 sizes */
		xsize = vw.width;
		ysize = vw.height;
		
		/*
		 *	Now check the image in question
		 */
		 
		printf("Capturing %d,%d\n", xsize, ysize);
		
		if(use_shm)
		{
			if(capture==NULL || capture_w!=xsize || capture_h!=ysize)
			{
				capture_type = GDK_IMAGE_FASTEST;
				
				if(capture!=NULL)
				{
					gdk_image_destroy(capture);
					capture=NULL;
				}
#ifdef __alpha__
				/*
				 *	Taken from the sane xcam
				 */
				 
				/* Some X servers seem to have a problem with shared images that
				    have a width that is not a multiple of 8.  Duh... ;-( */
				if (xsize % 8)
					capture_type = GDK_IMAGE_NORMAL;
#endif
				
				capture=gdk_image_new(capture_type, 
					gdk_window_get_visual(t),
					xsize,ysize);
					
				capture_h=ysize;
				capture_w=xsize;
			}
			/*
			 *	Capture and draw
			 */
			 
			d=capture_running;
			capture_running=0;
			do_painting();
			capture_running=d;
			
			/*
			 *	Start grabbing again
			 */
			
			grab_image(capture->mem, x,y);
			
		}				
	}
}

static void show_pos(void)
{
	tv_set_window(tv_fd);
}


static void capture_on(void)
{
	tv_set_window(tv_fd);
	tv_capture(tv_fd, 1);
	do_painting();
}

static void capture_off(void)
{
	tv_capture(tv_fd,0);
	do_painting();
}

static void set_audio(GtkObject *foo, void *bar)
{
	int n=(int)bar;
	struct video_audio va;
	va.audio = 0;	/* For now FIXME FIXME */
	if(ioctl(tv_fd, VIDIOCGAUDIO, &va)==-1)
	{
		perror("get audio");
		return;
	}
	if(n)
		va.flags&=~VIDEO_AUDIO_MUTE;
	else
		va.flags|=VIDEO_AUDIO_MUTE;
		
	if(ioctl(tv_fd, VIDIOCSAUDIO, &va)==-1)
		perror("set audio");
}
		
static void configure_event(void)
{
	tv_set_window(tv_fd);
}

static void exposure_event(void)
{
	tv_set_window(tv_fd);
}



/*
 *	Check for a repaint job
 */
  
static gint painting_event(GtkWidget *self, GdkEvent *event)
{
	switch(event->type)
	{
		case GDK_EXPOSE:
			if(!painter_gc)
			{
				painter_gc=gdk_gc_new(painter->window);
				gdk_gc_set_foreground(painter_gc, &chroma_colour);
				gdk_gc_set_background(painter_gc, &chroma_colour);
			}	
			do_painting();
			break;
		default:
			break;
	}
	return FALSE;
}

 
static void set_channel(GtkObject *o, void *p)
{
	int channel=(int)p;
	if(ioctl(tv_fd, VIDIOCSCHAN, &channel)==-1)
		perror("set channel");
	/* FIXME : hack for now 8683 is my video ;) */
	if(p==0)
	{
		struct video_tuner vt;
		unsigned long f=8683;
		vt.tuner=0;
		vt.mode=VIDEO_MODE_PAL;
		ioctl(tv_fd, VIDIOCSTUNER, &vt);
		ioctl(tv_fd, VIDIOCSFREQ, &f);
	}
}

static void set_format(GtkObject *o, void *p)
{
	struct video_tuner vt;
	int format = (int)p;
	
	vt.tuner = 0;		/* FIXME : support multiple tuners */
	
	if(ioctl(tv_fd, VIDIOCGTUNER, &vt)==-1)
		perror("get tuner");

	vt.mode = format;
	
	if(ioctl(tv_fd, VIDIOCSTUNER, &vt)==-1)
		perror("set tuner");
}

		
/*
 *	Widget Stuff
 */
 
void format_bar(GtkWidget *menubar)
{
	GtkWidget* menu;
	GtkWidget* menuitem;
	struct video_tuner vt;
	int n;
	static char *name[4]= {
		"PAL",
		"NTSC",
		"SECAM",
		"AUTO"
	};

	menu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_widget_show(menuitem);

	for(n=0;n<4;n++)
	{
		
		vt.tuner = 0; 	/* FIXME: support multiple tuners */
		if(ioctl(tv_fd, VIDIOCGTUNER, &vt)==0)
		{
			vt.mode = n;
			if(ioctl(tv_fd, VIDIOCSTUNER, &vt)==-1)
				continue;
		}
		menuitem = gtk_menu_item_new_with_label(name[n]);
		gtk_menu_append(GTK_MENU(menu), menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) set_format,(void *)n);
		gtk_widget_show(menuitem);
	}

	menuitem = gtk_menu_item_new_with_label("Format");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu);
	gtk_container_add(GTK_CONTAINER(menubar), menuitem);
	gtk_widget_show(menuitem);
	vt.tuner = 0; 	/* FIXME: support multiple tuners */
	if(ioctl(tv_fd, VIDIOCGTUNER, &vt)==0)
	{
		vt.mode=0;
		ioctl(tv_fd, VIDIOCSTUNER, &vt);
	}
}

void channel_bar(GtkWidget *menubar)
{
	GtkWidget* menu;
	GtkWidget* menuitem;
	struct video_capability vc;
	struct video_channel chan;
	int n;
	char buf[256];

	menu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_widget_show(menuitem);

	if(ioctl(tv_fd, VIDIOCGCAP, &vc)==-1)
	{
		perror("get capabilities");
		return;
	}
	
	sprintf(buf, "Miguelivison 1.01: %s", vc.name);
	
	gtk_window_set_title(GTK_WINDOW(window), buf);
	
	for(n=0;n<vc.channels;n++)
	{
		chan.channel=n;
		if(ioctl(tv_fd, VIDIOCGCHAN, &chan)==-1)
		{
			perror("get channel");
			continue;
		}
		menuitem = gtk_menu_item_new_with_label(chan.name);
		gtk_menu_append(GTK_MENU(menu), menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) set_channel,(void *)n);
		gtk_widget_show(menuitem);
	}
	
	menuitem = gtk_menu_item_new_with_label("Input");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu);
	gtk_container_add(GTK_CONTAINER(menubar), menuitem);
	gtk_widget_show(menuitem);
}

GtkWidget *tv_menu_bar(void)
{
	GtkWidget* menubar;
	GtkWidget* menu;
	GtkWidget* menuitem;

	menubar = gtk_menu_bar_new();
	menu = gtk_menu_new();
	gtk_widget_show(menubar);

	menuitem = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_label("Show");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) show_pos, NULL);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_label("Capture On");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) capture_on, NULL);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_label("Capture Off");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) capture_off, NULL);
	gtk_widget_show(menuitem);

	if(vcap.audios!=0)
	{
		menuitem = gtk_menu_item_new_with_label("Audio On");
		gtk_menu_append(GTK_MENU(menu), menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) set_audio, (void *)1);
		gtk_widget_show(menuitem);

		menuitem = gtk_menu_item_new_with_label("Audio Off");
		gtk_menu_append(GTK_MENU(menu), menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
	                     (GtkSignalFunc) set_audio, (void *)0);
		gtk_widget_show(menuitem);
	}
	
	menuitem = gtk_menu_item_new_with_label("Exit");
	gtk_menu_append(GTK_MENU(menu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                     (GtkSignalFunc) gtk_main_quit, NULL);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_label("File");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu);
	gtk_container_add(GTK_CONTAINER(menubar), menuitem);
	gtk_widget_show(menuitem);

	channel_bar(menubar);
	
	format_bar(menubar);
	
	return menubar;
}

void make_tv_set(void)
{
	GtkWidget* menubar;
	GtkWidget* hbox, *vbox;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
 
	gtk_signal_connect(GTK_OBJECT(window), "destroy", capture_off, NULL);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event", gtk_main_quit, NULL);
	gtk_signal_connect(GTK_OBJECT(window), "destroy", capture_off, NULL);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event", gtk_main_quit, NULL);


	vbox = gtk_vbox_new(FALSE, 0);

	/*
	 *	FIXME: now I can't resize below this user size. How do
	 *	I fix that ?
	 */
	 	
	hbox = tv = gtk_event_box_new();
	gtk_widget_set_usize(tv, xsize, ysize);
	
	painter=gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(painter), xsize, ysize);
	gtk_widget_set_events(painter, GDK_EXPOSURE_MASK);
	gtk_signal_connect(GTK_OBJECT(painter), "event", (GtkSignalFunc)
		painting_event, 0);
	gtk_widget_show(painter);
	gtk_container_add(GTK_CONTAINER(tv), painter);

	menubar = tv_menu_bar();
	gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(window), vbox);
	
	gtk_signal_connect(GTK_OBJECT(tv), "expose_event", 
		(GtkSignalFunc)exposure_event, NULL);
	gtk_signal_connect(GTK_OBJECT(tv), "configure_event",
		(GtkSignalFunc)configure_event, NULL);
		
	gtk_widget_set_events(tv, GDK_STRUCTURE_MASK|GDK_EXPOSURE_MASK);

	gtk_widget_show(hbox);
	gtk_widget_show(vbox);
	gtk_widget_show(window);
	tv_set_window(tv_fd);
}


/*
 *	Run the TV set
 */

int main(int argc, char *argv[])
{
	int n=0;
	gtk_init(&argc,&argv);
	if(argv[1])
	{
		sscanf(argv[1],"%d",&n);
	}
	if((tv_fd=open_tv_card(n))==-1)
		return 1;
	make_tv_set();
	gtk_main();
	close_tv_card(tv_fd);
	return 0;
}
