/* $Id: utmp.c,v 1.1 1995/05/24 18:53:29 stuart Exp $ */

/*{{{  Notes*/
/*

Written by Michael Haardt, 1994.

*/
/*}}}  */
/*{{{  #includes*/
#define _MINIX          1
#define _POSIX_SOURCE   1
#define _POSIX_C_SOURCE 2

#include <xdr/utmp.h>
#include <assert.h>
#include <fcntl.h>
#include <limits.h>
#include <unistd.h>
/*}}}  */

/*{{{  variables*/
static char name[_POSIX_PATH_MAX]=_PATH_UTMP;
static struct utmp utmp;
static off_t offset;
static int fd=-1;
/*}}}  */

/*{{{  utopen    -- open utmp file if not already open*/
static int utopen(void)
{
  if (fd==-1) fd=open(name,O_RDWR);
  return fd;
}
/*}}}  */

/*{{{  setutent  -- rewind utmp file*/
void setutent(void)
{
  if (utopen()!=-1) lseek(fd,0,SEEK_SET);
}
/*}}}  */
/*{{{  endutent  -- close utmp file*/
void endutent(void)
{
  if (fd!=-1) { close(fd); fd=-1; }
}
/*}}}  */
/*{{{  utmpname  -- set file name for further operations*/
void utmpname(const char *file)
{
  if (fd!=-1) { close(fd); fd=-1; }
  assert(strlen(file)<sizeof(name));
  strcpy(name,file);
}
/*}}}  */
/*{{{  getutent  -- read next utmp entry*/
struct utmp *getutent(void)
{
  char buf[XDR_UTMPSIZE];
  XDR xdrs;

  if (utopen()==-1) return (struct utmp*)0;
  offset=lseek(fd,0,SEEK_CUR);
  if (read(fd,buf,XDR_UTMPSIZE)!=XDR_UTMPSIZE) return (struct utmp*)0;
  xdrmem_create(&xdrs,buf,XDR_UTMPSIZE,XDR_DECODE);
  if (xdr_utmp(&xdrs,&utmp)==0) return (struct utmp*)0;
  xdr_destroy(&xdrs);
  return &utmp;
}
/*}}}  */
/*{{{  getutid   -- read next utmp entry with a specific type (and possibly id)*/
struct utmp *getutid(struct utmp *id)
{
  struct utmp u;

  u=*id;
  switch (u.ut_type)
  {
    case RUN_LVL:
    case BOOT_TIME:
    case OLD_TIME:
    case NEW_TIME:
    {
      while (u.ut_type!=utmp.ut_type) if (getutent()==(struct utmp*)0) return (struct utmp*)0;
      return &utmp;
    }
    case INIT_PROCESS:
    case LOGIN_PROCESS:
    case USER_PROCESS:
    case DEAD_PROCESS:
    {
      while ((utmp.ut_type!=INIT_PROCESS && utmp.ut_type!=LOGIN_PROCESS && utmp.ut_type!=USER_PROCESS && utmp.ut_type!=DEAD_PROCESS) || strncmp(utmp.ut_id,u.ut_id,sizeof(utmp.ut_id))) if (getutent()==(struct utmp*)0) return (struct utmp*)0;
      return &utmp;
    }
  }
  return (struct utmp*)0;
}
/*}}}  */
/*{{{  getutline -- return next LOGIN or USER entry with a specific line*/
struct utmp *getutline(struct utmp *line)
{
  struct utmp u;

  u=*line;
  while ((utmp.ut_type!=LOGIN_PROCESS && utmp.ut_type!=USER_PROCESS) || strncpy(u.ut_line,utmp.ut_line,sizeof(utmp.ut_line))) if (getutent()==(struct utmp*)0) return (struct utmp*)0;
  return &utmp;
}
/*}}}  */
/*{{{  pututline -- overwrite entry searched for with getutid*/
void pututline(struct utmp *new)
{
  struct utmp u;
  char buf[XDR_UTMPSIZE];
  XDR xdrs;

  u=*new;
  getutid(&u);
  lseek(fd,offset,SEEK_SET);
  xdrmem_create(&xdrs,buf,XDR_UTMPSIZE,XDR_ENCODE);
  if (xdr_utmp(&xdrs,&u)==1) write(fd,buf,XDR_UTMPSIZE);
  xdr_destroy(&xdrs);
  utmp=u;
}
/*}}}  */
