/*{{{}}}*/
/*{{{  Notes*/
/*

Written by Michael Haardt, 1994.

Using the dotshadow option, the format of /etc/passwd is

  login:uid:gid:gecos:home:maxage:expire

and the format of ~/.passwd is

  password:shell

Using the traditional option, the format of /etc/passwd is

  login:encrypted password:uid:gid:gecos:home

and there is no shadow file.    

*/
/*}}}  */
/*{{{  #includes*/
#ifdef _MINIX
#include <minix/libconf.h>
#endif
#ifdef __linux__
extern int _pwd_conf;
void _libconf(void);
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <paths.h>
#include <pwd.h>
#include <string.h>
#include <time.h>
/*}}}  */
/*{{{  #defines*/
#define setpwent _setpwent
#define endpwent _endpwent
#define getpwent _getpwent
#define getpwnam _getpwnam
#define getpwuid _getpwuid
#ifdef _MINIX
#define lstat(x,y) stat(x,y)
#endif
#define BUFSIZE 512
/*}}}  */

/*{{{  variables*/
static int passwdfd=-1;
static char buf[BUFSIZE];
static size_t cur,size;
/*}}}  */

/*{{{  gc*/
static int gc(int fd, char *buf, size_t *size, size_t *cur)
{
  if (*size==-1) *size=0;
  if (*size==*cur)
  {
    *size=read(fd,buf,BUFSIZE);
    *cur=0;
    if (*size<=0) return -1;
  }
  return (buf[(*cur)++]);
}
/*}}}  */
/*{{{  setpwent*/
int setpwent(void)
{
  if (passwdfd==-1)
  {
    passwdfd=open(_PATH_PASSWD,O_RDONLY);
    size=read(passwdfd,buf,sizeof(buf));
    cur=0;
  }
  else
  {
    lseek(passwdfd,(off_t)0,SEEK_SET);
    size=read(passwdfd,buf,sizeof(buf));
    cur=0;
  }
  return (passwdfd==-1 ? -1 : 0);
}
/*}}}  */
/*{{{  endpwent*/
void endpwent(void)
{
  if (passwdfd!=-1) close(passwdfd);
  passwdfd=-1;
}
/*}}}  */
/*{{{  nextpwent*/
static struct passwd *nextpwent(uid_t uid, const char *name)
{
  /*{{{  variables*/
  static char login[17];
  static char password[33];
  static char gecos[128];
  static char home[_POSIX_PATH_MAX];
  static char shell[_POSIX_PATH_MAX];
  int c;
  char *p;
  unsigned long ul;
  time_t now;
  static struct passwd pw=
  {
    login,
    password,
    (uid_t)0,
    (gid_t)0,
    gecos,
    home,
    shell,
    (time_t)0,
    (time_t)0,
    (time_t)0
  };
  /*}}}  */

  /*{{{  open passwd if needed or rewind else*/
  _libconf();
  if (passwdfd==-1) if (setpwent()<0)
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  /*{{{  login*/
  p=login;
  while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!='\n' && c!=':') *p++=c;
  *p='\0';
  if (c!=':')
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  if (_pwd_conf==0)
  /*{{{  encrypted password*/
  {
    p=password;
    while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!=':') *p++=c;
    *p='\0';
    if (c!=':')
    {
      endpwent();
      return ((struct passwd*)0);
    }
  }
  /*}}}  */
  /*{{{  uid*/
  pw.pw_uid=0;
  while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!='\n' && c!=':') pw.pw_uid=pw.pw_uid*10+c-'0';
  if (c!=':')
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  /*{{{  gid*/
  pw.pw_gid=0;
  while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!='\n' && c!=':') pw.pw_gid=pw.pw_gid*10+c-'0';
  if (c!=':')
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  /*{{{  gecos*/
  p=gecos;
  while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!=':') *p++=c;
  *p='\0';
  if (c!=':')
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  /*{{{  home*/
  p=home;
  while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!=':') *p++=c;
  *p='\0';
  if (c!=':')
  {
    endpwent();
    return ((struct passwd*)0);
  }
  /*}}}  */
  if (_pwd_conf==0)
  /*{{{  shell*/
  {
    p=shell;
    while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!='\n') *p++=c;
    *p='\0';
    time(&pw.pw_expire); pw.pw_expire+=3600; pw.pw_maxage=pw.pw_expire;
  }
  /*}}}  */
  else
  {
    /*{{{  maxage*/
    ul=0;
    while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!=':') ul=ul*10+(c-'0');
    if (c!=':')
    {
      endpwent();
      return ((struct passwd*)0);
    }
    pw.pw_maxage=ul;
    /*}}}  */
    /*{{{  account expiration date*/
    ul=0;
    while ((c=gc(passwdfd,buf,&size,&cur))!=-1 && c!='\n') ul=ul*10+(c-'0');
    if (c!='\n')
    {
      endpwent();
      return ((struct passwd*)0);
    }
    pw.pw_expire=ul;
    /*}}}  */
    /*{{{  password and shell*/
    {
      int fd;
      char file[_POSIX_PATH_MAX];
      struct stat buf;
      #ifdef __linux__
      uid_t myeuid;
      #endif
    
      strcpy(file,home);
      strcat(file,"/.passwd");
      #ifdef __linux__
      myeuid=geteuid();
      setreuid(-1,pw.pw_uid);
      #endif
      if 
      (
      (
      (uid!=(uid_t)-1 && uid==pw.pw_uid) ||
      (name!=(const char*)0 && strcmp(name,login)==0) ||
      (uid==(uid_t)-1 && name==(const char*)0)
      ) &&
      lstat(file,&buf)==0 && (fd=open(file,O_RDONLY))!=-1
      )
      {
        char ln[BUFSIZE];
        size_t size,cur;
    
        size=read(fd,ln,BUFSIZE); cur=0;
        pw.pw_age=buf.st_ctime;
        p=password;
        while ((c=gc(fd,ln,&size,&cur))!=EOF && p<(password+sizeof(password)-1) && c!=':') *p++=c;
        *p='\0';
        p=shell;
        while ((c=gc(fd,ln,&size,&cur))!=EOF && p<(shell+sizeof(shell)-1) && c!='\n') *p++=c;
        *p='\0';
        close(fd);
        if ((buf.st_mode&0777)!=0600 || buf.st_uid!=pw.pw_uid || buf.st_gid!=pw.pw_gid) strcpy(password,"*");
      }
      else
      {
        pw.pw_age=(time_t)0;
        strcpy(password,"*");
        strcpy(shell,"/noshell");
      }
      #ifdef __linux__
      setreuid(-1,myeuid);
      #endif
    }
    /*}}}  */
    /*{{{  check if account has expired*/
    time(&now);
    if (now>=pw.pw_expire) strcpy(password,"*");
    /*}}}  */
  }
  return &pw;
}
/*}}}  */
/*{{{  getpwent*/
struct passwd *getpwent(void)
{
  return (nextpwent((uid_t)-1,(const char*)0));
}
/*}}}  */

/*{{{  getpwnam*/
struct passwd *getpwnam(const char *name)
{
  /*{{{  variables*/
  struct passwd *pw;
  /*}}}  */

  setpwent();
  while ((pw=nextpwent((uid_t)-1,name))!=(struct passwd*)0) if (strcmp(pw->pw_name,name)==0)
  {
    endpwent();
    return pw;
  }
  endpwent();
  return (struct passwd*)0;
}
/*}}}  */
/*{{{  getpwuid*/
struct passwd *getpwuid(uid_t uid)
{
  /*{{{  variables*/
  struct passwd *pw;
  /*}}}  */

  setpwent();
  while ((pw=nextpwent(uid,(const char*)0))!=(struct passwd*)0) if (uid==pw->pw_uid)
  {
    endpwent();
    return pw;
  }
  endpwent();
  return (struct passwd*)0;
}
/*}}}  */
